/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;
import org.dbgl.db.Database;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.KeyValuePair;
import org.dbgl.model.Settings;
import org.dbgl.model.conf.Configuration;


public final class DFendImportThread extends Thread {

    private static final String[] LINKS = new String[] { "", "", "", "" };
    private static final String[] CUST_STRINGS = new String[] { "", "", "", "", "", "", "", "" };
    private static final int[] CUST_INTS = new int[] { 0, 0 };
    private static final int[] CUST_IDS = new int[] { 0, 0, 0, 0 };

    private final DosboxVersion defaultDBVersion;
    private final Configuration standardProfile;
    private final Configuration profiles;
    private final Text log;
    private final ProgressBar progressBar;
    private final Label profileLabel;
    private final boolean performCleanup;
    private final Display display;
    private String title;
    private int profileNumber;


    public DFendImportThread(final File dfendProfilesFile, final boolean performCleanup,
            final DosboxVersion defaultDBVersion, final Text log, final ProgressBar progressBar,
            final Label profileLabel) throws IOException {
        this.performCleanup = performCleanup;
        this.defaultDBVersion = defaultDBVersion;
        this.standardProfile = new Configuration(defaultDBVersion.getCanonicalConfFile(), true);
        this.log = log;
        this.progressBar = progressBar;
        this.profileLabel = profileLabel;
        this.display = log.getShell().getDisplay();
        this.profiles = new Configuration(dfendProfilesFile, false);
        this.progressBar.setMaximum(profiles.getSectionKeys().size());
        this.profileNumber = 0;
    }

    public void run() {
        Database dbase = Database.getInstance();
        final StringBuffer displayedLog = new StringBuffer();
        final StringBuffer messageLog = new StringBuffer();
        
        try {
            for (String titleLoop: profiles.getSectionKeys()) {
                title = titleLoop;
                String profFile = profiles.getValue(title, "prof");
                String confFile = profiles.getValue(title, "conf");
                boolean favorite = profiles.getValue(title, "fav").equals("1");
                try {
                    Configuration dfendExtra = new Configuration(new File(profFile), false);
                    String setup = dfendExtra.getValue("Extra", "Setup");
                    String setupParams = dfendExtra.getValue("Extra", "SetupParameters");
                    String notes = dfendExtra.getValue("ExtraInfo", "Notes");
                    String dev = dfendExtra.getValue("ExtraInfo", "Developer");
                    String pub = dfendExtra.getValue("ExtraInfo", "Publisher");
                    String gen = dfendExtra.getValue("ExtraInfo", "Genre");
                    String year = dfendExtra.getValue("ExtraInfo", "Year");
                    String status = Settings.getInstance().msg("dialog.dfendimport.defaultprofilestatus");
                    String cap = dfendExtra.getValue("dosbox", "captures");
                    int devId = KeyValuePair.findIdByValue(dbase.readDevelopersList(), dev);
                    int publId = KeyValuePair.findIdByValue(dbase.readPublishersList(), pub);
                    int genId = KeyValuePair.findIdByValue(dbase.readGenresList(), gen);
                    int yrId = KeyValuePair.findIdByValue(dbase.readYearsList(), year);
                    int statId = KeyValuePair.findIdByValue(dbase.readStatusList(), status);

                    try {
                        int profileId = dbase.addOrEditProfile(title, dev, pub, gen, year, status, setup, setupParams,
                                notes, favorite, devId, publId, genId, yrId, statId, defaultDBVersion.getId(),
                                LINKS, CUST_STRINGS, CUST_INTS, CUST_IDS, LINKS, -1);
                        Configuration dfendProfile = new Configuration(new File(confFile), true);

                        String dstCap = FileUtils.constructCapturesDir(profileId);
                        String dstCapRelative = FileUtils.constructRelativeCapturesDir(profileId);
                        File dstCapAbsolute = FileUtils.canonicalToData(dstCap);
                        FileUtils.createDir(dstCapAbsolute);
                        FileUtils.copyFiles(new File(cap), dstCapAbsolute);
                        dfendProfile.setValue("dosbox", "captures", dstCapRelative);

                        if (performCleanup) {
                            dfendProfile.removeSection("directserial");
                            dfendProfile.removeSection("modem");
                            dfendProfile.removeSection("ipx");
                            dfendProfile.removeSection("sdl");
                        }
                        
                        standardProfile.update(dfendProfile);
                        
                        if (defaultDBVersion.isMultiConfig()) {
                            dfendProfile.substract(standardProfile);
                        }

                        String newConfFile = FileUtils.constructUniqueConfigFileString(profileId, title, dfendProfile);
                        dfendProfile.saveToFile(FileUtils.canonicalToData(newConfFile), false, null);

                        dbase.updateProfileConf(newConfFile, dstCap, profileId);

                        messageLog.append(dfendExtra.getParsingProblems());
                        messageLog.append(dfendProfile.getParsingProblems());
                    } catch (SQLException se) {
                        messageLog.append(se.getMessage()).append('\n');
                    }
                } catch (IOException e) {
                    messageLog.append(e.getMessage()).append('\n');
                }

                if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) {
                    break;
                }
                display.asyncExec(new Runnable() {
                    public void run() {
                        profileLabel.setText(Settings.getInstance().msg("dialog.dfendimport.importing", new Object[] {title}));
                        profileLabel.pack();
                        if (messageLog.length() > displayedLog.length()) {
                            String newOutput = messageLog.substring(displayedLog.length());
                            log.append(newOutput);
                            displayedLog.append(newOutput);
                        }
                        progressBar.setSelection(profileNumber++);
                    }
                });
            }
        } catch (SQLException se) {
            messageLog.append(se.getMessage());
            display.asyncExec(new Runnable() {
                public void run() {
                    if (messageLog.length() > displayedLog.length()) {
                        String newOutput = messageLog.substring(displayedLog.length());
                        log.append(newOutput);
                        displayedLog.append(newOutput);
                    }
                }
            });
        }
    }
}
