/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import org.eclipse.swt.program.Program;
import org.dbgl.gui.StreamGobbler;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;
import javax.swing.filechooser.FileSystemView;


public final class PlatformUtils {

	public static final String OS_NAME = System.getProperty("os.name");
	public static final boolean IS_WINDOWS = OS_NAME.startsWith("Windows");
	public static final boolean IS_LINUX = OS_NAME.startsWith("Linux");
	public static final boolean IS_OSX = OS_NAME.startsWith("Mac OS");
	public static final String DB_EXECUTABLE = IS_WINDOWS? "DOSBox.exe": "dosbox";
	public static final String EOLN = System.getProperty("line.separator");
	
	
	public static String toNativePath(final String dbFileLocation) {
	    return dbFileLocation.replace('\\', File.separatorChar);
	}

	public static String toDosboxPath(final String hostFileLocation) {
	    return hostFileLocation.replace(File.separatorChar, '\\');
	}
	
	public static String archiveToNativePath(final String archiveEntry) {
	    // 7zip uses / as file seperator but to be safe in the future
	    // both \\ and / are converted to the host seperator
	    return archiveEntry.replace('/', File.separatorChar).replace('\\', File.separatorChar);
	}

	public static void openForEditing(final File file) {
	    Program prog = Program.findProgram(".conf");
	    if (prog == null) { prog = Program.findProgram(".txt");}
	    if (prog == null) { prog = Program.findProgram(".inf");}
	    if (prog == null) { prog = Program.findProgram(".ini");}
	    if (prog == null) { prog = Program.findProgram(".cfg");}
	    if (prog == null) { prog = Program.findProgram(".java");}
	    if (prog == null) { prog = Program.findProgram(".doc");}
	    if (prog == null) {
	        System.err.println(Settings.getInstance().msg("general.error.openfile", new Object[] {file}));
	    } else {
	        prog.execute(file.getPath());
	    }
	}
	
	public static void openForBrowsing(final String url) {
	    if (IS_WINDOWS) {
	        try {
	            Runtime.getRuntime().exec("rundll32 url.dll,FileProtocolHandler " + url);
	        } catch (IOException e) {
	            System.err.println(Settings.getInstance().msg("general.error.openurl", new Object[] {url}));
	        } 
	    } else {
	        Program prog = Program.findProgram(".html");
	        if (prog == null) {
	            System.err.println(Settings.getInstance().msg("general.error.openurl", new Object[] {url}));
	        } else {
	            prog.execute (url);
	        }
	    }
	}
	
	public static void openDirForViewing(final File file) {
	    try {
	        if (IS_WINDOWS) {
	            Runtime.getRuntime().exec(new String[]{"explorer", file.getPath()}, null, null);
	        } else if (IS_LINUX) {
	            try {
	                Runtime.getRuntime().exec(new String[]{"nautilus", file.getPath()}, null, null);
	            } catch (IOException e) {
	                Runtime.getRuntime().exec(new String[]{"kfmclient", file.getPath()}, null, null);
	            }
	        } else if (IS_OSX) {
	            Runtime.getRuntime().exec(new String[]{"open", file.getPath()}, null, null);
	        }
	    } catch (IOException e) {
	        System.err.println(Settings.getInstance().msg("general.error.opendir", new Object[] {file}));
	    }
	}

	public static void createShortcut(final Profile profile, final List<DosboxVersion> dbversionsList) throws IOException {
	    int dbversionIndex = DosboxVersion.findById(dbversionsList, profile.getDbversionId());
	    DosboxVersion dbversion = dbversionsList.get(dbversionIndex);
	    StringBuffer params = new StringBuffer(128);
	    if (dbversion.isMultiConfig()) {
	        params.append("-conf \"\"").append(dbversion.getCanonicalConfFile()).append("\"\" ");
	    }
	    params.append("-conf \"\"").append(profile.getCanonicalConfFile()).append("\"\"");
	    if (Settings.getInstance().getBooleanValue("dosbox", "hideconsole")) {
	        params.append(" -noconsole");
	    }
	    
    	File desktopDir = FileSystemView.getFileSystemView().getHomeDirectory();
    	String strictFilename = profile.getTitle().replaceAll("[\\/:*?\"<>|]", " ").trim();
    	File lnkFile = new File(desktopDir, strictFilename + ".lnk");
    	File vbsFile = FileUtils.canonicalToData("shortcut.vbs");
		BufferedWriter vbsWriter = new BufferedWriter(new FileWriter(vbsFile));
		vbsWriter.write("Set oWS = WScript.CreateObject(\"WScript.Shell\")" + EOLN);
		vbsWriter.write("Set oLink = oWS.CreateShortcut(\"" + lnkFile.getCanonicalPath() + "\")" + EOLN);
		vbsWriter.write("oLink.TargetPath = \"" + dbversion.getCanonicalExecutable().getPath() + "\"" + EOLN);
		vbsWriter.write("oLink.Arguments = \"" + params.toString() + "\"" + EOLN);
		vbsWriter.write("oLink.Description = \"" + Settings.getInstance().msg("general.shortcut.title", new Object[] {strictFilename}) + "\"" + EOLN);
		vbsWriter.write("oLink.WorkingDirectory = \"" + FileUtils.getDosRoot() + "\"" + EOLN);
		vbsWriter.write("oLink.Save" + EOLN);
		vbsWriter.flush();
		vbsWriter.close();
		Process proc = Runtime.getRuntime().exec(new String[] {"CSCRIPT", vbsFile.getCanonicalPath()}, null, vbsFile.getParentFile());
		StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "CSCRIPT stderr");            
        StreamGobbler outputGobbler = new StreamGobbler(proc.getInputStream(), "CSCRIPT stdout");
        outputGobbler.start();
        errorGobbler.start();
		try { proc.waitFor(); } catch (InterruptedException e) {}
        FileUtils.removeFile(vbsFile);
	}
}
