package org.dbgl.gui;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.ExpGame;
import org.dbgl.util.FileUtils;
import org.dbgl.util.ImportThread;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import swing2swt.layout.BorderLayout;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;


public class ImportDialog extends Dialog {

    private Text extract;
    private Text notes;
    private Table impDbVersionsList;
    private List myDbVersionsList;
    private Table profilesTable;
    private Button importProfileSettingsButton;
    private Button importCapturesButton;
    private Button importGameDataButton;
    private Text logText;
    private Label profileLabel;
    private Button cancelButton;
    protected Shell shell;

    private File archive;
    private String importTitle;
    private String importNotes;
    private boolean capturesAvailable = false;
    private boolean gamedataAvailable = false;
    private ExpGame[] profs;
    private java.util.List<DosboxVersion> dbversionsList;
    private java.util.List<Integer> dbmapping;
    private SortedSet<DosboxVersion> dbSet;
    private ImportThread importThread;


    public ImportDialog(Shell parent, int style) {
        super(parent, style);
    }

    public ImportDialog(Shell parent, java.util.List<DosboxVersion> dbList, File arc, InputStream xml)
            throws XPathExpressionException, ParserConfigurationException, SAXException, IOException {
        this(parent, SWT.NONE);
        dbversionsList = dbList;
        archive = arc;
        
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(xml);
        xml.close();

        XPathFactory xfactory = XPathFactory.newInstance();
        XPath xPath = xfactory.newXPath();
        importTitle = xPath.evaluate("//title", doc);
        importNotes = xPath.evaluate("//notes", doc);
        capturesAvailable = Boolean.valueOf(xPath.evaluate("//captures-available", doc));
        gamedataAvailable = Boolean.valueOf(xPath.evaluate("//gamedata-available", doc));

        NodeList profNodes = (NodeList) xPath.evaluate("/document/profile", doc, XPathConstants.NODESET);
        profs = new ExpGame[profNodes.getLength()];

        NodeList dbNodes = (NodeList) xPath.evaluate("/document/profile/dosbox", doc, XPathConstants.NODESET);
        DosboxVersion[] dbversions = new DosboxVersion[dbNodes.getLength()];
        for (int i = 0; i < dbversions.length; i++) {
            Node n = dbNodes.item(i);
            dbversions[i] = new DosboxVersion(i, xPath.evaluate("title", n), "", true, false, false, "",
                    xPath.evaluate("version", n));
        }

        dbSet = new TreeSet<DosboxVersion>();
        for (DosboxVersion db: dbversions)
            dbSet.add(db);

        for (int i = 0; i < profs.length; i++) {
            Node n = profNodes.item(i);
            profs[i] = new ExpGame(n, dbSet);
        }
    }

    public Object open() {
        createContents();
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (importThread != null && !importThread.isAlive() && !cancelButton.getText().equals("Close")) {
                GeneralPurposeDialogs.infoMessage(shell, "Import completed successfully.");
                profileLabel.setText("Please review the log below for possible problems");
                profileLabel.pack();
                cancelButton.setText("Close");
            }
            if (!display.readAndDispatch()) display.sleep();
        }
        return importThread;
    }

    protected void createContents() {
        shell = new Shell(getParent(), SWT.TITLE | SWT.CLOSE | SWT.BORDER | SWT.RESIZE | SWT.APPLICATION_MODAL);
        shell.setLayout(new BorderLayout(0, 0));
        shell.setSize(500, 630);
        shell.setText("Import " + importTitle);

        final Group settingsGroup = new Group(shell, SWT.NONE);
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 4;
        settingsGroup.setLayout(gridLayout);
        settingsGroup.setText("Options");
        settingsGroup.setLayoutData(BorderLayout.NORTH);

        notes = new Text(settingsGroup, SWT.WRAP | SWT.V_SCROLL | SWT.READ_ONLY | SWT.MULTI | SWT.H_SCROLL | SWT.BORDER);
        final GridData gridData_3 = new GridData(SWT.FILL, SWT.CENTER, true, false, 4, 1);
        gridData_3.heightHint = 40;
        notes.setLayoutData(gridData_3);
        notes.setText(importNotes);

        importProfileSettingsButton = new Button(settingsGroup, SWT.CHECK);
        importProfileSettingsButton.setSelection(true);
        importProfileSettingsButton.setLayoutData(new GridData());
        importProfileSettingsButton.setText("Import Profile settings");

        final Button fullSettingsButton = new Button(settingsGroup, SWT.RADIO);
        fullSettingsButton.setSelection(true);
        fullSettingsButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
        fullSettingsButton.setText("Full settings");

        final Button incrementalSettingsButton = new Button(settingsGroup, SWT.RADIO);
        incrementalSettingsButton.setLayoutData(new GridData());
        incrementalSettingsButton.setText("Incremental settings");
        new Label(settingsGroup, SWT.NONE);

        importCapturesButton = new Button(settingsGroup, SWT.CHECK);
        importCapturesButton.setSelection(capturesAvailable);
        importCapturesButton.setEnabled(capturesAvailable);
        importCapturesButton.setText("Import Captures");
        new Label(settingsGroup, SWT.NONE);
        new Label(settingsGroup, SWT.NONE);
        new Label(settingsGroup, SWT.NONE);

        importGameDataButton = new Button(settingsGroup, SWT.CHECK);
        importGameDataButton.setLayoutData(new GridData());
        importGameDataButton.setSelection(gamedataAvailable);
        importGameDataButton.setEnabled(gamedataAvailable);
        importGameDataButton.setText("Import Game data");

        final Label belowLabel = new Label(settingsGroup, SWT.NONE);
        belowLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
        belowLabel.setText("Into");

        extract = new Text(settingsGroup, SWT.BORDER);
        extract.setText(FileUtils.DOSROOT_DIR);
        extract.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        importProfileSettingsButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                incrementalSettingsButton.setEnabled(!incrementalSettingsButton.isEnabled());
                fullSettingsButton.setEnabled(!fullSettingsButton.isEnabled());
                importCapturesButton.setEnabled(!importCapturesButton.isEnabled());
            }
        });

        final Button button = new Button(settingsGroup, SWT.NONE);
        button.setText("button");
        //browseButton = new BrowseButton(settingsGroup, SWT.NONE);
        //browseButton.connect(shell, extract, null, BrowseType.DIR, CanonicalType.GPA, null);

        final Group profilesGroup = new Group(settingsGroup, SWT.NONE);
        profilesGroup.setText("Profiles");
        profilesGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, true, 4, 1));
        final GridLayout gridLayout_1 = new GridLayout();
        gridLayout_1.numColumns = 2;
        profilesGroup.setLayout(gridLayout_1);

        profilesTable = new Table(profilesGroup, SWT.FULL_SELECTION | SWT.CHECK | SWT.BORDER);
        profilesTable.setHeaderVisible(true);
        final GridData gridData_1 = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 2);
        gridData_1.heightHint = 80;
        profilesTable.setLayoutData(gridData_1);
        profilesTable.setLinesVisible(true);

        final TableColumn title = new TableColumn(profilesTable, SWT.NONE);
        title.setWidth(260);
        title.setText("Title");

        final TableColumn subdir = new TableColumn(profilesTable, SWT.NONE);
        subdir.setWidth(120);
        subdir.setText("Subdirectory");

        for (ExpGame prof: profs) {
            TableItem item = new TableItem(profilesTable, SWT.NONE);
            item.setText(prof.getTitle());
            item.setText(1, FileUtils.makeRelativeTo(prof.getGameDir(), new File(extract.getText())).getPath());
        }
        
        profilesTable.addListener(SWT.Selection, new Listener() {
            public void handleEvent(Event event) {
                if (event.detail == SWT.CHECK)
                    refillImportedDBVersionsList();
            }
        });
        
        final Button allButton = new Button(profilesGroup, SWT.NONE);
        allButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                for (TableItem item: profilesTable.getItems())
                    item.setChecked(true);
                refillImportedDBVersionsList();
            }
        });
        allButton.setLayoutData(new GridData(50, SWT.DEFAULT));
        allButton.setText("All");

        final Button noneButton = new Button(profilesGroup, SWT.NONE);
        noneButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                for (TableItem item: profilesTable.getItems())
                    item.setChecked(false);
                refillImportedDBVersionsList();
            }
        });
        final GridData gridData = new GridData(SWT.LEFT, SWT.BOTTOM, false, false);
        gridData.widthHint = 50;
        noneButton.setLayoutData(gridData);
        noneButton.setText("None");

        final Group dosboxVersionsGroup = new Group(settingsGroup, SWT.NONE);
        dosboxVersionsGroup.setText("DOSBox versions");
        dosboxVersionsGroup.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 4, 1));
        final GridLayout gridLayout_2 = new GridLayout();
        gridLayout_2.numColumns = 3;
        dosboxVersionsGroup.setLayout(gridLayout_2);

        final Label importedVersionsLabel = new Label(dosboxVersionsGroup, SWT.NONE);
        importedVersionsLabel.setText("Imported versions");

        final Label label_2 = new Label(dosboxVersionsGroup, SWT.SEPARATOR);
        label_2.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, true, 1, 2));

        final Label configuredVersionsLabel = new Label(dosboxVersionsGroup, SWT.NONE);
        configuredVersionsLabel.setText("Configured versions");

        impDbVersionsList = new Table(dosboxVersionsGroup, SWT.FULL_SELECTION | SWT.BORDER);
        impDbVersionsList.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                int impIdx = impDbVersionsList.getSelectionIndex();
                int mappedId = dbmapping.get(impIdx);
                int myIdx = DosboxVersion.findById(dbversionsList, mappedId);
                myDbVersionsList.select(myIdx);
            }
        });
        impDbVersionsList.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        myDbVersionsList = new List(dosboxVersionsGroup, SWT.V_SCROLL | SWT.BORDER);
        myDbVersionsList.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                int idx = impDbVersionsList.getSelectionIndex();
                if (idx != -1) {
                    int myIdx = myDbVersionsList.getSelectionIndex();
                    dbmapping.add(idx, dbversionsList.get(myIdx).getId());
                }
            }
        });
        final GridData gridData_4 = new GridData(SWT.FILL, SWT.CENTER, true, false);
        gridData_4.heightHint = 80;
        myDbVersionsList.setLayoutData(gridData_4);
        for (DosboxVersion ver: dbversionsList)
            myDbVersionsList.add(ver.getTitle() + " (" + ver.getVersion() + ")");

        dbmapping = new ArrayList<Integer>();
        for (DosboxVersion dbversion: dbSet) {
            TableItem item = new TableItem(impDbVersionsList, SWT.NONE);
            item.setText(dbversion.getTitle() + " (" + dbversion.getVersion() + ")");
            dbmapping.add(dbversion.findBestMatchId(dbversionsList));
        }

        for (TableItem item: profilesTable.getItems())
            item.setChecked(true);
        refillImportedDBVersionsList();
        
        final Button startButton = new Button(settingsGroup, SWT.NONE);
        shell.setDefaultButton(startButton);
        startButton.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
        startButton.setText("Start Import");

        cancelButton = new Button(settingsGroup, SWT.NONE);
        cancelButton.setLayoutData(new GridData(80, SWT.DEFAULT));
        cancelButton.setText("Cancel");
        cancelButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                shell.close();
            }
        });
        new Label(settingsGroup, SWT.NONE);

        final Group progressGroup = new Group(shell, SWT.NONE);
        progressGroup.setLayout(new GridLayout());
        progressGroup.setText("Progress");

        final ProgressBar progressBar = new ProgressBar(progressGroup, SWT.NONE);
        progressBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));

        profileLabel = new Label(progressGroup, SWT.NONE);

        logText = new Text(progressGroup, SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER | SWT.H_SCROLL);
        final GridData gridData_2 = new GridData(SWT.FILL, SWT.FILL, true, true);
        gridData_2.minimumHeight = 100;
        logText.setLayoutData(gridData_2);

        startButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                if (!isValid()) return;
                try {
                    java.util.List<ExpGame> profiles = new ArrayList<ExpGame>();
                    for (int i = 0; i < profs.length; i++) {
                        TableItem it = profilesTable.getItem(i);
                        if (it.getChecked()) {
                            int oldId = profs[i].getDbversionId();
                            int newId = getDosboxVersionId(oldId);
                            profs[i].setDbversionId(newId);
                            profs[i].setGameDir(new File(it.getText(1)));
                            profiles.add(profs[i]);
                        }
                    }
                    if (importGameDataButton.getSelection() && !isValidDirs()) return;
                    
                    importThread = new ImportThread(importProfileSettingsButton.getSelection(),
                            importCapturesButton.getSelection(), importGameDataButton.getSelection(),
                            fullSettingsButton.getSelection(), archive, profiles, dbversionsList,
                            logText, progressBar, profileLabel);
                    startButton.setEnabled(false);
                    importThread.start();
                } catch (IOException ex) {
                    GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
                    importThread = null;
                }
            }
        });
        new Label(settingsGroup, SWT.NONE);
        
        final TableEditor editor = new TableEditor(profilesTable);
        editor.horizontalAlignment = SWT.LEFT;
        editor.grabHorizontal = true;
        editor.minimumWidth = 50;
        profilesTable.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                // Clean up any previous editor control
                Control oldEditor = editor.getEditor();
                if (oldEditor != null) oldEditor.dispose();
        
                // Identify the selected row
                TableItem item = (TableItem)e.item;
                if (item == null) return;
        
                // The control that will be the editor must be a child of the Table
                Text newEditor = new Text(profilesTable, SWT.NONE);
                newEditor.setText(item.getText(1));
                newEditor.addModifyListener(new ModifyListener() {
                    public void modifyText(ModifyEvent me) {
                        Text text = (Text)editor.getEditor();
                        editor.getItem().setText(1, text.getText());
                    }
                });
                newEditor.selectAll();
                //newEditor.setFocus();
                editor.setEditor(newEditor, item, 1);
            }});
        
        importGameDataButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                // Clean up any previous editor control
                Control oldEditor = editor.getEditor();
                if (oldEditor != null) oldEditor.dispose();
            }
        });
    }
    
    private boolean isValid() {
        GeneralPurposeDialogs.initErrorDialog();
        if (!haveProfileToImport())
            GeneralPurposeDialogs.addError("You must select at least 1 profile to import.", profilesTable);
        if (!haveCategoryToImport())
            GeneralPurposeDialogs.addError("You must select at least 1 category to import.", profilesTable);
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }
    
    private boolean isValidDirs() {
        GeneralPurposeDialogs.initErrorDialog();
        for (ExpGame prof: profs) {
            File location = new File(new File(extract.getText()), prof.getGameDir().getPath());
            if (location.isDirectory())
                GeneralPurposeDialogs.addError("At least 1 directory (" + location + ") already exists. You cannot install a game to that location.", profilesTable);
        }
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }

    private void refillImportedDBVersionsList() {
        int idx = 0;
        for (DosboxVersion dbversion: dbSet) {
            int dbid = dbversion.getId();
            TableItem ti = impDbVersionsList.getItem(idx);
            Display d = shell.getDisplay();
            Color c = isUsed(dbid) ? null: d.getSystemColor(SWT.COLOR_GRAY);
            ti.setForeground(c);
            idx++;
        }
    }

    private boolean isUsed(int dbVersionId) {
        for (int i = 0; i < profs.length; i++) {
            TableItem item = profilesTable.getItem(i);
            int dbid = profs[i].getDbversionId();
            if (item.getChecked() && dbVersionId == dbid) return true;
        }
        return false;
    }
    
    private boolean haveProfileToImport() {
        for (TableItem item: profilesTable.getItems())
            if (item.getChecked()) return true;
        return false;
    }
    
    private boolean haveCategoryToImport() {
        return importProfileSettingsButton.getSelection() || importGameDataButton.getSelection();
    }

    private int getDosboxVersionId(int oldId) {
        int index = 0;
        for (DosboxVersion ver: dbSet) {
            if (ver.getId() == oldId) return dbmapping.get(index);
            index++;
        }
        return -1;
    }
}
