/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import org.dbgl.model.conf.Configuration;


public final class Settings extends Configuration {

    private static final String SAMPLE_RATES = "8000 11025 16000 22050 32000 44100 48000 49716";
    private static final String BASE_ADDRS = "220 240 260 280 2a0 2c0 2e0 300";
    private static final String IRQS = "3 5 7 9 10 11 12";
    private static final String DMAS = "0 1 3 5 6 7";
    
    private static final File SETTINGS_CONF = new File("settings.conf");
    
    private static Locale locale;
    private static ResourceBundle mes;
    private static MessageFormat formatter;


    private Settings() {
        super();
        init();
        try {
            if (SETTINGS_CONF.isFile() && SETTINGS_CONF.canRead()) {
                parseConfigurationFile(SETTINGS_CONF, true);
            }
        } catch (IOException e) {
            // if settings could not be read, use only the defaults
        }
        
        locale = new Locale(getValue("locale", "language"), getValue("locale", "country"));
        try {
        	mes = ResourceBundle.getBundle("plugins/i18n/MessagesBundle", locale);
        } catch (MissingResourceException me) {
        	mes = ResourceBundle.getBundle("i18n/MessagesBundle", locale);
        }
        formatter = new MessageFormat("");
        formatter.setLocale(locale);
    }


    private static class SettingsHolder {
        private static Settings instance = new Settings();
    }


    public static Settings getInstance() {
        return SettingsHolder.instance;
    }

    public void save() throws IOException {
        saveToFile(SETTINGS_CONF, true, null);
    }

    private void init() {
        setGuiDefaults();

        setIntValue("profiledefaults", "confpath", 0);
        setIntValue("profiledefaults", "conffile", 0);

        setBooleanValue("dosbox", "hideconsole", false);

        setBooleanValue("communication", "port_enabled", true);
        setValue("communication", "port", "4740");

        setValue("database", "connectionstring", "jdbc:hsqldb:file:./db/database");
        setValue("database", "username", "sa");
        setValue("database", "pasword", "");

        setValue("directory", "data", ".");
        setValue("directory", "dosbox", ".");
        
        setValue("locale", "language", "en");
        setValue("locale", "country", "");

        setValue("mobygames", "platform_filter", "dos pc<space>booter");
        setBooleanValue("mobygames", "set_title", true);
        setBooleanValue("mobygames", "set_developer", true);
        setBooleanValue("mobygames", "set_publisher", true);
        setBooleanValue("mobygames", "set_year", true);
        setBooleanValue("mobygames", "set_genre", true);
        setBooleanValue("mobygames", "set_link", true);
        setBooleanValue("mobygames", "set_description", true);
        setBooleanValue("mobygames", "set_rank", true);

        setBooleanValue("environment", "use", false);
        setValue("environment", "value", "");

        setProfileDefaults();
    }

    private void setProfileDefaults() {
        setValue("profile", "priority_active", "lowest lower normal higher highest");
        setValue("profile", "priority_inactive", "lowest lower normal higher highest pause");
        setValue("profile", "output", "ddraw direct3d overlay opengl openglhq openglnb surface");
        setValue("profile", "frameskip", "0 1 2 3 4 5 6 7 8 9 10");
        setValue("profile", "scaler", "none normal2x normal3x advmame2x advmame3x advinterp2x advinterp3x "
                + "tv2x tv3x rgb2x rgb3x scan2x scan3x supereagle super2xsai 2xsai hq2x hq3x hardware2x hardware3x "
                + "tv tvadvmame2x normal4x");
        setValue("profile", "pixelshader", "none 2xsai.fx 2xsai_srgb.fx bilinear.fx cartoon.fx gs2x.fx "
                + "gs4x.fx gs4xcolor.fx gs4xcolorfilter.fx gs4xcolorscale.fx gs4xfilter.fx gs4xscale.fx matrix.fx "
                + "mcgreen.fx mcorange.fx point.fx scale2x.fx scale2x_ps14.fx sketch.fx");
        setValue("profile", "fullresolution", "original 0x0 320x200 640x480 800x600 1024x768 1280x768 1280x960 1280x1024");
        setValue("profile", "windowresolution", "original 320x200 640x480 800x600 1024x768 1280x768 1280x960 1280x1024");
        setValue("profile", "svgachipset", "none et3000 et4000 et4000new pvga1a s3");
        setValue("profile", "videoram", "256 512 1024 2048 3072 4096");
        setValue("profile", "vsyncmode", "off on force");
        setValue("profile", "lfbglide", "full read write none");
        setValue("profile", "machine", "cga hercules pcjr tandy vga");
        setValue("profile", "machine073", "cga hercules pcjr tandy ega vgaonly svga_s3 svga_et3000 svga_et4000 svga_paradise vesa_nolfb vesa_oldvbe");
        setValue("profile", "cputype", "auto 386 386_slow 486_slow pentium_slow 386_prefetch");
        setValue("profile", "core", "dynamic full normal simple auto");
        setValue("profile", "cycles",
                "350 500 750 1000 2000 3000 4000 5000 7500 10000 12500 15000 17500 20000 "
                + "25000 30000 32500 35000 40000 45000 50000 55000 60000 auto max<space>50% max<space>80% max<space>90% max");
        setValue("profile", "cycles_up", "20 50 100 500 1000 2000 5000 10000");
        setValue("profile", "cycles_down", "20 50 100 500 1000 2000 5000 10000");
        setValue("profile", "memsize", "0 1 2 4 8 16 32 63");
        setValue("profile", "umb", "false true max");
        setValue("profile", "loadfix_value", "1 63 127");
        setValue("profile", "rate", SAMPLE_RATES);
        setValue("profile", "blocksize", "256 512 1024 2048 4096 8192");
        setValue("profile", "prebuffer", "10");
        setValue("profile", "mpu401", "none intelligent uart");
        setValue("profile", "device", "alsa default coreaudio coremidi mt32 none oss win32");
        setValue("profile", "mt32rate", SAMPLE_RATES);
        setValue("profile", "sbtype", "none sb1 sb2 sbpro1 sbpro2 sb16");
        setValue("profile", "oplrate", SAMPLE_RATES);
        setValue("profile", "oplmode", "auto cms opl2 dualopl2 opl3");
        setValue("profile", "oplemu", "default compat fast old");
        setValue("profile", "sbbase", BASE_ADDRS);
        setValue("profile", "irq", IRQS);
        setValue("profile", "dma", DMAS);
        setValue("profile", "hdma", DMAS);
        setValue("profile", "gusrate", SAMPLE_RATES);
        setValue("profile", "gusbase", BASE_ADDRS);
        setValue("profile", "irq1", IRQS);
        setValue("profile", "irq2", IRQS);
        setValue("profile", "dma1", DMAS);
        setValue("profile", "dma2", DMAS);
        setValue("profile", "pcrate", SAMPLE_RATES);
        setValue("profile", "tandy", "auto off on");
        setValue("profile", "tandyrate", SAMPLE_RATES);
        setValue("profile", "sensitivity", "10 20 30 40 50 60 70 80 90 100 125 150 175 200 250 300 350 400 "
                + "450 500 550 600 700 800 900 1000");
        setValue("profile", "joysticktype", "auto none 2axis 4axis 4axis_2 ch fcs");
        setValue("profile", "mount_type", "cdrom dir floppy");
        setValue("profile", "zipmount_type", "cdrom dir floppy");
        setValue("profile", "freesize", "1 10 100 200 500 1000");
        setValue("profile", "lowlevelcd_type", "aspi ioctl ioctl_dx ioctl_dio ioctl_mci noioctl");
    }

    private void setGuiDefaults() {
        setValue("gui", "width", "904");
        setValue("gui", "height", "475");
        setValue("gui", "x", "10");
        setValue("gui", "y", "10");
        setValue("gui", "profiledialog_width", "660");
        setValue("gui", "profiledialog_height", "522");
        setValue("gui", "multiprofiledialog_width", "600");
        setValue("gui", "multiprofiledialog_height", "375");
        setValue("gui", "dosboxdialog_width", "600");
        setValue("gui", "dosboxdialog_height", "335");
        setValue("gui", "templatedialog_width", "660");
        setValue("gui", "templatedialog_height", "522");
        setValue("gui", "mountdialog_width", "640");
        setValue("gui", "mountdialog_height", "500");
        setValue("gui", "archivebrowser_width", "400");
        setValue("gui", "mobygamesbrowser_width", "400");
        setValue("gui", "mobygamesbrowser_height", "375");
        setValue("gui", "archivebrowser_height", "375");
        setValue("gui", "dfendimportdialog_width", "600");
        setValue("gui", "dfendimportdialog_height", "375");
        setValue("gui", "exportlistdialog_width", "550");
        setValue("gui", "exportlistdialog_height", "190");
        setValue("gui", "migratedialog_width", "600");
        setValue("gui", "migratedialog_height", "375");
        setValue("gui", "settingsdialog_width", "600");
        setValue("gui", "settingsdialog_height", "400");
        setValue("gui", "filterdialog_width", "600");
        setValue("gui", "filterdialog_height", "335");
        setValue("gui", "filtertab", "0");
        setBooleanValue("gui", "maximized", false);
        setValue("gui", "column1width", "150");
        setValue("gui", "column2width", "48");
        setValue("gui", "column3width", "100");
        setValue("gui", "column4width", "100");
        setValue("gui", "column5width", "70");
        setValue("gui", "column6width", "40");
        setValue("gui", "column7width", "60");
        setValue("gui", "column8width", "60");
        setValue("gui", "column9width", "38");
        setValue("gui", "column10width", "40");
        setValue("gui", "column11width", "70");
        setValue("gui", "column12width", "70");
        setValue("gui", "column13width", "70");
        setValue("gui", "column14width", "70");
        setValue("gui", "column15width", "70");
        setValue("gui", "column16width", "70");
        setValue("gui", "column17width", "70");
        setValue("gui", "column18width", "70");
        setValue("gui", "column19width", "44");
        setValue("gui", "column20width", "44");
        setValue("gui", "column21width", "82");
        setBooleanValue("gui", "column1visible", true);
        setBooleanValue("gui", "column2visible", true);
        setBooleanValue("gui", "column3visible", true);
        setBooleanValue("gui", "column4visible", true);
        setBooleanValue("gui", "column5visible", true);
        setBooleanValue("gui", "column6visible", true);
        setBooleanValue("gui", "column7visible", true);
        setBooleanValue("gui", "column8visible", true);
        setBooleanValue("gui", "column9visible", true);
        setBooleanValue("gui", "column10visible", false);
        setBooleanValue("gui", "column11visible", false);
        setBooleanValue("gui", "column12visible", false);
        setBooleanValue("gui", "column13visible", false);
        setBooleanValue("gui", "column14visible", false);
        setBooleanValue("gui", "column15visible", false);
        setBooleanValue("gui", "column16visible", false);
        setBooleanValue("gui", "column17visible", false);
        setBooleanValue("gui", "column18visible", false);
        setBooleanValue("gui", "column19visible", false);
        setBooleanValue("gui", "column20visible", false);
        setValue("gui", "column2_1width", "300");
        setValue("gui", "column2_2width", "250");
        setValue("gui", "column2_3width", "50");
        setValue("gui", "column2_4width", "38");
        setValue("gui", "column3_1width", "500");
        setValue("gui", "column3_2width", "50");
        setValue("gui", "sortcolumn", "0 8");
        setValue("gui", "sortascending", "true true");
        setValue("gui", "columnorder", "0 1 2 3 4 5 6 7 8");
        setValue("gui", "sashweights", "777 222");
        setValue("gui", "screenshotsheight", "100");
        setValue("gui", "screenshotscolumnheight", "50");
        setBooleanValue("gui", "autosortonupdate", false);
        setBooleanValue("gui", "screenshotsfilename", true);
        setIntValue("gui", "buttondisplay", 0);
        setValue("gui", "custom1", "Custom1");
        setValue("gui", "custom2", "Custom2");
        setValue("gui", "custom3", "Custom3");
        setValue("gui", "custom4", "Custom4");
        setValue("gui", "custom5", "Custom5");
        setValue("gui", "custom6", "Custom6");
        setValue("gui", "custom7", "Custom7");
        setValue("gui", "custom8", "Custom8");
        setValue("gui", "custom9", "Custom9");
        setValue("gui", "custom10", "Custom10");
    }
    
    public String msg(String key) {
    	try {
    		return mes.getString(key);
    	} catch(MissingResourceException me) {
    		return "[" + key + "]";
    	}
    }
    
    public String msg(String key, Object[] objs) {
        formatter.applyPattern(msg(key));
        return formatter.format(objs);
    }
}
