/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;
import org.dbgl.db.Database;
import org.dbgl.model.MobyProfile;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;


public final class MobyGamesThread extends Thread {

    private List<Profile> profiles;
    private String title;
    
    private final Text log;
    private final ProgressBar progressBar;
    private final Label profileLabel;
    private final Display display;
    private final String lineDelimiter;


    public MobyGamesThread(final List<Profile> profs, final Text log, final ProgressBar progressBar, final Label profileLabel) throws IOException {
        this.profiles = profs;
        this.log = log;
        this.lineDelimiter = log.getLineDelimiter();
        this.progressBar = progressBar;
        this.profileLabel = profileLabel;
        this.display = log.getShell().getDisplay();
    }

    public void run() {
        Database dbase = Database.getInstance();
        Settings settings = Settings.getInstance();
        final StringBuffer displayedLog = new StringBuffer();
        final StringBuffer messageLog = new StringBuffer();
        
        while (true) {
        	try {
        		Profile prof;
        		synchronized (profiles) {
        			prof = profiles.remove(0);
        		}
        		title = prof.getTitle();
        		try {
        			java.util.List<MobyProfile> mobyGamesList = MobyGamesUtils.getEntries(title, settings.getValues("mobygames", "platform_filter"));
        			if (mobyGamesList.size() == 1) {
        				MobyProfile thisGame = MobyGamesUtils.getEntry(mobyGamesList.get(0));
        				synchronized (this) {
        					String title = settings.getBooleanValue("mobygames", "set_title")? thisGame.getTitle(): prof.getTitle();
        					String dev = settings.getBooleanValue("mobygames", "set_developer")? thisGame.getDeveloperName(): prof.getDeveloperName();
        					String publ = settings.getBooleanValue("mobygames", "set_publisher")? thisGame.getPublisherName(): prof.getPublisherName();
        					String year = settings.getBooleanValue("mobygames", "set_year")? thisGame.getYear(): prof.getYear();
        					String genre = settings.getBooleanValue("mobygames", "set_genre")? thisGame.getGenre(): prof.getGenre();
        					String link1 = settings.getBooleanValue("mobygames", "set_link")? thisGame.getUrl(): prof.getLink(0);
        					String[] links = new String[] {link1, prof.getLink(1), prof.getLink(2), prof.getLink(3)};
        					String linkTitle1 = settings.getBooleanValue("mobygames", "set_link")? settings.msg("dialog.profile.mobygames.link.maininfo"): prof.getLinkTitle(0);
        					String[] linkTitles = new String[] {linkTitle1, prof.getLinkTitle(1), prof.getLinkTitle(2), prof.getLinkTitle(3)};
        					StringBuffer notes = new StringBuffer(prof.getNotes());
        					String p = thisGame.getNotes().replaceAll("\n", lineDelimiter);
        					if ((settings.getBooleanValue("mobygames", "set_description")) && (!prof.getNotes().endsWith(p))) {
        						if (notes.length() > 0) {
        							notes.append(lineDelimiter + lineDelimiter);
        						}
        						notes.append(p);
        					}
        					int rank = settings.getBooleanValue("mobygames", "set_rank")? thisGame.getRank(): prof.getCustomInt(0);
        					int[] customInts = new int[] {rank, prof.getCustomInt(1)};
        					dbase.addOrEditProfile(title, dev, publ, genre, year, 
        							prof.getStatus(), prof.getSetup(), prof.getSetupParameters(), notes.toString(), prof.isDefault(), 
        							prof.getDbversionId(), links, prof.getCustomStrings(), customInts, linkTitles, prof.getId());
        				}
        			} else if (mobyGamesList.size() == 0) {
        				messageLog.append(title).append(": ").append(settings.msg("general.notice.mobygamesnoresults", new String[] {title})).append('\n');
        			} else {
        				messageLog.append(title).append(": ").append(settings.msg("dialog.multiprofile.notice.titlenotunique", new String[] {title})).append('\n');
        			}
        		} catch (SQLException se) {
        			messageLog.append(se.getMessage()).append('\n');
        		}
        	} catch (IOException e) {
        		messageLog.append(title).append(": ").append(settings.msg("general.error.retrieveinfomobygames", new String[] {title, e.toString()})).append('\n');
        	} catch (IndexOutOfBoundsException iobe) {
        		return; // queue empty, we're done
        	}
        	
        	display.asyncExec(new Runnable() {
        		public void run() {
        			profileLabel.setText(Settings.getInstance().msg("dialog.multiprofile.updating", new Object[] {title}));
                    profileLabel.pack();
                    if (messageLog.length() > displayedLog.length()) {
                        String newOutput = messageLog.substring(displayedLog.length());
                        log.append(newOutput);
                        displayedLog.append(newOutput);
                    }
        			progressBar.setSelection(progressBar.getSelection() + 1);
        		}
        	});
        }
        
    }
}
