/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.sql.SQLException;
import java.util.List;
import org.dbgl.db.Database;
import org.dbgl.gui.MainWindow;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.ExpProfile;
import org.dbgl.model.KeyValuePair;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;
import org.dbgl.model.conf.CompositeConfiguration;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;


public final class ImportThread extends Thread {

	private static final String PREFIX_OK = "  + ";
	private static final String PREFIX_ERR = "  - ";
	
	private List<ExpProfile> profs;
	private List<DosboxVersion> dbversionsList;
    private boolean importCaptures;
    private boolean importGameData;
    private boolean importFullSettings;
    private boolean customValues;
    private String[] customFields;
    private boolean useExistingConf;
    private File zipfile;
    private Text log;
    private ProgressBar progressBar;
    private Label status;
    private Display display;


    public ImportThread(List<ExpProfile> p, List<DosboxVersion> dbList, File zipfile, boolean captures, boolean useExistingConf,
    		boolean gamedata, boolean fullSettings, boolean customValues, String[] customFields, Text log, ProgressBar progressBar, Label status) throws IOException {
    	this.profs = p;
    	this.dbversionsList = dbList;
    	this.zipfile = zipfile;
    	this.importCaptures = captures;
    	this.useExistingConf = useExistingConf;
        this.importGameData = gamedata;
        this.importFullSettings = fullSettings;
        this.customValues = customValues;
        this.customFields = customFields;
        this.log = log;
        this.progressBar = progressBar;
        this.progressBar.setMaximum(profs.size());
        this.status = status;
        this.display = log.getShell().getDisplay();
    }

    public void run() {
    	final Settings settings = Settings.getInstance();
        Database db = Database.getInstance();
        final StringBuffer messageLog = new StringBuffer();
        final StringBuffer displayedLog = new StringBuffer();
        
        for (final ExpProfile p: profs) {
        	if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) break;
            display.asyncExec(new Runnable() {
                public void run() {
                    status.setText(settings.msg("dialog.import.importing", new Object[] { p.getTitle() }));
                    status.pack();
                }
            });
        	
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            PrintStream ps = new PrintStream(bos);
            
            try {
            	messageLog.append(p.getTitle()).append(":\n");
            	
                int devId = KeyValuePair.findIdByValue(db.readDevelopersList(), p.getDeveloperName());
                int publId = KeyValuePair.findIdByValue(db.readPublishersList(), p.getPublisherName());
                int genId = KeyValuePair.findIdByValue(db.readGenresList(), p.getGenre());
                int yrId = KeyValuePair.findIdByValue(db.readYearsList(), p.getYear());
                int statId = KeyValuePair.findIdByValue(db.readStatusList(), p.getStatus());
                String[] customStrings = DFendImportThread.CUST_STRINGS;
                int[] customInts = DFendImportThread.CUST_INTS;
                int[] custIDs = DFendImportThread.CUST_IDS;
                if (customValues) {
                	customStrings = p.getCustomStrings();
                	customInts = p.getCustomInts();
                	for (int i = 0; i < 4; i++) {
                        custIDs[i] = KeyValuePair.findIdByValue(db.readCustomList(i), p.getCustomString(i));
                	}
                }
                int profileId = db.addOrEditProfile(p.getTitle(), p.getDeveloperName(), p.getPublisherName(),
                        p.getGenre(), p.getYear(), p.getStatus(), p.getSetup(), p.getSetupParameters(),
                        p.getNotes(), p.isDefault(), devId, publId, genId, yrId, statId, p.getDbversionId(),
                        p.getLinks(), customStrings, customInts, custIDs, p.getLinkTitles(), -1);

                String newCapturesString = FileUtils.constructCapturesDir(profileId);
                File relativeCapturesDirInZip = new File(p.getCaptures());
                File canonicalCapturesDir = FileUtils.canonicalToData(newCapturesString);
                if (!canonicalCapturesDir.exists()) {
                    FileUtils.createDir(canonicalCapturesDir);
                    messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.createddir", new Object[] { canonicalCapturesDir })).append('\n');
                    if (importCaptures) {
                    	try {
                            FileUtils.extractZip(zipfile, relativeCapturesDirInZip, canonicalCapturesDir);
                            messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.extractedcaptures", new Object[] { canonicalCapturesDir })).append('\n');;
                        } catch (IOException e) {
                            messageLog.append(PREFIX_ERR).append(settings.msg("dialog.import.error.capturesextraction", new Object[] { e.getMessage() })).append('\n');;
                        }
                    }
                } else {
                    messageLog.append(PREFIX_ERR).append(settings.msg("dialog.import.error.capturesdirexists", new Object[] { canonicalCapturesDir })).append('\n');;
                }
                
                if (importGameData) {
                	File relativeGameDir = p.getGameDir();
                    File relativeGameDirInZip = new File(FileUtils.DOSROOT_DIR, new File(String.valueOf(p.getImportedId()), relativeGameDir.getPath()).getPath());
                    File canonicalGameDir = FileUtils.canonicalToDosroot(new File(p.getBaseDir(), p.getGameDir().getPath()).getPath());
                    if (!canonicalGameDir.exists()) {
	                    FileUtils.createDir(canonicalGameDir);
	                    messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.createddir", new Object[] { canonicalGameDir })).append('\n');;
	                    try {
	                        FileUtils.extractZip(zipfile, relativeGameDirInZip, canonicalGameDir);
	                        messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.extractedgamedata", new Object[] { canonicalGameDir })).append('\n');;
	                    } catch (IOException e) {
	                        messageLog.append(PREFIX_ERR).append(settings.msg("dialog.import.error.gamedataextraction", new Object[] { e.getMessage() })).append('\n');;
	                    }
                    }
                }
                
                DosboxVersion assocDBVersion = dbversionsList.get(DosboxVersion.findById(dbversionsList, p.getDbversionId()));
                String newConfString;
                if (useExistingConf && FileUtils.areRelated(new File(FileUtils.getDosRoot()), p.getCanonicalConfFile()) && FileUtils.isExistingFile(p.getCanonicalConfFile())) {
                	newConfString = p.getCanonicalConfFile().getPath();
                	messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.usingexistingconf", new Object[] { p.getCanonicalConfFile() })).append('\n');
                } else {
                	CompositeConfiguration gameCConf = new CompositeConfiguration(assocDBVersion, null, importFullSettings? p.getImportedFullConfig(): p.getImportedIncrConfig(), false, null, ps);
                	gameCConf.getCompositeConf().makeUnRelative(p.getBaseDir());
                	newConfString = FileUtils.constructUniqueConfigFileString(profileId, p.getTitle(), gameCConf.getCompositeConf());
                    Profile newProfile = new Profile(profileId, newConfString, newCapturesString, p);
                    gameCConf.save(newProfile, null, ps);
                    if (bos.size() > 0) {
                        messageLog.append(PREFIX_ERR).append(bos.toString());
                    } else {
                    	messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.createdconf", new Object[] { newProfile.getCanonicalConfFile() })).append('\n');
                    }
                }
                
                db.updateProfileConf(newConfString, newCapturesString, profileId);
                
                String setup = p.getSetup();
                if (setup.length() > 0) {
                    setup = FileUtils.prefixAndSanitizeToDosroot(p.getBaseDir(), new File(setup)).getPath();
                }
                String[] links = p.getLinks();
                for (int i = 0; i < links.length; i++) {
                    if (!links[i].equals("") && !links[i].contains("://")) {
                   		links[i] = FileUtils.makeRelativeToDosroot(FileUtils.canonicalToData(links[i])).getPath();
                        links[i] = FileUtils.prefixAndSanitizeToDosroot(p.getBaseDir(), new File(links[i])).getPath();
                        if (!p.getBaseDir().isAbsolute()) {
                        	links[i] = FileUtils.DOSROOT_DIR + links[i];
                        }
                    }
                }
                db.updateProfileSetupAndLinks(setup, links, profileId);
                
                messageLog.append(PREFIX_OK).append(settings.msg("dialog.import.notice.createddbentry", new Object[] { profileId, newConfString, newCapturesString, assocDBVersion.getTitle() })).append('\n');;
                
            } catch (SQLException e) {
                messageLog.append(e.getMessage()).append('\n');
            }

            if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) break;
            display.asyncExec(new Runnable() {
                public void run() {
                	if (messageLog.length() > displayedLog.length()) {
                		String newOutput = messageLog.substring(displayedLog.length());
                        log.append(newOutput);
                        displayedLog.append(newOutput);
                    }
                    progressBar.setSelection(progressBar.getSelection() + 1);
                }
            });
        }
        
        if (customFields != null) {
        	for (int i = 0; i < MainWindow.EDIT_COLUMN_NAMES; i++) {
        		settings.setValue("gui", "custom" + (i + 1), customFields[i]);
        	}
        }
    }

	public String[] getCustomFields() {
		return customFields;
	}
}
