/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *  
 *  Many thanks to Manuel J. Gallego for his work on MobyGames querying
 *  for TincoreADB. This file is based on his code.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.dbgl.model.MobyProfile;
import org.apache.commons.lang.StringEscapeUtils;


public class MobyGamesUtils {
	
	private static final String HTTP_PROTOCOL = "http://";
	private static final String HTML_HREF_OPEN = " href=\"";
	private static final String HTML_QUOTE = "\"";
	
	private static final String HTML_SPAN_OPEN = "<span ";
	private static final String HTML_SPAN_CLOSE = "</span>";
	private static final String HTML_ANCHOR_OPEN = "<a ";
	private static final String HTML_ANCHOR_CLOSE = "</a>";
	private static final String HTML_DIV_OPEN = "<div";
	private static final String HTML_DIV_CLOSE = "</div>";
	private static final String HTML_BLOCKQUOTE_OPEN = "<blockquote>";
	private static final String HTML_BLOCKQUOTE_CLOSE = "</blockquote>";
	private static final String HTML_I_OPEN = "<i>";
	private static final String HTML_I_CLOSE = "</i>";
	private static final String HTML_UL_OPEN = "<ul>";
	private static final String HTML_UL_CLOSE = "</ul>";
	private static final String HTML_OL_OPEN = "<ol>";
	private static final String HTML_OL_CLOSE = "</ol>";
	private static final String HTML_LI_OPEN = "<li>";
	private static final String HTML_LI_CLOSE = "</li>";
	private static final String HTML_B_OPEN = "<b>";
	private static final String HTML_B_CLOSE = "</b>";
	private static final String HTML_P_OPEN = "<p>";
	private static final String HTML_P_CLOSE = "</p>";
	private static final String HTML_EM_OPEN = "<em>";
	private static final String HTML_EM_CLOSE = "</em>";
	private static final String HTML_BR_UNCLOSED = "<br>";
	private static final String HTML_BR_CLOSED = "<br/>";
	
	private static final String HTML_MULTIPLE_RESULT_MARKER_START = "<div class=\"searchSubSection\"><div>Results";
	private static final String HTML_GAME_TITLE_START = "<div class=\"searchTitle\">";
	private static final String HTML_GAME_END_MARKER = "<br clear=\"all\"></div>";

	private final static String MOBY_GAMES_HOST_NAME = "www.mobygames.com";
	private final static int RESULTS_PER_PAGE = 50;


	public static List<MobyProfile> getEntries(final String title, String[] platforms) throws IOException {
		int pageIdx = 0;
		int pages = 1;
		List<MobyProfile> allEntries = new ArrayList<MobyProfile>();
		
		while (pageIdx < pages) {
			String content = getResponseContent(HTTP_PROTOCOL + MOBY_GAMES_HOST_NAME + "/search/quick?ajax=1&sFilter=1&p=-1&sG=on&q=" + 
					URLEncoder.encode(title.replaceAll("/", " "), "UTF-8") + "&offset=" + (pageIdx * RESULTS_PER_PAGE));
			if (pageIdx == 0) 
				pages = getPages(content);
			allEntries.addAll(extractEntries(content));
			pageIdx++;
		}
		
		List<MobyProfile> entries = new ArrayList<MobyProfile>();
		for (MobyProfile prof: allEntries)
			if (isAllowed(prof, platforms))
				entries.add(prof);
		if (entries.isEmpty()) 
			entries = allEntries;
		Collections.sort(entries);
		return entries;
	}

	private static boolean isAllowed(final MobyProfile prof, final String[] platforms) {
		boolean allowed = (platforms.length == 0);
		for (String p: platforms) {
			if (prof.getPlatform().equalsIgnoreCase(p))
				return true;
		}
		return allowed;
	}

	private static int getPages(String htmlChunk) {
		int i = htmlChunk.indexOf('{');
		int j = htmlChunk.indexOf('}', i);
		return StringUtils.countCharacters(htmlChunk.substring(i, j), ',');
	}
	
	private static List<MobyProfile> extractEntries(String mobyGamesHttpResponse) {
		List<MobyProfile> allEntries = new ArrayList<MobyProfile>();
		mobyGamesHttpResponse = mobyGamesHttpResponse.replaceAll("\\\\\"", "\"");
		int gameMatchEntryIndex = mobyGamesHttpResponse.indexOf(HTML_MULTIPLE_RESULT_MARKER_START);
		if (gameMatchEntryIndex != -1)
			gameMatchEntryIndex += HTML_MULTIPLE_RESULT_MARKER_START.length();
		
		while (gameMatchEntryIndex != -1) {
			
			gameMatchEntryIndex = mobyGamesHttpResponse.indexOf(HTML_DIV_OPEN, gameMatchEntryIndex);
			
			int gameTitleIdx = mobyGamesHttpResponse.indexOf(HTML_GAME_TITLE_START, gameMatchEntryIndex);
			String gameTitleData = extractNextContent(mobyGamesHttpResponse, gameTitleIdx, HTML_DIV_OPEN, HTML_DIV_CLOSE);
			String gameTitle = unescapeHtml(removeAllTags(gameTitleData)).substring(6);
			String url = extractNextHrefContent(mobyGamesHttpResponse, gameTitleIdx);
			
			String details = extractNextContent(mobyGamesHttpResponse, gameTitleIdx + gameTitleData.length(), HTML_DIV_OPEN, HTML_DIV_CLOSE);
			int platformIdx = details.indexOf(HTML_SPAN_OPEN);
			
			while (platformIdx != -1) {
				String platform = extractNextContent(details, platformIdx, HTML_SPAN_OPEN, HTML_SPAN_CLOSE);
				if (platform.indexOf(HTML_ANCHOR_OPEN) != -1) {
					platform = extractNextContent(details, platformIdx, HTML_ANCHOR_OPEN, HTML_ANCHOR_CLOSE);
					url = extractNextHrefContent(details, platformIdx);
				} else {
					int yrIdx = platform.indexOf(" (");
					if (yrIdx != -1) {
						platform = platform.substring(0, yrIdx);
					}
				}
				if (!url.startsWith(HTTP_PROTOCOL)) {
					url = HTTP_PROTOCOL + MOBY_GAMES_HOST_NAME + url;
				}
				String year = extractNextContent(details, platformIdx, HTML_EM_OPEN, HTML_EM_CLOSE);

				MobyProfile gameEntry = new MobyProfile();
				gameEntry.setTitle(gameTitle);
				gameEntry.setUrl(url);
				gameEntry.setPlatform(platform);
				gameEntry.setPublisherName("");
				gameEntry.setYear(year);
				allEntries.add(gameEntry);
				platformIdx = details.indexOf(HTML_SPAN_OPEN, platformIdx + 1);
			}

			int endIdx = mobyGamesHttpResponse.indexOf(HTML_GAME_END_MARKER, gameTitleIdx);
			gameMatchEntryIndex = mobyGamesHttpResponse.indexOf(HTML_DIV_OPEN, endIdx + HTML_GAME_END_MARKER.length());
		}
		return allEntries;
	}
	
	public static int getEntryFirstMatchIndex(final String title, final List<MobyProfile> profs) {
		for (int i = 0; i < profs.size(); i++) {
			if (title.equalsIgnoreCase(profs.get(i).getTitle()))
				return i; 
		}
		return -1;
	}

	public static MobyProfile getEntry(final MobyProfile entry) throws UnknownHostException, IOException {
		MobyProfile result = new MobyProfile();

		result.setTitle(entry.getTitle());
		result.setYear(entry.getYear());
		result.setUrl(entry.getUrl());

		String responseEntry = getResponseContent(entry.getUrl());

		result.setDeveloperName(extractCategory(responseEntry, ">Developed by</div>"));
		result.setPublisherName(extractCategory(responseEntry, ">Published by</div>"));
		result.setGenre(extractCategory(responseEntry, ">Genre</div>"));
		result.setNotes(extractDescription(responseEntry));
		result.setRank(extractRank(responseEntry));

		return result;
	}
	
	private static String getResponseContent(final String url) throws IOException {
		try {
	        URL urlConnection = new URL(url);
	        BufferedReader in = new BufferedReader(new InputStreamReader(urlConnection.openStream(), "UTF-8"));
	        StringBuffer result = new StringBuffer(8192);
	        String str;
	        while ((str = in.readLine()) != null) {
	          result.append(str);
	        }
	        in.close();
	        return result.toString();
	    } catch (MalformedURLException e) {
	    	return null;
	    } 
	}
	
	private static String extractNextContent(final String htmlChunk, final int startIndex, final String openTag, final String closeTag) {
		int divStartIndex = htmlChunk.indexOf(openTag, startIndex);
		divStartIndex = htmlChunk.indexOf(">", divStartIndex) + 1;
		int divEndIndex = htmlChunk.indexOf(closeTag, divStartIndex);
		return htmlChunk.substring(divStartIndex, divEndIndex);
	}

	private static String extractNextHrefContent(final String htmlChunk, final int startIndex) {
		int aStartIndex = htmlChunk.indexOf(HTML_HREF_OPEN, startIndex) + HTML_HREF_OPEN.length();
		int aEndIndex = htmlChunk.indexOf(HTML_QUOTE, aStartIndex);
		return htmlChunk.substring(aStartIndex, aEndIndex);
	}
	
	private static String extractCategory(final String htmlChunk, final String marker) {
		int startIndex = htmlChunk.indexOf(marker);
		if (startIndex != -1) {
			int endIndex = htmlChunk.indexOf(HTML_DIV_CLOSE, startIndex + marker.length());
			return unescapeHtml(removeAllTags(htmlChunk.substring(startIndex + marker.length(), endIndex + HTML_DIV_CLOSE.length())));
		} else {
			return "";
		}
	}

	private static String extractDescription(final String htmlChunk) {
		String marker = "<h2 class=\"m5\">Description</h2>";
		int startIndex = htmlChunk.indexOf(marker) + marker.length();
		int endIndex = htmlChunk.indexOf(HTML_DIV_OPEN, startIndex);
		return unescapeHtml(removeAllTags(htmlChunk.substring(startIndex, endIndex)));
	}
	
	private static Integer extractRank(final String htmlChunk) {
		String header = "scoreBoxBig";
		int startIndex = htmlChunk.indexOf(">", htmlChunk.indexOf(header) + header.length()) + 1;
		int endIndex = htmlChunk.indexOf("<", startIndex);
		try {
			return new Integer(htmlChunk.substring(startIndex, endIndex));
		} catch (NumberFormatException e) {
			return 0;
		}
	}
	
	private static String removeAllTags(final String htmlChunk) {
		String result = removeSpecificTag(HTML_DIV_OPEN, HTML_DIV_CLOSE, htmlChunk);
		result = removeSpecificTag(HTML_ANCHOR_OPEN, HTML_ANCHOR_CLOSE, result);
		result = result.replace(HTML_I_OPEN, "").replace(HTML_I_CLOSE, "");
		result = result.replace(HTML_B_OPEN, "").replace(HTML_B_CLOSE, "");
		result = result.replace(HTML_LI_OPEN, "").replace(HTML_LI_CLOSE, "\n");
		result = result.replace(HTML_EM_OPEN, "").replace(HTML_EM_CLOSE, "");
		result = result.replace(HTML_UL_OPEN, "\n\n").replace(HTML_UL_CLOSE, "\n");
		result = result.replace(HTML_OL_OPEN, "\n\n").replace(HTML_OL_CLOSE, "\n");
		result = result.replace(HTML_BLOCKQUOTE_OPEN, "\n\n").replace(HTML_BLOCKQUOTE_CLOSE, "\n");
		result = result.replace(HTML_P_OPEN, "\n").replace(HTML_P_CLOSE, "");
		return result;
	}
	
	private static String removeSpecificTag(final String openTag, final String closeTag, final String htmlChunk) {
		StringBuffer result = new StringBuffer(htmlChunk);
		int divOpenIndex = result.indexOf(openTag);
		while (divOpenIndex != -1) {
			result.delete(divOpenIndex, result.indexOf(">", divOpenIndex + openTag.length()) + 1);
			int endIndex = result.indexOf(closeTag);
			result.delete(endIndex, endIndex + closeTag.length());
			divOpenIndex = result.indexOf(openTag);
		}
		return result.toString();
	}

	private static String unescapeHtml(final String htmlChunk) {
		return StringEscapeUtils.unescapeHtml(htmlChunk.replace(HTML_BR_UNCLOSED, "\n").replace(HTML_BR_CLOSED, "\n").replace("&nbsp;", " "));
	}
}
