/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.SortedMap;
import java.util.TreeMap;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.dbgl.model.Settings;
import org.dbgl.model.conf.Configuration;
import swing2swt.layout.BorderLayout;


public class SettingsDialog extends Dialog {

    private static final java.util.List<String> SUPPORTED_LANGUAGES =
        Arrays.asList("da", "de", "en", "es", "it", "nl", "ru", "zh", "zh_TW");

    private final static SortedMap<String, Locale> allLocales;
    static {
    	allLocales = new TreeMap<String, Locale>();
    	for (Locale loc: Locale.getAvailableLocales()) {
        	allLocales.put(loc.toString(), loc);
        }
    }

    private static final int EDITABLE_COLUMN = 0;

    public final static String[] confLocations = {
        Settings.getInstance().msg("dialog.settings.confindbgldir"), 
        Settings.getInstance().msg("dialog.settings.confingamedir") };
    public final static String[] confFilenames = {
        Settings.getInstance().msg("dialog.settings.conffilebyid"),
        Settings.getInstance().msg("dialog.settings.conffilebytitle") };
    public final static String[] buttonDisplayOptions = {
        Settings.getInstance().msg("dialog.settings.displaybuttonimageandtext"),
        Settings.getInstance().msg("dialog.settings.displaybuttontextonly"),
        Settings.getInstance().msg("dialog.settings.displaybuttonimageonly") };

    private Table visible_columns;
    private boolean changedVisColumns = false;
    private TableItem[] visibleColumns;
    private TabItem columnsTabItem;
    private Combo confFilename;
    private Combo confLocation;
    private Combo localeCombo;
    private Text port;
    private Label heightValue, columnHeightValue;
    private Scale screenshotsHeight, screenshotsColumnHeight;
    private Combo buttonDisplay;
    private int previousSelection = -1;
    private Text values, platformFilterValues, envValues;
    private Configuration conf;
    private List options;
    protected Object result;
    protected Shell shell;
    private Settings settings;


    public SettingsDialog(final Shell parent, final int style) {
        super(parent, style);
    }

    public SettingsDialog(final Shell parent) {
        this(parent, SWT.NONE);
    }

    public Object open() {
        settings = Settings.getInstance();
        conf = new Configuration(settings, false, null, false);
        createContents();
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
        result = changedVisColumns;
        return result;
    }

    protected void createContents() {
        shell = new Shell(getParent(), SWT.TITLE | SWT.CLOSE | SWT.BORDER | SWT.RESIZE | SWT.APPLICATION_MODAL);
        shell.setLayout(new BorderLayout(0, 0));
        shell.addControlListener(new SizeControlAdapter(shell, "settingsdialog"));
        shell.setText(settings.msg("dialog.settings.title"));

        final TabFolder tabFolder = new TabFolder(shell, SWT.NONE);

        final TabItem generalTabItem = new TabItem(tabFolder, SWT.NONE);
        generalTabItem.setText(settings.msg("dialog.settings.tab.general"));

        final Composite composite = new Composite(tabFolder, SWT.NONE);
        composite.setLayout(new GridLayout());
        generalTabItem.setControl(composite);

        final Group dosboxGroup = new Group(composite, SWT.NONE);
        dosboxGroup.setText(settings.msg("dialog.settings.dosbox"));
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        dosboxGroup.setLayout(gridLayout);

        final Label showConsoleLabel = new Label(dosboxGroup, SWT.NONE);
        showConsoleLabel.setText(settings.msg("dialog.settings.hidestatuswindow"));

        final Button console = new Button(dosboxGroup, SWT.CHECK);
        console.setSelection(conf.getBooleanValue("dosbox", "hideconsole"));

        final Group sendToGroup = new Group(composite, SWT.NONE);
        sendToGroup.setText(settings.msg("dialog.settings.sendto"));
        final GridLayout gridLayout_2 = new GridLayout();
        gridLayout_2.numColumns = 2;
        sendToGroup.setLayout(gridLayout_2);

        final Label enableCommLabel = new Label(sendToGroup, SWT.NONE);
        enableCommLabel.setText(settings.msg("dialog.settings.enableport"));

        final Button portEnabled = new Button(sendToGroup, SWT.CHECK);
        portEnabled.setSelection(conf.getBooleanValue("communication", "port_enabled"));
        portEnabled.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                port.setEnabled(portEnabled.getSelection());
            }
        });

        final Label portnumberLabel = new Label(sendToGroup, SWT.NONE);
        portnumberLabel.setText(settings.msg("dialog.settings.port"));

        port = new Text(sendToGroup, SWT.BORDER);
        port.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        port.setText(conf.getValue("communication", "port"));
        port.setEnabled(portEnabled.getSelection());

        final Group profileDefGroup = new Group(composite, SWT.NONE);
        profileDefGroup.setText(settings.msg("dialog.settings.profiledefaults"));
        final GridLayout gridLayout_3 = new GridLayout();
        gridLayout_3.numColumns = 3;
        profileDefGroup.setLayout(gridLayout_3);

        final Label configFileLabel = new Label(profileDefGroup, SWT.NONE);
        configFileLabel.setText(settings.msg("dialog.settings.configfile"));

        confLocation = new Combo(profileDefGroup, SWT.READ_ONLY);
        confLocation.setItems(confLocations);
        confLocation.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        confLocation.select(conf.getIntValue("profiledefaults", "confpath"));

        confFilename = new Combo(profileDefGroup, SWT.READ_ONLY);
        confFilename.setItems(confFilenames);
        confFilename.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        confFilename.select(conf.getIntValue("profiledefaults", "conffile"));

        final Group i18nGroup = new Group(composite, SWT.NONE);
        i18nGroup.setText(settings.msg("dialog.settings.i18n"));
        final GridLayout gridLayout_30 = new GridLayout();
        gridLayout_30.numColumns = 2;
        i18nGroup.setLayout(gridLayout_30);

        final Label languageLabel = new Label(i18nGroup, SWT.NONE);
        languageLabel.setText(settings.msg("dialog.settings.languagecountry"));

        localeCombo = new Combo(i18nGroup, SWT.READ_ONLY);
        Locale locale = new Locale(settings.getValue("locale", "language"), settings.getValue("locale", "country"));
        
        final SortedMap<String, Locale> locales = new TreeMap<String, Locale>();
        String locString = "";

        java.util.List<String> supportedLanguages = new ArrayList<String>(SUPPORTED_LANGUAGES);
        File[] files = new File("./plugins/i18n").listFiles();
        if (files != null) {
            for (File file: files) {
                String name = file.getName();
                if (name.startsWith("MessagesBundle_") && name.endsWith(".properties")) {
                	String code = name.substring("MessagesBundle_".length(), name.indexOf(".properties"));
                	if (code.length() > 0) {
                		supportedLanguages.add(code);
                	}
                }
            }
        }

        for (String lang: supportedLanguages) {
        	Locale loc = allLocales.get(lang);
        	if (loc != null) {
	    	    StringBuffer s = new StringBuffer(loc.getDisplayLanguage(Locale.getDefault()));
	            if (loc.getCountry().length() > 0)
	                s.append(" - ").append(loc.getDisplayCountry(Locale.getDefault()));
	            locales.put(s.toString(), loc);
	            if (loc.equals(locale)) {
	                locString = s.toString();
	            }
        	}
        }

        for (String sloc: locales.keySet()) {
            localeCombo.add(sloc);
        }
        localeCombo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        localeCombo.setText(locString);
        localeCombo.setVisibleItemCount(20);
        
        columnsTabItem = new TabItem(tabFolder, SWT.NONE);
        columnsTabItem.setText(settings.msg("dialog.settings.tab.profiletable"));

        final Composite composite2 = new Composite(tabFolder, SWT.NONE);
        composite2.setLayout(new BorderLayout(0, 0));
        columnsTabItem.setControl(composite2);

        final Group visColumnsGroup = new Group(composite2, SWT.NONE);
        visColumnsGroup.setLayout(new FillLayout());
        visColumnsGroup.setText(settings.msg("dialog.settings.visiblecolunms"));

        visible_columns = new Table(visColumnsGroup, SWT.FULL_SELECTION | SWT.BORDER | SWT.CHECK);
        visible_columns.setLinesVisible(true);

        TableColumn column1 = new TableColumn(visible_columns, SWT.NONE);
        column1.setWidth(350);

        visibleColumns = new TableItem[MainWindow.columnNames.length];

        for (int i = 0; i < MainWindow.columnNames.length; i++) {
            visibleColumns[i] = new TableItem(visible_columns, SWT.BORDER);
            visibleColumns[i].setText(MainWindow.columnNames[i]);
            visibleColumns[i].setChecked(conf.getBooleanValue("gui", "column" + (i + 1) + "visible"));
        }

        final TableEditor editor = new TableEditor(visible_columns);
        editor.horizontalAlignment = SWT.LEFT;
        editor.grabHorizontal = true;
        editor.minimumWidth = 50;

        visible_columns.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                // Clean up any previous editor control
                Control oldEditor = editor.getEditor();
                if (oldEditor != null) {
                    oldEditor.dispose();
                }

                // Identify the selected row
                TableItem item = (TableItem) event.item;
                if (item == null) {
                    return;
                }
                int idx = item.getParent().getSelectionIndex();
                if (idx < MainWindow.RO_COLUMN_NAMES || idx == 20 || idx == 21) {
                    return;
                }

                // The control that will be the editor must be a child of the table
                Text newEditor = new Text(visible_columns, SWT.NONE);
                newEditor.setText(item.getText(EDITABLE_COLUMN));
                newEditor.addModifyListener(new ModifyListener() {
                    public void modifyText(final ModifyEvent mEvent) {
                        Text text = (Text) editor.getEditor();
                        editor.getItem().setText(EDITABLE_COLUMN, text.getText());
                    }
                });
                newEditor.selectAll();
                newEditor.setFocus();
                editor.setEditor(newEditor, item, EDITABLE_COLUMN);
            }
        });

        final Group addProfGroup = new Group(composite2, SWT.NONE);
        final GridLayout gridLayout_4 = new GridLayout();
        gridLayout_4.numColumns = 2;
        addProfGroup.setLayout(gridLayout_4);
        addProfGroup.setText(settings.msg("dialog.settings.addeditduplicateprofile"));
        addProfGroup.setLayoutData(BorderLayout.SOUTH);

        final Label autoSortLabel = new Label(addProfGroup, SWT.NONE);
        autoSortLabel.setText(settings.msg("dialog.settings.autosort"));

        final Button autosort = new Button(addProfGroup, SWT.CHECK);
        autosort.setSelection(conf.getBooleanValue("gui", "autosortonupdate"));

        final TabItem dynTabItem = new TabItem(tabFolder, SWT.NONE);
        dynTabItem.setText(settings.msg("dialog.settings.tab.dynamicoptions"));

        final Composite composite_1 = new Composite(tabFolder, SWT.NONE);
        composite_1.setLayout(new FillLayout());
        dynTabItem.setControl(composite_1);

        final Group dynOptionsGroup = new Group(composite_1, SWT.NONE);
        final GridLayout gridLayout_1 = new GridLayout();
        gridLayout_1.numColumns = 2;
        dynOptionsGroup.setLayout(gridLayout_1);
        dynOptionsGroup.setText(settings.msg("dialog.settings.dynamicoptions"));

        final Label optionsLabel = new Label(dynOptionsGroup, SWT.NONE);
        optionsLabel.setText(settings.msg("dialog.settings.options"));

        final Label valuesLabel = new Label(dynOptionsGroup, SWT.NONE);
        valuesLabel.setText(settings.msg("dialog.settings.values"));

        options = new List(dynOptionsGroup, SWT.V_SCROLL | SWT.BORDER);
        options.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                storeValues();
                previousSelection = options.getSelectionIndex();
                if (previousSelection != -1) {
                	values.setText(conf.getMultilineValues("profile", options.getItem(previousSelection), values.getLineDelimiter()));
                }
            }
        });
        options.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        for (String s: conf.getItems("profile")) {
            options.add(s);
        }

        values = new Text(dynOptionsGroup, SWT.V_SCROLL | SWT.MULTI | SWT.BORDER | SWT.H_SCROLL);
        values.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        final TabItem guiTabItem = new TabItem(tabFolder, SWT.NONE);
        guiTabItem.setText(settings.msg("dialog.settings.tab.gui"));
        Composite composite1 = new Composite(tabFolder, SWT.NONE);
        GridLayout composite1Layout = new GridLayout();
        composite1Layout.makeColumnsEqualWidth = true;
        composite1.setLayout(composite1Layout);
        guiTabItem.setControl(composite1);
        Group screenshots = new Group(composite1, SWT.NONE);
        GridLayout screenshotsLayout = new GridLayout();
        screenshotsLayout.numColumns = 3;
        screenshots.setLayout(screenshotsLayout);
        GridData screenshotsLData = new GridData();
        screenshotsLData.grabExcessHorizontalSpace = true;
        screenshotsLData.horizontalAlignment = GridData.FILL;
        screenshots.setLayoutData(screenshotsLData);
        screenshots.setText(settings.msg("dialog.settings.screenshots"));
        Label heightLabel = new Label(screenshots, SWT.NONE);
        heightLabel.setText(settings.msg("dialog.settings.height"));
        GridData sshotsHeightData = new GridData();
        sshotsHeightData.grabExcessHorizontalSpace = true;
        sshotsHeightData.horizontalAlignment = GridData.FILL;
        screenshotsHeight = new Scale(screenshots, SWT.NONE);
        screenshotsHeight.setMaximum(750);
        screenshotsHeight.setMinimum(50);
        screenshotsHeight.setLayoutData(sshotsHeightData);
        screenshotsHeight.setIncrement(25);
        screenshotsHeight.setPageIncrement(100);
        screenshotsHeight.setSelection(conf.getIntValue("gui", "screenshotsheight"));
        screenshotsHeight.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent evt) {
                heightValue.setText(screenshotsHeight.getSelection() + settings.msg("dialog.settings.px"));
                heightValue.pack();
            }
        });
        heightValue = new Label(screenshots, SWT.NONE);
        heightValue.setText(screenshotsHeight.getSelection() + settings.msg("dialog.settings.px"));
        
        final Label displayFilenameLabel = new Label(screenshots, SWT.NONE);
        displayFilenameLabel.setText(settings.msg("dialog.settings.screenshotsfilename"));
        final Button displayFilename = new Button(screenshots, SWT.CHECK);
        displayFilename.setSelection(conf.getBooleanValue("gui", "screenshotsfilename"));
        
        Group screenshotsColumn = new Group(composite1, SWT.NONE);
        screenshotsColumn.setLayout(screenshotsLayout);
        screenshotsColumn.setLayoutData(screenshotsLData);
        screenshotsColumn.setText(settings.msg("dialog.settings.screenshotscolumn"));
        Label columnHeightLabel = new Label(screenshotsColumn, SWT.NONE);
        columnHeightLabel.setText(settings.msg("dialog.settings.height"));
        screenshotsColumnHeight = new Scale(screenshotsColumn, SWT.NONE);
        screenshotsColumnHeight.setMaximum(200);
        screenshotsColumnHeight.setMinimum(16);
        screenshotsColumnHeight.setLayoutData(sshotsHeightData);
        screenshotsColumnHeight.setIncrement(4);
        screenshotsColumnHeight.setPageIncrement(16);
        screenshotsColumnHeight.setSelection(conf.getIntValue("gui", "screenshotscolumnheight"));
        screenshotsColumnHeight.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent evt) {
                columnHeightValue.setText(screenshotsColumnHeight.getSelection() + settings.msg("dialog.settings.px"));
                columnHeightValue.pack();
            }
        });
        columnHeightValue = new Label(screenshotsColumn, SWT.NONE);
        columnHeightValue.setText(screenshotsColumnHeight.getSelection() + settings.msg("dialog.settings.px"));

        Group buttonsGroup = new Group(composite1, SWT.NONE);
        buttonsGroup.setLayout(gridLayout_30);
        buttonsGroup.setText(settings.msg("dialog.settings.buttons"));
        
        final Label buttonLabel = new Label(buttonsGroup, SWT.NONE);
        buttonLabel.setText(settings.msg("dialog.settings.display"));

        buttonDisplay = new Combo(buttonsGroup, SWT.READ_ONLY);
        buttonDisplay.setItems(buttonDisplayOptions);
        buttonDisplay.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        buttonDisplay.select(conf.getIntValue("gui", "buttondisplay"));
        
        final TabItem mobygamesTabItem = new TabItem(tabFolder, SWT.NONE);
        mobygamesTabItem.setText(settings.msg("dialog.settings.tab.mobygames"));
        Composite composite3 = new Composite(tabFolder, SWT.NONE);
        GridLayout composite3Layout = new GridLayout();
        composite3Layout.makeColumnsEqualWidth = true;
        composite3.setLayout(composite1Layout);
        mobygamesTabItem.setControl(composite3);
        Group consult = new Group(composite3, SWT.NONE);
        GridLayout consultLayout = new GridLayout();
        consultLayout.numColumns = 2;
        consult.setLayout(consultLayout);
        GridData consultLData = new GridData();
        consultLData.grabExcessHorizontalSpace = true;
        consultLData.horizontalAlignment = GridData.FILL;
        consultLData.grabExcessVerticalSpace = true;
        consultLData.verticalAlignment = GridData.FILL;
        consult.setLayoutData(consultLData);
        consult.setText(settings.msg("dialog.settings.consult"));
        Label titleLabel = new Label(consult, SWT.NONE);
        titleLabel.setText(settings.msg("dialog.settings.settitle"));
        final Button setTitle = new Button(consult, SWT.CHECK);
        setTitle.setSelection(conf.getBooleanValue("mobygames", "set_title"));
        Label devLabel = new Label(consult, SWT.NONE);
        devLabel.setText(settings.msg("dialog.settings.setdeveloper"));
        final Button setDev = new Button(consult, SWT.CHECK);
        setDev.setSelection(conf.getBooleanValue("mobygames", "set_developer"));
        Label pubLabel = new Label(consult, SWT.NONE);
        pubLabel.setText(settings.msg("dialog.settings.setpublisher"));
        final Button setPub = new Button(consult, SWT.CHECK);
        setPub.setSelection(conf.getBooleanValue("mobygames", "set_publisher"));
        Label yearLabel = new Label(consult, SWT.NONE);
        yearLabel.setText(settings.msg("dialog.settings.setyear"));
        final Button setYear = new Button(consult, SWT.CHECK);
        setYear.setSelection(conf.getBooleanValue("mobygames", "set_year"));
        Label genreLabel = new Label(consult, SWT.NONE);
        genreLabel.setText(settings.msg("dialog.settings.setgenre"));
        final Button setGenre = new Button(consult, SWT.CHECK);
        setGenre.setSelection(conf.getBooleanValue("mobygames", "set_genre"));
        Label linkLabel = new Label(consult, SWT.NONE);
        linkLabel.setText(settings.msg("dialog.settings.setlink"));
        final Button setLink = new Button(consult, SWT.CHECK);
        setLink.setSelection(conf.getBooleanValue("mobygames", "set_link"));
        Label rankLabel = new Label(consult, SWT.NONE);
        rankLabel.setText(settings.msg("dialog.settings.setrank", new Object[] {MainWindow.columnNames[MainWindow.RO_COLUMN_NAMES + 8]}));
        final Button setRank = new Button(consult, SWT.CHECK);
        setRank.setSelection(conf.getBooleanValue("mobygames", "set_rank"));
        Label descrLabel = new Label(consult, SWT.NONE);
        descrLabel.setText(settings.msg("dialog.settings.setdescription"));
        final Button setDescr = new Button(consult, SWT.CHECK);
        setDescr.setSelection(conf.getBooleanValue("mobygames", "set_description"));
        Label filterLabel = new Label(consult, SWT.NONE);
        filterLabel.setText(settings.msg("dialog.settings.platformfilter"));
        platformFilterValues = new Text(consult, SWT.V_SCROLL | SWT.MULTI | SWT.BORDER | SWT.H_SCROLL);
        platformFilterValues.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        platformFilterValues.setText(conf.getMultilineValues("mobygames", "platform_filter", platformFilterValues.getLineDelimiter()));
        
        final TabItem envTabItem = new TabItem(tabFolder, SWT.NONE);
        envTabItem.setText(settings.msg("dialog.settings.tab.environment"));
        Composite composite4 = new Composite(tabFolder, SWT.NONE);
        GridLayout composite4Layout = new GridLayout();
        composite4Layout.makeColumnsEqualWidth = true;
        composite4.setLayout(composite1Layout);
        envTabItem.setControl(composite4);
        Group envGroup = new Group(composite4, SWT.NONE);
        GridLayout envLayout = new GridLayout();
        envLayout.numColumns = 2;
        envGroup.setLayout(envLayout);
        GridData envLData = new GridData();
        envLData.grabExcessHorizontalSpace = true;
        envLData.horizontalAlignment = GridData.FILL;
        envLData.grabExcessVerticalSpace = true;
        envLData.verticalAlignment = GridData.FILL;
        envGroup.setLayoutData(envLData);
        envGroup.setText(settings.msg("dialog.settings.environment"));
        Label enableEnvLabel = new Label(envGroup, SWT.NONE);
        enableEnvLabel.setText(settings.msg("dialog.settings.enableenvironment"));
        final Button enableEnv = new Button(envGroup, SWT.CHECK);
        enableEnv.setSelection(conf.getBooleanValue("environment", "use"));
        enableEnv.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
            	envValues.setEnabled(enableEnv.getSelection());
            }
        });
        Label envLabel = new Label(envGroup, SWT.NONE);
        envLabel.setText(settings.msg("dialog.settings.environmentvariables"));
        envValues = new Text(envGroup, SWT.V_SCROLL | SWT.MULTI | SWT.BORDER | SWT.H_SCROLL);
        envValues.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        envValues.setText(conf.getMultilineValues("environment", "value", envValues.getLineDelimiter()));
        envValues.setEnabled(enableEnv.getSelection());

        final Composite composite_7 = new Composite(shell, SWT.NONE);
        composite_7.setLayout(new GridLayout(2, true));
        composite_7.setLayoutData(BorderLayout.SOUTH);

        final Button okButton = new Button(composite_7, SWT.NONE);
        okButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                if (!isValid()) {
                    return;
                }

                for (int i = 0; i < MainWindow.columnNames.length; i++) {
                    if (conf.getBooleanValue("gui", "column" + (i + 1) + "visible") != visibleColumns[i].getChecked()) {
                        changedVisColumns = true;
                    }
                }
                for (int i = 0; i < MainWindow.EDIT_COLUMN_NAMES; i++) {
                    if (!MainWindow.columnNames[i + MainWindow.RO_COLUMN_NAMES].equals(
                                 visibleColumns[i + MainWindow.RO_COLUMN_NAMES].getText())) {
                        changedVisColumns = true;
                    }
                }
                if (changedVisColumns) {
                    resetColumnSettings();
                }

                conf.setBooleanValue("dosbox", "hideconsole", console.getSelection());
                conf.setBooleanValue("communication", "port_enabled", portEnabled.getSelection());
                conf.setValue("communication", "port", port.getText());
                conf.setIntValue("profiledefaults", "confpath", confLocation.getSelectionIndex());
                conf.setIntValue("profiledefaults", "conffile", confFilename.getSelectionIndex());
                conf.setValue("locale", "language", locales.get(localeCombo.getText()).getLanguage());
                conf.setValue("locale", "country", locales.get(localeCombo.getText()).getCountry());
                for (int i = 0; i < MainWindow.columnNames.length; i++) {
                    conf.setBooleanValue("gui", "column" + (i + 1) + "visible", visibleColumns[i].getChecked());
                }
                conf.setBooleanValue("gui", "autosortonupdate", autosort.getSelection());
                for (int i = 0; i < MainWindow.EDIT_COLUMN_NAMES; i++) {
                    conf.setValue("gui", "custom" + (i + 1),
                            visibleColumns[i + MainWindow.RO_COLUMN_NAMES].getText());
                }
                conf.setIntValue("gui", "screenshotsheight", screenshotsHeight.getSelection());
                conf.setBooleanValue("gui", "screenshotsfilename", displayFilename.getSelection());
                conf.setIntValue("gui", "screenshotscolumnheight", screenshotsColumnHeight.getSelection());
                
                Rectangle rec = shell.getBounds();
                conf.setIntValue("gui", "settingsdialog_width", rec.width);
                conf.setIntValue("gui", "settingsdialog_height", rec.height);
                conf.setIntValue("gui", "buttondisplay", buttonDisplay.getSelectionIndex());

                conf.setBooleanValue("mobygames", "set_title", setTitle.getSelection());
                conf.setBooleanValue("mobygames", "set_developer", setDev.getSelection());
                conf.setBooleanValue("mobygames", "set_publisher", setPub.getSelection());
                conf.setBooleanValue("mobygames", "set_year", setYear.getSelection());
                conf.setBooleanValue("mobygames", "set_genre", setGenre.getSelection());
                conf.setBooleanValue("mobygames", "set_link", setLink.getSelection());
                conf.setBooleanValue("mobygames", "set_description", setDescr.getSelection());
                conf.setBooleanValue("mobygames", "set_rank", setRank.getSelection());
                conf.setMultilineValues("mobygames", "platform_filter", platformFilterValues.getText(), platformFilterValues.getLineDelimiter());

                conf.setBooleanValue("environment", "use", enableEnv.getSelection());
                conf.setMultilineValues("environment", "value", envValues.getText(), envValues.getLineDelimiter());
                
                storeValues();
                settings.updateAndExtendWithValuesFrom(conf);
                shell.close();
            }
        });
        shell.setDefaultButton(okButton);
        final GridData gridData = new GridData();
        gridData.horizontalAlignment = SWT.FILL;
        gridData.widthHint = 80;
        okButton.setLayoutData(gridData);
        okButton.setText(settings.msg("button.ok"));

        final Button cancelButton = new Button(composite_7, SWT.NONE);
        final GridData gridData_1 = new GridData();
        gridData_1.horizontalAlignment = SWT.FILL;
        cancelButton.setLayoutData(gridData_1);
        cancelButton.setText(settings.msg("button.cancel"));
        cancelButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                shell.close();
            }
        });
    }

    private void storeValues() {
        if (previousSelection != -1) {
            conf.setMultilineValues("profile", options.getItem(previousSelection), values.getText(),
                    values.getLineDelimiter());
        }
    }

    private int countVisibleColumns() {
        int result = 0;
        for (int i = 0; i < MainWindow.columnNames.length; i++) {
            if (visibleColumns[i].getChecked()) {
                result++;
            }
        }
        return result;
    }

    private int getFirstVisibleColumn() {
        for (int i = 0; i < MainWindow.columnNames.length; i++) {
            if (visibleColumns[i].getChecked()) {
                return i;
            }
        }
        return -1;
    }

    private void resetColumnSettings() {
        conf.setIntValue("gui", "sortcolumn", getFirstVisibleColumn());
        conf.setBooleanValue("gui", "sortascending", true);
        StringBuffer colOrder = new StringBuffer();
        for (int i = 0; i < countVisibleColumns(); i++) {
            colOrder.append(i).append(' ');
        }
        conf.setValue("gui", "columnorder", colOrder.toString().trim());
    }

    private boolean isValid() {
        GeneralPurposeDialogs.initErrorDialog();
        if (countVisibleColumns() == 0) {
            GeneralPurposeDialogs.addError(settings.msg("dialog.settings.required.onevisiblecolumn"), visible_columns, columnsTabItem);
        }
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }
}
