/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model;

import java.io.File;
import java.util.List;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.dbgl.model.conf.CompositeConfiguration;
import org.dbgl.util.FileUtils;
import org.dbgl.util.PlatformUtils;
import org.dbgl.util.XmlUtils;


public class ExpProfile extends Profile {
    
	private final static XPathFactory xfactory = XPathFactory.newInstance();
    private final static XPath xPath = xfactory.newXPath();
    
    private CompositeConfiguration conf;
    private File baseDir, gameDir;
    private String importedFullConfig, importedIncrConfig;
    private int importedId;
    
    
	public ExpProfile(final int id, final CompositeConfiguration conf, final File gDir, final Profile prof) {
        super(id, prof.getConfPathAndFile(), prof.getCaptures(), prof);
        this.conf = conf;
        this.baseDir = FileUtils.makeRelativeToDosroot(new File("."));
        this.gameDir = gDir;
    }
	
	public ExpProfile(final Node n, final int dbversionIndex) throws XPathExpressionException {
    	super(xPath.evaluate("title", n), Boolean.valueOf(xPath.evaluate("meta-info/favorite", n)));
    	this.importedId = Integer.valueOf(xPath.evaluate("id", n));
    	this.confPathAndFile = PlatformUtils.pathToNativePath(xPath.evaluate("config-file/raw", n));
    	this.captures = PlatformUtils.pathToNativePath(xPath.evaluate("captures/raw", n));
    	this.baseDir = FileUtils.makeRelativeToDosroot(new File("."));
    	this.gameDir = new File(PlatformUtils.pathToNativePath(xPath.evaluate("game-dir/raw", n)));
    	this.setup = PlatformUtils.pathToNativePath(xPath.evaluate("setup", n));
        this.setupParams = xPath.evaluate("setup-parameters", n);
        this.developerName = xPath.evaluate("meta-info/developer", n);
        this.publisherName = xPath.evaluate("meta-info/publisher", n);
        this.year = xPath.evaluate("meta-info/year", n);
        this.genre = xPath.evaluate("meta-info/genre", n);
        this.status = xPath.evaluate("meta-info/status", n);
        this.notes = xPath.evaluate("meta-info/notes", n);
        this.customString = new String[] { xPath.evaluate("meta-info/custom1", n),
                xPath.evaluate("meta-info/custom2", n), xPath.evaluate("meta-info/custom3", n),
                xPath.evaluate("meta-info/custom4", n), xPath.evaluate("meta-info/custom5", n),
                xPath.evaluate("meta-info/custom6", n), xPath.evaluate("meta-info/custom7", n),
                xPath.evaluate("meta-info/custom8", n) };
        this.customInt = new int[] { Integer.valueOf(xPath.evaluate("meta-info/custom9", n)),
                Integer.valueOf(xPath.evaluate("meta-info/custom10", n)) };
        this.link = new String[] { PlatformUtils.pathToNativePath(xPath.evaluate("meta-info/link1/raw", n)),
        		PlatformUtils.pathToNativePath(xPath.evaluate("meta-info/link2/raw", n)),
        		PlatformUtils.pathToNativePath(xPath.evaluate("meta-info/link3/raw", n)),
        		PlatformUtils.pathToNativePath(xPath.evaluate("meta-info/link4/raw", n)) };
        this.linkTitle = new String[] { xPath.evaluate("meta-info/link1/title", n), xPath.evaluate("meta-info/link2/title", n),
                xPath.evaluate("meta-info/link3/title", n), xPath.evaluate("meta-info/link4/title", n) };
        this.importedFullConfig = xPath.evaluate("full-configuration", n);
        this.importedIncrConfig = xPath.evaluate("incremental-configuration", n);
        this.dbversionId = dbversionIndex;
    }
	
	public void setDbversionId(final int id) {
        this.dbversionId = id;
    }
	
	public int getImportedId() {
		return importedId;
	}

	public void setImportedId(int importedId) {
		this.importedId = importedId;
	}

	public String getCapturesExport() {
        return FileUtils.constructCapturesDir(getId());
    }
	
	public File getBaseDir() {
		return baseDir;
	}

	public File getGameDir() {
		return gameDir;
	}

	public void setBaseDir(File baseDir) {
		this.baseDir = baseDir;
	}

	public void setGameDir(File gameDir) {
		this.gameDir = gameDir;
	}
	
    public String getImportedFullConfig() {
		return importedFullConfig;
	}

	public String getImportedIncrConfig() {
		return importedIncrConfig;
	}

	public CompositeConfiguration getConf() {
		return conf;
	}

	public void setConf(CompositeConfiguration conf) {
		this.conf = conf;
	}

    public Element getXml(final Document xmldoc, final List<DosboxVersion> dbversionsList) {
    	boolean listExport = (conf == null);
        Element profEmt = xmldoc.createElement("profile");
        XmlUtils.addCDataElement(xmldoc, profEmt, "title", getTitle());
        XmlUtils.addElement(xmldoc, profEmt, "id", String.valueOf(getId()));
        Element captures = xmldoc.createElement("captures");
        XmlUtils.addElement(xmldoc, captures, "raw", getCapturesExport());
        if (listExport) {
        	XmlUtils.addElement(xmldoc, captures, "url", getCapturesAsUrl());
        }
        profEmt.appendChild(captures);
        Element config = xmldoc.createElement("config-file");
        XmlUtils.addElement(xmldoc, config, "raw", FileUtils.makeRelativeToData(new File(getConfPathAndFile())).getPath());
        if (listExport) {
        	XmlUtils.addElement(xmldoc, config, "url", getConfFileAsUrl());
        }
        profEmt.appendChild(config);
        if (gameDir != null) {
            Element gamedir = xmldoc.createElement("game-dir");
            XmlUtils.addElement(xmldoc, gamedir, "raw", PlatformUtils.toDosboxPath(gameDir.getPath()));
            profEmt.appendChild(gamedir);
        }
        XmlUtils.addElement(xmldoc, profEmt, "setup", getSetup());
        XmlUtils.addElement(xmldoc, profEmt, "setup-parameters", getSetupParameters());
        Element meta = xmldoc.createElement("meta-info");
        XmlUtils.addCDataElement(xmldoc, meta, "developer", getDeveloperName());
        XmlUtils.addCDataElement(xmldoc, meta, "publisher", getPublisherName());
        XmlUtils.addCDataElement(xmldoc, meta, "year", getYear());
        XmlUtils.addCDataElement(xmldoc, meta, "genre", getGenre());
        XmlUtils.addCDataElement(xmldoc, meta, "status", getStatus());
        XmlUtils.addElement(xmldoc, meta, "favorite", String.valueOf(isDefault()));
        XmlUtils.addCDataElement(xmldoc, meta, "notes", XmlUtils.cleanEolnForXml(getNotes()));
        for (int i = 0; i < 8; i++) {
            XmlUtils.addCDataElement(xmldoc, meta, "custom" + (i + 1), getCustomString(i));
        }
        for (int i = 0; i < 2; i++) {
            XmlUtils.addElement(xmldoc, meta, "custom" + (i + 9), String.valueOf(getCustomInt(i)));
        }
        for (int i = 0; i < 4; i++) {
            Element link = xmldoc.createElement("link" + (i + 1));
            XmlUtils.addElement(xmldoc, link, "raw", getLink(i));
            if (listExport) {
            	XmlUtils.addElement(xmldoc, link, "url", getLinkAsUrl(i));
            }
            XmlUtils.addCDataElement(xmldoc, link, "title", getLinkTitle(i));
            meta.appendChild(link);
        }
        profEmt.appendChild(meta);
        if (conf != null) {
            XmlUtils.addCDataElement(xmldoc, profEmt, "full-configuration", XmlUtils.cleanEolnForXml(conf.getFullConf()));
        }
        if (conf != null) {
            XmlUtils.addCDataElement(xmldoc, profEmt, "incremental-configuration", XmlUtils.cleanEolnForXml(conf.getIncrConf()));
        }
        DosboxVersion dbv = dbversionsList.get(DosboxVersion.findById(dbversionsList, getDbversionId()));
        Element dosbox = xmldoc.createElement("dosbox");
        XmlUtils.addCDataElement(xmldoc, dosbox, "title", dbv.getValue());
        XmlUtils.addElement(xmldoc, dosbox, "version", dbv.getVersion());
        profEmt.appendChild(dosbox);
        return profEmt;
    }
}
