/*
 *  Copyright (C) 2006-2010  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model;

import java.io.File;
import java.util.List;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.dbgl.interfaces.Configurable;
import org.dbgl.model.conf.Conf;
import org.dbgl.util.FileUtils;
import org.dbgl.util.PlatformUtils;
import org.dbgl.util.XmlUtils;


public class ExpProfile extends Profile implements Configurable {
    
    private Conf conf;
    private File baseDir, gameDir;
    private String importedFullConfig, importedIncrConfig;
    private int importedId;
    
    
    public ExpProfile(final Conf conf) {
        super(null, false);
        this.conf = conf;
        this.baseDir = null;
        this.gameDir = null;
    }

	public ExpProfile(final int id, final Conf conf, final File gDir, final Profile prof) {
        super(id, prof.getConfPathAndFile(), prof.getCaptures(), prof);
        this.conf = conf;
        this.baseDir = FileUtils.makeRelativeToDosroot(new File("."));
        this.gameDir = gDir;
    }
	
	public ExpProfile(final Element n, final int dbversionIndex) {
    	super(XmlUtils.getTextValue(n, "title"), Boolean.valueOf(XmlUtils.getTextValue(XmlUtils.getNode(n, "meta-info"), "favorite")));
    	this.importedId = Integer.valueOf(XmlUtils.getTextValue(n, "id"));
    	this.confPathAndFile = PlatformUtils.pathToNativePath(XmlUtils.getTextValue(XmlUtils.getNode(n, "config-file"), "raw"));
    	this.captures = PlatformUtils.pathToNativePath(XmlUtils.getTextValue(XmlUtils.getNode(n, "captures"), "raw"));
    	this.baseDir = FileUtils.makeRelativeToDosroot(new File("."));
    	this.gameDir = new File(PlatformUtils.pathToNativePath(XmlUtils.getTextValue(XmlUtils.getNode(n, "game-dir"), "raw")));
    	this.setup = PlatformUtils.pathToNativePath(XmlUtils.getTextValue(n, "setup"));
        this.setupParams = XmlUtils.getTextValue(n, "setup-parameters");
        Element metainfo = XmlUtils.getNode(n, "meta-info");
        this.developerName = XmlUtils.getTextValue(metainfo, "developer");
        this.publisherName = XmlUtils.getTextValue(metainfo, "publisher");
        this.year = XmlUtils.getTextValue(metainfo, "year");
        this.genre = XmlUtils.getTextValue(metainfo, "genre");
        this.status = XmlUtils.getTextValue(metainfo, "status");
        this.notes = XmlUtils.getTextValue(metainfo, "notes");
        this.customString = new String[] { XmlUtils.getTextValue(metainfo, "custom1"),
        		XmlUtils.getTextValue(metainfo, "custom2"), XmlUtils.getTextValue(metainfo, "custom3"),
        		XmlUtils.getTextValue(metainfo, "custom4"), XmlUtils.getTextValue(metainfo, "custom5"),
        		XmlUtils.getTextValue(metainfo, "custom6"), XmlUtils.getTextValue(metainfo, "custom7"),
        		XmlUtils.getTextValue(metainfo, "custom8") };
        this.customInt = new int[] { Integer.valueOf(XmlUtils.getTextValue(metainfo, "custom9")),
                Integer.valueOf(XmlUtils.getTextValue(metainfo, "custom10")) };
        Element link1 = XmlUtils.getNode(metainfo, "link1");
        Element link2 = XmlUtils.getNode(metainfo, "link2");
        Element link3 = XmlUtils.getNode(metainfo, "link3");
        Element link4 = XmlUtils.getNode(metainfo, "link4");
        this.link = new String[] { PlatformUtils.pathToNativePath(XmlUtils.getTextValue(link1, "raw")),
        		PlatformUtils.pathToNativePath(XmlUtils.getTextValue(link2, "raw")),
        		PlatformUtils.pathToNativePath(XmlUtils.getTextValue(link3, "raw")),
        		PlatformUtils.pathToNativePath(XmlUtils.getTextValue(link4, "raw")) };
        this.linkTitle = new String[] { XmlUtils.getTextValue(link1, "title"), XmlUtils.getTextValue(link2, "title"),
        		XmlUtils.getTextValue(link3, "title"), XmlUtils.getTextValue(link4, "title") };
        this.importedFullConfig = XmlUtils.getTextValue(n, "full-configuration");
        this.importedIncrConfig = XmlUtils.getTextValue(n, "incremental-configuration");
        this.dbversionId = dbversionIndex;
    }
	
	public ExpProfile(final ExpProfile p1, final ExpProfile p2, final Conf conf) {
		super(p1, p2);
		this.conf = conf;
	}
	
	public ExpProfile(final ExpProfile p1, final Profile p2) {
		super(p1, p2, false);
		this.conf = p1.conf;
	}
	
	public void setDbversionId(final int id) {
        this.dbversionId = id;
    }
	
	public int getImportedId() {
		return importedId;
	}

	public void setImportedId(int importedId) {
		this.importedId = importedId;
	}

	public String getCapturesExport() {
        return FileUtils.constructCapturesDir(getId());
    }
	
	public File getBaseDir() {
		return baseDir;
	}

	public File getGameDir() {
		return gameDir;
	}

	public void setBaseDir(File baseDir) {
		this.baseDir = baseDir;
	}

	public void setGameDir(File gameDir) {
		this.gameDir = gameDir;
	}
	
    public String getImportedFullConfig() {
		return importedFullConfig;
	}

	public String getImportedIncrConfig() {
		return importedIncrConfig;
	}

	public Conf getConf() {
		return conf;
	}

	public void setConf(Conf conf) {
		this.conf = conf;
	}

    public Element getXml(final Document xmldoc, final List<DosboxVersion> dbversionsList) {
    	boolean listExport = (conf == null);
        Element profEmt = xmldoc.createElement("profile");
        XmlUtils.addCDataElement(xmldoc, profEmt, "title", getTitle());
        XmlUtils.addElement(xmldoc, profEmt, "id", String.valueOf(getId()));
        Element captures = xmldoc.createElement("captures");
        XmlUtils.addElement(xmldoc, captures, "raw", getCapturesExport());
        if (listExport) {
        	XmlUtils.addElement(xmldoc, captures, "url", getCapturesAsUrl());
        }
        profEmt.appendChild(captures);
        Element config = xmldoc.createElement("config-file");
        XmlUtils.addElement(xmldoc, config, "raw", FileUtils.makeRelativeToData(new File(getConfPathAndFile())).getPath());
        if (listExport) {
        	XmlUtils.addElement(xmldoc, config, "url", getConfFileAsUrl());
        }
        profEmt.appendChild(config);
        if (gameDir != null) {
            Element gamedir = xmldoc.createElement("game-dir");
            XmlUtils.addElement(xmldoc, gamedir, "raw", PlatformUtils.toDosboxPath(gameDir.getPath()));
            profEmt.appendChild(gamedir);
        }
        XmlUtils.addElement(xmldoc, profEmt, "setup", getSetup());
        XmlUtils.addElement(xmldoc, profEmt, "setup-parameters", getSetupParameters());
        Element meta = xmldoc.createElement("meta-info");
        XmlUtils.addCDataElement(xmldoc, meta, "developer", getDeveloperName());
        XmlUtils.addCDataElement(xmldoc, meta, "publisher", getPublisherName());
        XmlUtils.addCDataElement(xmldoc, meta, "year", getYear());
        XmlUtils.addCDataElement(xmldoc, meta, "genre", getGenre());
        XmlUtils.addCDataElement(xmldoc, meta, "status", getStatus());
        XmlUtils.addElement(xmldoc, meta, "favorite", String.valueOf(isDefault()));
        XmlUtils.addCDataElement(xmldoc, meta, "notes", XmlUtils.cleanEolnForXml(getNotes()));
        for (int i = 0; i < 8; i++) {
            XmlUtils.addCDataElement(xmldoc, meta, "custom" + (i + 1), getCustomString(i));
        }
        for (int i = 0; i < 2; i++) {
            XmlUtils.addElement(xmldoc, meta, "custom" + (i + 9), String.valueOf(getCustomInt(i)));
        }
        for (int i = 0; i < 4; i++) {
            Element link = xmldoc.createElement("link" + (i + 1));
            XmlUtils.addElement(xmldoc, link, "raw", getLink(i));
            if (listExport) {
            	XmlUtils.addElement(xmldoc, link, "url", getLinkAsUrl(i));
            }
            XmlUtils.addCDataElement(xmldoc, link, "title", getLinkTitle(i));
            meta.appendChild(link);
        }
        profEmt.appendChild(meta);
        if (conf != null) {
            XmlUtils.addCDataElement(xmldoc, profEmt, "full-configuration", XmlUtils.cleanEolnForXml(conf.toFullConfString()));
        }
        if (conf != null) {
            XmlUtils.addCDataElement(xmldoc, profEmt, "incremental-configuration", XmlUtils.cleanEolnForXml(conf.toIncrConfString()));
        }
        DosboxVersion dbv = dbversionsList.get(DosboxVersion.findById(dbversionsList, getDbversionId()));
        Element dosbox = xmldoc.createElement("dosbox");
        XmlUtils.addCDataElement(xmldoc, dosbox, "title", dbv.getValue());
        XmlUtils.addElement(xmldoc, dosbox, "version", dbv.getVersion());
        profEmt.appendChild(dosbox);
        return profEmt;
    }
}
