/*
 *  Copyright (C) 2006-2010  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui;

import java.io.File;
import java.util.List;
import org.dbgl.gui.BrowseButton.BrowseType;
import org.dbgl.gui.BrowseButton.CanonicalType;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.ExpProfile;
import org.dbgl.util.ExportThread;
import org.dbgl.util.FileUtils;
import org.dbgl.util.XmlUtils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.Document;
import swing2swt.layout.BorderLayout;


public class ExportDialog extends Wizard {

	private Text logText;
	private Button settingsOnly, fullGames, exportCapturesButton;
	private Table profilesTable;
	private Text title, notes, author, filename;
	private Button start;
	private Label profileLabel;

    private List<DosboxVersion> dbversionsList;
	private List<ExpProfile> expProfileList;
    private ExportThread exportThread;


    public ExportDialog(final Shell parent, final int style) {
        super(parent, style);
    }

    public ExportDialog(final Shell parent, final List<DosboxVersion> dbList, final java.util.List<ExpProfile> expProfiles) {
        this(parent, SWT.NONE);
        this.dbversionsList = dbList;
        this.expProfileList = expProfiles;
    }

    public Object open() {
    	createContents(settings.msg("dialog.export.title", new Object[] {expProfileList.size()}), "export");
    	
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (exportThread != null && !exportThread.isAlive() && !finish.getEnabled()) {
            	if (exportThread.isEverythingOk()) {
                	GeneralPurposeDialogs.infoMessage(shell, settings.msg("dialog.export.notice.exportok"));
                } else {
                	GeneralPurposeDialogs.warningMessage(shell, settings.msg("dialog.export.error.problem"));
                }
                profileLabel.setText(settings.msg("dialog.export.reviewlog"));
                profileLabel.pack();
                finish.setEnabled(true);
                shell.setDefaultButton(finish);
            }
            if (!display.readAndDispatch()) display.sleep();
        }
        return exportThread;
    }
    
    private Control page1() {
    	final Group group1 = new Group(shell, SWT.NONE);
        group1.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        group1.setText(settings.msg("dialog.export.step1"));
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        group1.setLayout(gridLayout);
        
        final Label settingsOnlyLabel = new Label(group1, SWT.NONE);
        settingsOnlyLabel.setText(settings.msg("dialog.export.export"));
        settingsOnly = new Button(group1, SWT.RADIO);
        settingsOnly.setText(settings.msg("dialog.export.export.profiles"));
        settingsOnly.setSelection(true);
        
        new Label(group1, SWT.NONE);
        fullGames = new Button(group1, SWT.RADIO);
        fullGames.setText(settings.msg("dialog.export.export.games"));
        
        new Label(group1, SWT.NONE);
        exportCapturesButton = new Button(group1, SWT.CHECK);
        exportCapturesButton.setText(settings.msg("dialog.template.captures"));
        exportCapturesButton.setSelection(true);
        return group1;
    }
    
    private Control page2() {
    	final Group reviewDirsGroup = new Group(shell, SWT.NONE);
    	reviewDirsGroup.setText(settings.msg("dialog.export.step2"));
    	reviewDirsGroup.setLayout(new GridLayout());
    	
    	profilesTable = new Table(reviewDirsGroup, SWT.FULL_SELECTION | SWT.BORDER);
        profilesTable.setHeaderVisible(true);
        final GridData gridData_1 = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 2);
        gridData_1.heightHint = 80;
        profilesTable.setLayoutData(gridData_1);
        profilesTable.setLinesVisible(true);

        final TableColumn titleColumn = new TableColumn(profilesTable, SWT.NONE);
        titleColumn.setWidth(260);
        titleColumn.setText(settings.msg("dialog.main.profiles.column.title"));

        final TableColumn subdirColumn = new TableColumn(profilesTable, SWT.NONE);
        subdirColumn.setWidth(120);
        subdirColumn.setText(settings.msg("dialog.export.column.gamedir"));
		
        for (ExpProfile expProfile: expProfileList) {
        	TableItem item = new TableItem(profilesTable, SWT.NONE);
            item.setText(expProfile.getTitle());
            item.setText(1, expProfile.getGameDir().getPath());
        }
        
        profilesTable.addMouseListener(new MouseAdapter() {
			public void mouseDoubleClick(final MouseEvent event) {
			    int idx = profilesTable.getSelectionIndex();
			    ExpProfile prof = expProfileList.get(idx);
			    DirectoryDialog dialog = new DirectoryDialog(shell);
                dialog.setFilterPath(FileUtils.canonicalToDosroot(prof.getGameDir().getPath()).getPath());
                String result = dialog.open();
                if (result != null) {
                	File newGameDir = FileUtils.makeRelativeToDosroot(new File(result));
                	prof.setGameDir(newGameDir);
                    profilesTable.getSelection()[0].setText(1, newGameDir.getPath());
                }
			}
		});
        return reviewDirsGroup;
    }
    
    private Control page3() {
    	final Group settingsGroup = new Group(shell, SWT.NONE);
        final GridLayout gridLayout2 = new GridLayout();
        gridLayout2.numColumns = 3;
        settingsGroup.setLayout(gridLayout2);
        settingsGroup.setText(settings.msg("dialog.export.step3"));
        settingsGroup.setLayoutData(BorderLayout.NORTH);

        final Label titleLabel = new Label(settingsGroup, SWT.NONE);
        titleLabel.setText(settings.msg("dialog.export.exporttitle"));

        title = new Text(settingsGroup, SWT.BORDER);
        title.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));

        final Label authorLabel = new Label(settingsGroup, SWT.NONE);
        authorLabel.setText(settings.msg("dialog.export.author"));

        author = new Text(settingsGroup, SWT.BORDER);
        author.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
        
        final Label notesLabel = new Label(settingsGroup, SWT.NONE);
        notesLabel.setText(settings.msg("dialog.export.notes"));

        notes = new Text(settingsGroup, SWT.WRAP | SWT.V_SCROLL | SWT.MULTI | SWT.H_SCROLL | SWT.BORDER);
        final GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 2, 2);
        gridData.heightHint = 40;
        notes.setLayoutData(gridData);
        new Label(settingsGroup, SWT.NONE);

        final Label fileLabel = new Label(settingsGroup, SWT.NONE);
        fileLabel.setText(settings.msg("dialog.export.file"));

        filename = new Text(settingsGroup, SWT.BORDER);
        filename.setText(FileUtils.EXPORT_DIR + "games.dbgl.zip");
        filename.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        final BrowseButton browseButton = new BrowseButton(settingsGroup, SWT.NONE);
        browseButton.connect(shell, filename, null, BrowseType.FILE, CanonicalType.DBGLZIP, true, null);
        return settingsGroup;
    }
    
    private Control page4() {
    	final Group progressGroup = new Group(shell, SWT.NONE);
    	progressGroup.setText(settings.msg("dialog.export.step4"));
    	progressGroup.setLayout(new GridLayout());
    	
    	start = new Button(progressGroup, SWT.NONE);
        shell.setDefaultButton(start);
        start.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
        start.setText(settings.msg("dialog.export.start"));

        final ProgressBar progressBar = new ProgressBar(progressGroup, SWT.NONE);
        progressBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));

        profileLabel = new Label(progressGroup, SWT.NONE);

        logText = new Text(progressGroup, SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER);
        logText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        start.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                try {
                	back.setEnabled(false);
                	start.setEnabled(false);
                	cancel.setEnabled(false);
                    Document doc = XmlUtils.getFullProfilesXML(expProfileList, dbversionsList, title.getText(),
                            notes.getText(), author.getText(), exportCapturesButton.getSelection(),
                            fullGames.getSelection());
                    exportThread = new ExportThread(logText, progressBar, profileLabel,
                    		expProfileList, doc, exportCapturesButton.getSelection(),
                            fullGames.getSelection(), FileUtils.canonicalToData(filename.getText()));
                    exportThread.start();
                } catch (Exception ex) {
                    GeneralPurposeDialogs.warningMessage(shell, ex);
                    exportThread = null;
                }
            }
        });
        return progressGroup;
    }
    
    protected void fillPages() {
		addStep(page1());
        addStep(page2());
		addStep(page3());
	    addStep(page4());
	}
    
    protected int stepSize(boolean up) {
		return ((((stepNr == 0 && up) || (stepNr == 2 && !up)) && settingsOnly.getSelection())? 2: 1);
	}
    
    protected boolean isValid() {
    	if (stepNr == 1) {
			if (!isValidGameDirs()) return false;
		} else if (stepNr == 2) {
			if (!isValidTargetZip()) return false;
		}
    	return true;
    }
    
    private boolean isValidTargetZip() {
        GeneralPurposeDialogs.initErrorDialog();
        if (title.getText().equals("")) {
        	GeneralPurposeDialogs.addError(settings.msg("dialog.export.required.title"), title);
        }
        String f = filename.getText();
        if (f.equals(""))
            GeneralPurposeDialogs.addError(settings.msg("dialog.export.required.filename"), filename);
        else if (FileUtils.isExistingFile(FileUtils.canonicalToData(f)))
            GeneralPurposeDialogs.addError(settings.msg("dialog.export.error.fileexists", new Object[] {FileUtils.canonicalToData(f)}), filename);
        else {
            File dir = FileUtils.canonicalToData(f).getParentFile();
            if (dir == null || !dir.exists()) 
                GeneralPurposeDialogs.addError(settings.msg("dialog.export.error.exportdirmissing", new Object[] {FileUtils.canonicalToData(f)}), filename);
        }
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }
    
    private boolean isValidGameDirs() {
        GeneralPurposeDialogs.initErrorDialog();
        for (ExpProfile prof: expProfileList) {
        	if (!FileUtils.canonicalToDosroot(prof.getGameDir().getPath()).exists()) {
        		GeneralPurposeDialogs.addError(settings.msg("dialog.export.error.gamedirmissing", new Object[] {prof.getGameDir()}), profilesTable);
        	}
        	if (prof.getGameDir().isAbsolute()) {
        		GeneralPurposeDialogs.addError(settings.msg("dialog.export.error.gamedirnotrelative", new Object[] {prof.getGameDir()}), profilesTable);
        	}
        	for (ExpProfile prof2: expProfileList) {
        		if (prof != prof2 && FileUtils.areRelated(prof.getGameDir(), prof2.getGameDir())) {
        			GeneralPurposeDialogs.addError(settings.msg("dialog.export.error.gamedirsconflict", new Object[] {prof.getGameDir(), prof.getTitle(), prof2.getGameDir(), prof2.getTitle()}), profilesTable);;
        		}
        	}
        }
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }
}
