/*
 *  Copyright (C) 2006-2010  Ronald Blankendaal
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util.searchengine;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.dbgl.model.SearchEngineImageInformation;
import org.dbgl.model.WebProfile;
import org.apache.commons.lang.StringEscapeUtils;


public abstract class WebSearchEngine {

	protected static final String HTTP_PROTOCOL = "http://";
	protected static final String HTML_HREF_OPEN = " href=\"";
	protected static final String HTML_SRC_OPEN = " src=\"";
	protected static final String HTML_QUOTE = "\"";

	protected static final String HTML_MOBY_OPEN = "</moby ";
	protected static final String HTML_MOBY_CLOSE = "</moby>";
	protected static final String HTML_SPAN_OPEN = "<span ";
	protected static final String HTML_SPAN_CLOSE = "</span>";
	protected static final String HTML_ANCHOR_OPEN = "<a ";
	protected static final String HTML_ANCHOR_CLOSE = "</a>";
	protected static final String HTML_DIV_OPEN = "<div";
	protected static final String HTML_DIV_CLOSE = "</div>";
	protected static final String HTML_BLOCKQUOTE_OPEN = "<blockquote>";
	protected static final String HTML_BLOCKQUOTE_CLOSE = "</blockquote>";
	protected static final String HTML_I_OPEN = "<i>";
	protected static final String HTML_I_CLOSE = "</i>";
	protected static final String HTML_UL_OPEN = "<ul>";
	protected static final String HTML_UL_CLOSE = "</ul>";
	protected static final String HTML_OL_OPEN = "<ol>";
	protected static final String HTML_OL_CLOSE = "</ol>";
	protected static final String HTML_LI_OPEN = "<li>";
	protected static final String HTML_LI_CLOSE = "</li>";
	protected static final String HTML_B_OPEN = "<b>";
	protected static final String HTML_B_CLOSE = "</b>";
	protected static final String HTML_P_OPEN = "<p>";
	protected static final String HTML_PU_OPEN = "<p ";
	protected static final String HTML_P_CLOSE = "</p>";
	protected static final String HTML_EM_OPEN = "<em>";
	protected static final String HTML_EM_CLOSE = "</em>";
	protected static final String HTML_BR_UNCLOSED = "<br>";
	protected static final String HTML_BR_CLOSED = "<br/>";
	protected static final String HTML_TD_OPEN = "<td>";
	protected static final String HTML_TD_CLOSE = "</td>";


	public abstract String getIcon();
	public abstract String getName();
	public abstract String getSimpleName();
	public abstract WebProfile getEntryDetailedInformation(final WebProfile entry) throws UnknownHostException, IOException;
	public abstract SearchEngineImageInformation[] getEntryImages(final WebProfile entry, int coverArtMax, int screenshotsMax) throws IOException;
	public abstract List<WebProfile> getEntries(final String title, String[] platforms) throws IOException;
	
	
	public static int getEntryFirstMatchIndex(final String title, final List<WebProfile> profs) {
		for (int i = 0; i < profs.size(); i++) {
			if (title.equalsIgnoreCase(profs.get(i).getTitle()))
				return i;
		}
		return -1;
	}

	protected static String getResponseContent(final String url) throws IOException {
		try {
			URL urlConnection = new URL(url);
			BufferedReader in = new BufferedReader(new InputStreamReader(urlConnection.openStream(), "UTF-8"));
			StringBuffer result = new StringBuffer(8192);
			String str;
			while ((str = in.readLine()) != null) {
				result.append(str);
			}
			in.close();
			return result.toString();
		} catch (MalformedURLException e) {
			return null;
		}
	}

	protected static String absoluteUrl(final String hostName, final String url) {
		return url.startsWith(HTTP_PROTOCOL) ? url : HTTP_PROTOCOL + hostName + (url.charAt(0)=='/'? "":'/') + url;
	}

	protected static String extractNextContent(final String htmlChunk, final int startIndex, final String openTag, final String closeTag) {
		int divStartIndex = htmlChunk.indexOf(openTag, startIndex);
		divStartIndex = htmlChunk.indexOf(">", divStartIndex) + 1;
		int divEndIndex = htmlChunk.indexOf(closeTag, divStartIndex);
		return htmlChunk.substring(divStartIndex, divEndIndex);
	}

	protected static String extractNextHrefContent(final String htmlChunk, final int startIndex) {
		int aStartIndex = htmlChunk.indexOf(HTML_HREF_OPEN, startIndex) + HTML_HREF_OPEN.length();
		int aEndIndex = htmlChunk.indexOf(HTML_QUOTE, aStartIndex);
		return htmlChunk.substring(aStartIndex, aEndIndex);
	}

	protected static String extractNextSrcContent(final String htmlChunk, final int startIndex) {
		int aStartIndex = htmlChunk.indexOf(HTML_SRC_OPEN, startIndex) + HTML_SRC_OPEN.length();
		int aEndIndex = htmlChunk.indexOf(HTML_QUOTE, aStartIndex);
		return htmlChunk.substring(aStartIndex, aEndIndex);
	}
	
	protected static String removeAllTags(final String htmlChunk) {
		String result = removeSpecificTag(HTML_DIV_OPEN, HTML_DIV_CLOSE, htmlChunk);
		result = removeSpecificTag(HTML_ANCHOR_OPEN, HTML_ANCHOR_CLOSE, result);
		result = removeSpecificTag(HTML_MOBY_OPEN, HTML_MOBY_CLOSE, result);
		result = result.replace(HTML_I_OPEN, "").replace(HTML_I_CLOSE, "");
		result = result.replace(HTML_B_OPEN, "").replace(HTML_B_CLOSE, "");
		result = result.replace(HTML_LI_OPEN, "").replace(HTML_LI_CLOSE, "\n");
		result = result.replace(HTML_EM_OPEN, "").replace(HTML_EM_CLOSE, "");
		result = result.replace(HTML_UL_OPEN, "\n\n").replace(HTML_UL_CLOSE, "\n");
		result = result.replace(HTML_OL_OPEN, "\n\n").replace(HTML_OL_CLOSE, "\n");
		result = result.replace(HTML_BLOCKQUOTE_OPEN, "\n\n").replace(HTML_BLOCKQUOTE_CLOSE, "\n");
		result = result.replace(HTML_P_OPEN, "\n").replace(HTML_P_CLOSE, "");
		return result;
	}

	protected static String removeSpecificTag(final String openTag, final String closeTag, final String htmlChunk) {
		StringBuffer result = new StringBuffer(htmlChunk);
		int openingIndex = result.indexOf(openTag);
		while (openingIndex != -1) {
			result.delete(openingIndex, result.indexOf(">", openingIndex + openTag.length()) + 1);
			int closingIndex = result.indexOf(closeTag);
			result.delete(closingIndex, closingIndex + closeTag.length());
			openingIndex = result.indexOf(openTag);
		}
		return result.toString();
	}

	protected static String unescapeHtml(final String htmlChunk) {
		return StringEscapeUtils.unescapeHtml(htmlChunk.replace(HTML_BR_UNCLOSED, "\n").replace(HTML_BR_CLOSED, "\n").replace("&nbsp;", " "));
	}
	
	private static boolean isAllowed(final WebProfile prof, final String[] platforms) {
		boolean allowed = (platforms.length == 0);
		for (String p: platforms) {
			if (prof.getPlatform().equalsIgnoreCase(p))
				return true;
		}
		return allowed;
	}
	
	protected static List<WebProfile> filterEntries(String[] platforms, List<WebProfile> allEntries) {
		List<WebProfile> entries = new ArrayList<WebProfile>();
		for (WebProfile prof: allEntries)
			if (isAllowed(prof, platforms))
				entries.add(prof);
		if (entries.isEmpty())
			entries = allEntries;
		Collections.sort(entries);
		return entries;
	}
}