/*
 *  Copyright (C) 2006-2011  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui;

import java.io.File;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.dbgl.model.conf.Settings;
import org.dbgl.swtdesigner.SWTImageManager;
import org.dbgl.util.FileUtils;
import org.dbgl.util.StringUtils;


public final class BrowseButton {

    public static final String DFEND_PATH = "C:\\Program Files\\D-Fend\\";
    public static final String DFEND_PROFILES = "Profiles.dat";

    public enum BrowseType { DIR, FILE };
    public enum CanonicalType { DOSROOT, DFEND, CDIMAGE, ZIP, DBGLZIP, DOSBOX, DOSBOXCONF, DOC, BOOTER, EXE, NONE };

    private final Button button;
    private Settings settings;


    private class Sa extends SelectionAdapter {

        private final Text field;
        private final Text alt;
        private final BrowseType browse;
        private final CanonicalType canon;
        private final boolean save;
        private final Combo combo;
        private final Shell shell;
        

        public Sa(final Shell shell, final Text control, final Text altControl, final BrowseType browse,
                final CanonicalType canon, final boolean save, final Combo combo) {
            this.shell = shell;
            this.field = control;
            this.alt = altControl;
            this.browse = browse;
            this.canon = canon;
            this.save = save;
            this.combo = combo;
        }

        private String filterPath() {
            String path = field.getText();
            CanonicalType fileType = canon;
            if (path.equals("") && (alt != null)) {
                path = alt.getText();
                fileType = CanonicalType.DOSROOT;
            }
            switch (fileType) {
                case DOC:
                case DBGLZIP:
                    return FileUtils.canonicalToData(path).getPath();
                case EXE:
                case ZIP:
                case BOOTER:
                case DOSROOT:
                    return FileUtils.canonicalToDosroot(path).getPath();
                case DOSBOX:
                case DOSBOXCONF:
                    return FileUtils.canonicalToDosbox(path).getPath();
                case DFEND:
                    return DFEND_PATH;
                case CDIMAGE:
                    String[] fPaths = StringUtils.textAreaToStringArray(field.getText(), field.getLineDelimiter());
                    if (fPaths.length > 0) {
                        return FileUtils.canonicalToDosroot(fPaths[0]).getPath();
                    }
                default:
                    return "";
            }
        }

        private String getResult(final String result, final File filterPath, final String[] filenames) {
            switch (canon) {
                case DOC:
                case DBGLZIP:
                    return FileUtils.makeRelativeToData(new File(result)).getPath();
                case EXE:
                    if (FileUtils.isArchive(result) || FileUtils.isIsoFile(result)) {
                        BrowseArchiveDialog dialog = new BrowseArchiveDialog(shell, SWT.OPEN);
                        dialog.setFileToBrowse(result);
                        String choice = (String)dialog.open();
                        return (choice == null)? null: FileUtils.makeRelativeToDosroot(new File(choice)).getPath();
                    }
                case ZIP:
                case BOOTER:
                case DOSROOT:
                    return FileUtils.makeRelativeToDosroot(new File(result)).getPath();
                case DOSBOX:
                case DOSBOXCONF:
                    return FileUtils.makeRelativeToDosbox(new File(result)).getPath();
                case CDIMAGE:
                    File path = FileUtils.makeRelativeToDosroot(filterPath);
                    StringBuffer images = new StringBuffer();
                    for (String file: filenames) {
                        images.append(new File(path, file)).append(field.getLineDelimiter());
                    }
                    return images.toString();
                default:
                    return result;
            }
        }

        public void widgetSelected(final SelectionEvent event) {
            String result = null;
            File filterPath = null;
            String[] filenames = null;

            shell.setEnabled(false);

            if (browse == BrowseType.DIR) {
                DirectoryDialog dialog = new DirectoryDialog(shell);
                dialog.setFilterPath(filterPath());
                result = dialog.open();
            } else if (browse == BrowseType.FILE && (canon == CanonicalType.EXE) && 
                    (FileUtils.isArchive(filterPath()) || FileUtils.isIsoFile(filterPath()))) {
                result = filterPath();
            } else if (browse == BrowseType.FILE) {
                int style = (canon == CanonicalType.CDIMAGE) ? SWT.OPEN | SWT.MULTI: SWT.OPEN;
                if (save) style = SWT.SAVE;
                FileDialog dialog = new FileDialog(shell, style);

                String[] filterNames = null;
                String[] filterExts = null;
                switch (canon) {
                    case DOC:
                        filterNames = new String[] { FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.ALL_FILTER };
                        break;
                    case EXE:
                        String defFilterTitle = settings.msg("filetype.exe");
                        String defaultFilter = FileUtils.EXE_FILTER;
                        if (combo.isEnabled()) { // check to see if this dbversion support physfs (Gulikoza/Ykhwong)
                            defFilterTitle += ", " + settings.msg("filetype.archive");
                            defaultFilter += ";" + FileUtils.ARC_FILTER;
                        }
                        filterNames = new String[] { defFilterTitle, settings.msg("filetype.cdimage"), settings.msg("filetype.archive") };
                        filterExts = new String[] { defaultFilter, FileUtils.CDI_FILTER, FileUtils.ARC_FILTER };
                        break;
                    case ZIP:
                        filterNames = new String[] { settings.msg("filetype.archive"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.ARC_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case DBGLZIP:
                        filterNames = new String[] { settings.msg("filetype.gamepack"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.DBGLZIP_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case BOOTER:
                        filterNames = new String[] { settings.msg("filetype.booterimage"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.BTR_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case DFEND:
                        filterNames = new String[] { settings.msg("filetype.dfendprofile") };
                        filterExts = new String[] { DFEND_PROFILES };
                        break;
                    case CDIMAGE:
                        filterNames = new String[] { settings.msg("filetype.cdimage"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.CDI_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case DOSBOXCONF:
                        filterNames = new String[] { settings.msg("filetype.conf"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.CNF_FILTER, FileUtils.ALL_FILTER };
                        break;
                    default:
                }
                if (filterNames != null) {
                    dialog.setFilterNames(filterNames);
                }
                if (filterExts != null) {
                    dialog.setFilterExtensions(filterExts);
                }

                if (canon == CanonicalType.DFEND) {
                    dialog.setFileName(DFEND_PROFILES);
                }

                File fpath = new File(filterPath());
                if (!fpath.isDirectory()) {
                	fpath = fpath.getParentFile();
                }
                dialog.setFilterPath(fpath.getPath());

                result = dialog.open();
                if (canon == CanonicalType.CDIMAGE) {
                    filterPath = new File(dialog.getFilterPath());
                    filenames = dialog.getFileNames();
                }
            }
            if (result != null) {
                result = getResult(result, filterPath, filenames);
                if (result != null) {
                    field.setText(result);
                    if ((canon == CanonicalType.DOSBOX) && (alt != null)) {
                    	alt.setText(FileUtils.constructRelativeDBConfLocation(result).getPath());
                    }
                }
            }
 
            try { Thread.sleep(100); } catch (InterruptedException e) {}
            while (shell.getDisplay().readAndDispatch());
            shell.setEnabled(true);
        }
    }

    public BrowseButton(final Composite composite, final int style) {
        settings = Settings.getInstance();
        button = GeneralPurposeGUI.createIconButton(composite, style, settings, settings.msg("button.browse"), SWTImageManager.IMG_FOLDER);
    }

    public void connect(final Shell shell, final Text control, final Text altControl, final BrowseType browse, final CanonicalType canon, final boolean save, final Combo combo) {
        button.addSelectionListener(new Sa(shell, control, altControl, browse, canon, save, combo));
    }

    public void setLayoutData(final Object arg0) {
        button.setLayoutData(arg0);
    }

    public void setEnabled(final boolean enabled) {
        button.setEnabled(enabled);
    }
}
