/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.controls;

import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Stream;
import org.dbgl.model.aggregate.Profile;
import org.dbgl.model.aggregate.Template;
import org.dbgl.model.conf.Autoexec;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;


public class MetaControl extends ChangeTrackingControl {

	@FunctionalInterface
	public interface TriConsumer<T, U, V> {
		public void accept(T t, U u, V v);

		public default TriConsumer<T, U, V> andThen(TriConsumer<? super T, ? super U, ? super V> after) {
			Objects.requireNonNull(after);
			return (a, b, c) -> {
				accept(a, b, c);
				after.accept(a, b, c);
			};
		}
	}

	private Function<Template, String> templateStringGetter_;
	private BiConsumer<Template, String> templateStringUpdater_;
	private Function<Profile, String> profileStringGetter_;
	private BiFunction<Profile, Integer, String> profileIndexedStringGetter_;
	private BiConsumer<Profile, String> profileStringUpdater_;
	private TriConsumer<Profile, Integer, String> profileIndexedStringUpdater_;
	private Function<Autoexec, String> autoexecStringGetter_;
	private BiConsumer<Autoexec, String> autoexecStringUpdater_;
	private BiFunction<Autoexec, Integer, String> autoexecIndexedStringGetter_;
	private TriConsumer<Autoexec, Integer, String> autoexecIndexedStringUpdater_;
	private int index_;

	public MetaControl(Label label, Control control) {
		super(label, new Control[] {control});
	}

	public MetaControl connectTemplate(Function<Template, String> getMethod, BiConsumer<Template, String> updateMethod) {
		templateStringGetter_ = getMethod;
		templateStringUpdater_ = updateMethod;
		return this;
	}

	public MetaControl connectProfile(Function<Profile, String> getMethod, BiConsumer<Profile, String> updateMethod) {
		profileStringGetter_ = getMethod;
		profileStringUpdater_ = updateMethod;
		return this;
	}

	public MetaControl connectProfile(int i, BiFunction<Profile, Integer, String> getMethod, TriConsumer<Profile, Integer, String> updateMethod) {
		index_ = i;
		profileIndexedStringGetter_ = getMethod;
		profileIndexedStringUpdater_ = updateMethod;
		return this;
	}

	public MetaControl connectAutoexec(Function<Autoexec, String> getMethod, BiConsumer<Autoexec, String> updateMethod) {
		autoexecStringGetter_ = getMethod;
		autoexecStringUpdater_ = updateMethod;
		return this;
	}

	public MetaControl connectAutoexec(int i, BiFunction<Autoexec, Integer, String> getMethod, TriConsumer<Autoexec, Integer, String> updateMethod) {
		index_ = i;
		autoexecIndexedStringGetter_ = getMethod;
		autoexecIndexedStringUpdater_ = updateMethod;
		return this;
	}

	@Override
	public String getCurrentValue() {
		String[] fieldValues = Stream.of(controls_).map(x -> getFieldValue(x, false)).filter(x -> x != null).toArray(String[]::new);
		return fieldValues.length == controls_.length ? defaultConvertor.toConfValue(fieldValues): null;
	}

	private void internalSetControl(String value, boolean multiEdit) {
		if (value == null) {
			setOriginalValue(value, multiEdit);
		} else {
			setFieldIfEnabled(value, controls_[0], false);
			setOriginalValue(getCurrentValue(), multiEdit);
		}
	}

	public void setControlByTemplate(Template template, boolean multiEdit) {
		if (allControlsDisabled())
			return;
		internalSetControl(templateStringGetter_.apply(template), multiEdit);
	}

	public void setControlByProfile(Profile profile, boolean multiEdit) {
		if (allControlsDisabled())
			return;
		internalSetControl((profileIndexedStringGetter_ != null) ? profileIndexedStringGetter_.apply(profile, index_): profileStringGetter_.apply(profile), multiEdit);
	}

	public void setControlByAutoexec(Autoexec overrideAuto, Autoexec combinedAuto, boolean multiEdit) {
		if (allControlsDisabled())
			return;
		Autoexec autoexec = multiEdit ? overrideAuto: combinedAuto;
		internalSetControl((autoexecIndexedStringGetter_ != null) ? autoexecIndexedStringGetter_.apply(autoexec, index_): autoexecStringGetter_.apply(autoexec), multiEdit);
	}

	public void updateTemplateByControl(Template template) {
		templateStringUpdater_.accept(template, getCurrentValue());
	}

	public void updateProfileByControl(Profile profile) {
		if (profileIndexedStringUpdater_ != null)
			profileIndexedStringUpdater_.accept(profile, index_, getCurrentValue());
		else
			profileStringUpdater_.accept(profile, getCurrentValue());
	}

	public void updateAutoexecByControl(Autoexec autoexec) {
		if (autoexecIndexedStringUpdater_ != null)
			autoexecIndexedStringUpdater_.accept(autoexec, index_, getCurrentValue());
		else
			autoexecStringUpdater_.accept(autoexec, getCurrentValue());
	}

	@Override
	public String toString() {
		return label_.getText();
	}
}
