/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.dialog;

import java.util.stream.Stream;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import org.apache.commons.lang3.StringUtils;
import org.dbgl.gui.GeneralPurposeDialogs;
import org.dbgl.gui.abstractdialog.SizeControlledTabbedDialog;
import org.dbgl.model.aggregate.Profile;
import org.dbgl.model.conf.Configuration;
import org.dbgl.model.entity.SharedConf;
import org.dbgl.util.StringRelatedUtils;
import org.dbgl.util.SystemUtils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;


public class ShareConfDialog extends SizeControlledTabbedDialog<Object> {

	private SharedConf sharedConf_;

	public ShareConfDialog(Shell parent, String gameTitle, String gameYear, Profile profile) {
		super(parent, "shareconfdialog");

		sharedConf_ = new SharedConf(StringUtils.EMPTY, StringUtils.EMPTY, gameTitle, StringUtils.EMPTY, gameYear, profile.getConfigurationForSharing().toString(null),
				profile.getCombinedConfiguration().toString(null), StringUtils.EMPTY, profile.getDosboxVersion().getTitle(), profile.getDosboxVersion().getVersion());
	}

	@Override
	protected String getDialogTitle() {
		return text_.get("dialog.confsharing.title");
	}

	@Override
	protected void onShellCreated() {
		Composite composite = createTabWithComposite(text_.get("dialog.confsharing.tab.info"), 3);

		Text author = createLabelAndText(composite, text_.get("dialog.confsharing.author"), 2);
		Text gameTitle = createLabelAndText(composite, text_.get("dialog.confsharing.gametitle"), 2);
		Text gameVersion = createLabelAndText(composite, text_.get("dialog.confsharing.gameversion"), 2);
		Text gameYear = createLabelAndText(composite, text_.get("dialog.confsharing.gameyear"), 2);
		createLabel(composite, text_.get("dialog.confsharing.explanation"));
		Tree incrConf = new Tree(composite, SWT.BORDER | SWT.CHECK);
		incrConf.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, true));
		Text explanation = createTextarea(composite, false, true, 1, 1, null);
		Text notes = createLabelAndTextarea(composite, text_.get("dialog.confsharing.notes"), false, true, 2, 1);

		createOkCancelButtons(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				if (!isValid(author, gameTitle, gameYear, explanation)) {
					return;
				}
				try {
					sharedConf_.setAuthor(author.getText());
					sharedConf_.setGameTitle(gameTitle.getText());
					sharedConf_.setGameVersion(gameVersion.getText());
					sharedConf_.setGameYear(gameYear.getText());
					sharedConf_.setIncrConf(extractConfFromTree(incrConf));
					sharedConf_.setExplanation(explanation.getText());
					sharedConf_.setNotes(notes.getText());

					Client client = ClientBuilder.newClient();
					SharedConf result = client.target(settings_.getValue("confsharing", "endpoint")).path("/submissions").request().post(Entity.entity(sharedConf_, MediaType.APPLICATION_XML),
						SharedConf.class);
					GeneralPurposeDialogs.infoMessage(shell_, text_.get("dialog.confsharing.confirmation", new Object[] {result.getGameTitle()}));
					client.close();
				} catch (Exception e) {
					GeneralPurposeDialogs.fatalMessage(shell_, text_.get("dialog.confsharing.error.submit", new Object[] {StringRelatedUtils.toString(e)}), e);
				}
				settings_.setValue("confsharing", "author", author.getText());
				shell_.close();
			}
		});

		// init values
		author.setText(StringUtils.defaultString(settings_.getValue("confsharing", "author")));
		gameTitle.setText(sharedConf_.getGameTitle());
		gameYear.setText(sharedConf_.getGameYear());
		fillStringIntoTree(sharedConf_.getIncrConf(), incrConf);

		if (StringUtils.isBlank(sharedConf_.getIncrConf()))
			explanation.setText("N/A");

		if (StringUtils.isBlank(gameTitle.getText()))
			gameTitle.setFocus();
		else if (StringUtils.isBlank(gameYear.getText()))
			gameYear.setFocus();
		else if (StringUtils.isBlank(explanation.getText()))
			explanation.setFocus();
		else
			notes.setFocus();
	}

	private boolean isValid(Text author, Text gameTitle, Text gameYear, Text explanation) {
		GeneralPurposeDialogs.initErrorDialog();
		if (StringUtils.isBlank(author.getText()))
			GeneralPurposeDialogs.addError(text_.get("dialog.confsharing.required.author"), author);
		if (StringUtils.isBlank(gameTitle.getText()))
			GeneralPurposeDialogs.addError(text_.get("dialog.confsharing.required.gametitle"), gameTitle);
		if (StringUtils.isBlank(gameYear.getText()))
			GeneralPurposeDialogs.addError(text_.get("dialog.confsharing.required.gameyear"), gameYear);
		if (StringUtils.isBlank(explanation.getText()))
			GeneralPurposeDialogs.addError(text_.get("dialog.confsharing.required.explanation"), explanation);
		return !GeneralPurposeDialogs.displayErrorDialog(shell_);
	}

	private void fillStringIntoTree(String conf, Tree tree) {
		String[] lines = StringUtils.split(conf, SystemUtils.EOLN);

		TreeItem sectionItem = null;
		for (String s: lines) {
			if (s.startsWith("[")) {
				sectionItem = new TreeItem(tree, SWT.NONE);
				sectionItem.setText(s);
				sectionItem.setChecked(true);
			} else {
				TreeItem node = new TreeItem(sectionItem, SWT.NONE);
				node.setText(s);
				node.setChecked(true);
			}
		}

		for (TreeItem item: tree.getItems())
			item.setExpanded(true);

		tree.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				if (event.detail == SWT.CHECK) {
					TreeItem tItem = (TreeItem)event.item;
					if (tItem.getParentItem() == null) {
						tItem.setGrayed(false);
						for (TreeItem item: tItem.getItems())
							item.setChecked(tItem.getChecked());
					} else {
						TreeItem parent = tItem.getParentItem();
						long checkedCount = Stream.of(parent.getItems()).filter(x -> x.getChecked()).count();
						parent.setChecked(checkedCount > 0);
						parent.setGrayed(checkedCount > 0 && checkedCount < parent.getItemCount());
					}
				}
			}
		});
	}

	private String extractConfFromTree(Tree tree) {
		Configuration conf = new Configuration();
		for (TreeItem sectionItem: tree.getItems()) {
			if (sectionItem.getChecked()) {
				for (TreeItem node: sectionItem.getItems()) {
					if (node.getChecked()) {
						String[] v = StringUtils.split(node.getText(), '=');
						conf.setValue(StringUtils.substring(sectionItem.getText(), 1, -1), v[0], v[1]);
					}
				}
			}
		}
		return conf.toString(null);
	}
}
