/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.dialog;

import java.util.stream.Stream;
import org.dbgl.gui.abstractdialog.BaseDialog;
import org.dbgl.service.ImageService;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Monitor;
import org.eclipse.swt.widgets.Shell;


public class Thumb extends BaseDialog<Object> {

	private static final int DIALOG_WIDTH_EXT = 24;
	private static final int DIALOG_HEIGHT_EXT = 50;

	private String filename_;
	private Image thumbImage_;

	public Thumb(Shell parent, String filename) {
		super(parent, SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		filename_ = filename;
	}

	@Override
	protected String getDialogTitle() {
		return text_.get("dialog.screenshot.title", new Object[] {filename_});
	}

	@Override
	protected boolean prepare() {
		Image orgThumb = ImageService.getImage(display_, filename_);
		ImageData bigThumb = orgThumb.getImageData();
		int width = bigThumb.width;
		int height = bigThumb.height;

		Monitor monitor = getActiveMonitor();
		double factor = Math.min((double)monitor.getClientArea().width / (width + DIALOG_WIDTH_EXT * 2), (double)monitor.getClientArea().height / (height + DIALOG_HEIGHT_EXT * 2));
		thumbImage_ = new Image(display_, bigThumb.scaledTo((int)(width * factor), (int)(height * factor)));
		orgThumb.dispose();
		return true;
	}

	@Override
	protected void onShellCreated() {
		if (thumbImage_ != null) {
			shell_.setSize(thumbImage_.getImageData().width + DIALOG_WIDTH_EXT, thumbImage_.getImageData().height + DIALOG_HEIGHT_EXT);
			shell_.setLayout(new FillLayout());

			Rectangle bounds = getActiveMonitor().getBounds();
			Rectangle rect = shell_.getBounds();
			shell_.setLocation(bounds.x + (bounds.width - rect.width) / 2, bounds.y + (bounds.height - rect.height) / 2);

			createImageButton(shell_, SWT.FLAT, thumbImage_, closeShellAdapter);
		}
	}

	@Override
	protected void onClose() {
		if (thumbImage_ != null && !thumbImage_.isDisposed())
			thumbImage_.dispose();
	}

	private Monitor getActiveMonitor() {
		return Stream.of(display_.getMonitors()).filter(x -> x.getBounds().intersects(getParent().getBounds())).findFirst().orElse(display_.getPrimaryMonitor());
	}
}
