/*
 *  Copyright (C) 2006-2020  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model;

import java.io.IOException;
import java.util.Comparator;
import org.apache.commons.lang3.StringUtils;
import org.dbgl.service.ImageService;
import org.dbgl.util.searchengine.WebSearchEngine;
import org.eclipse.swt.graphics.ImageData;
import org.w3c.dom.Element;


public class WebProfile implements Comparable<WebProfile> {

	private String title_, platform_, year_, url_, developerName_, publisherName_, genre_, notes_;
	private int rank_;
	private String coreGameCoverUrl_;
	private SearchEngineImageInformation[] webImages_;
	private Element xmlElementWithAllImages_;

	public WebProfile() {
	}

	public String getTitle() {
		return title_;
	}

	public void setTitle(String title) {
		title_ = title;
	}

	public String getPlatform() {
		return platform_;
	}

	public void setPlatform(String platform) {
		platform_ = platform;
	}

	public String getYear() {
		return year_;
	}

	public void setYear(String year) {
		year_ = year;
	}

	public String getUrl() {
		return url_;
	}

	public void setUrl(String url) {
		url_ = url;
	}

	public String getScreenshotsUrl() {
		return url_ + "/screenshots";
	}

	public String getCoverArtUrl() {
		return url_ + "/cover-art";
	}

	public String getCoreGameCoverUrl() {
		return coreGameCoverUrl_;
	}

	public String getCoreGameCoverUrlWithoutPathPrefix() {
		int index = StringUtils.ordinalIndexOf(coreGameCoverUrl_, "/", 3);
		if (index > 0)
			return coreGameCoverUrl_.substring(index + 1);
		return coreGameCoverUrl_;
	}

	public void setCoreGameCoverUrl(String coreGameCoverUrl) {
		coreGameCoverUrl_ = coreGameCoverUrl;
	}

	public ImageData getWebImage(int i) throws IOException {
		if (webImages_[i].getData() == null)
			webImages_[i].setData(ImageService.getImageData(WebSearchEngine.getInputStream(webImages_[i].getUrl())));
		return webImages_[i].getData();
	}

	public SearchEngineImageInformation[] getWebImages() {
		return webImages_.clone();
	}

	public void setWebImages(SearchEngineImageInformation[] webImages) {
		webImages_ = webImages.clone();
	}

	public String getDeveloperName() {
		return developerName_;
	}

	public void setDeveloperName(String developerName) {
		developerName_ = developerName;
	}

	public String getPublisherName() {
		return publisherName_;
	}

	public void setPublisherName(String publisherName) {
		publisherName_ = publisherName;
	}

	public String getGenre() {
		return genre_;
	}

	public void setGenre(String genre) {
		genre_ = genre;
	}

	public String getNotes() {
		return notes_;
	}

	public void setNotes(String notes) {
		notes_ = notes;
	}

	public int getRank() {
		return rank_;
	}

	public void setRank(int rank) {
		rank_ = rank;
	}

	public Element getXmlElementWithAllImages() {
		return xmlElementWithAllImages_;
	}

	public void setXmlElementWithAllImages(Element xmlElementWithAllImages) {
		xmlElementWithAllImages_ = xmlElementWithAllImages;
	}

	public int hashCode() {
		return platform_.hashCode() ^ title_.hashCode();
	}

	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		WebProfile otherProfile = (WebProfile)obj;
		return platform_.equalsIgnoreCase(otherProfile.platform_) && title_.equalsIgnoreCase(otherProfile.title_);
	}

	public int compareTo(WebProfile otherProfile) {
		if (otherProfile.platform_.equalsIgnoreCase(platform_)) {
			return title_.compareToIgnoreCase(otherProfile.title_);
		}
		return platform_.compareToIgnoreCase(otherProfile.platform_);
	}

	public static final class byTitle implements Comparator<WebProfile> {
		public int compare(WebProfile prof1, WebProfile prof2) {
			return prof1.title_.compareToIgnoreCase(prof2.title_);
		}
	}

	public static final class byYear implements Comparator<WebProfile> {
		public int compare(WebProfile prof1, WebProfile prof2) {
			return prof1.year_.compareToIgnoreCase(prof2.year_);
		}
	}

	public static final class byPlatform implements Comparator<WebProfile> {
		public int compare(WebProfile prof1, WebProfile prof2) {
			return prof1.platform_.compareToIgnoreCase(prof2.platform_);
		}
	}

	public String toString() {
		return title_ + '@' + platform_;
	}
}
