/*
 *  Copyright (C) 2006-2021  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.dialog;

import java.io.File;
import java.util.stream.Stream;

import org.dbgl.gui.abstractdialog.BaseDialog;
import org.dbgl.gui.controls.Button_;
import org.dbgl.service.ImageService;
import org.dbgl.service.SettingsService;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Shell;


public class Thumb extends BaseDialog<Object> {

	private static final int BUTTON_BORDER_WIDTH = 0;
	private static final int BUTTON_BORDER_HEIGHT = 24;

	private File[] files_;
	private int index_;

	private Rectangle monitorClientArea_;
	private Button imgButton_;
	private Rectangle imgBounds_;
	private Image thumbImage_;

	public Thumb(Shell parent, File[] files, int index) {
		super(parent, SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		files_ = files;
		index_ = index;
	}

	@Override
	protected String getDialogTitle() {
		return text_.get("dialog.screenshot.title", files_[index_].getPath());
	}

	@Override
	protected boolean prepare() {
		monitorClientArea_ = Stream.of(display_.getMonitors()).filter(x -> x.getBounds().intersects(getParent().getBounds())).findFirst().orElse(display_.getPrimaryMonitor()).getClientArea();

		return true;
	}

	@Override
	protected void onShellCreated() {
		shell_.setBounds(monitorClientArea_);
		shell_.setLayout(new GridLayout(3, true));

		imgButton_ = Button_.on(shell_).layoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 3, 1)).text().listen(closeShellAdapter).ctrl();
		imgButton_.addTraverseListener(new TraverseListener() {
			@Override
			public void keyTraversed(TraverseEvent e) {
				if ((e.keyCode == SWT.ARROW_LEFT) || (e.keyCode == SWT.ARROW_RIGHT)) {
					nextPrev(e.keyCode == SWT.ARROW_RIGHT);
					e.doit = false;
				}
			}
		});

		Button_.on(shell_).text().key("button.previousimage").listen(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				nextPrev(false);
			}
		}).build();
		Button_.on(shell_).text().key("button.close").listen(closeShellAdapter).build();
		Button_.on(shell_).text().key("button.nextimage").listen(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				nextPrev(true);
			}
		}).build();
	}

	@Override
	protected void onShellOpened() {
		imgBounds_ = imgButton_.getBounds();

		getThumbImage();
		imgButton_.setImage(thumbImage_);
		adjustShellBounds();
	}

	private void adjustShellBounds() {
		shell_.pack();
		Rectangle shellBounds = shell_.getBounds();
		shell_.setLocation(monitorClientArea_.x + (monitorClientArea_.width - shellBounds.width) / 2, monitorClientArea_.y + (monitorClientArea_.height - shellBounds.height) / 2);
	}

	private void nextPrev(boolean next) {
		if (next) {
			index_++;
			if (index_ >= files_.length)
				index_ = 0;
		} else {
			index_--;
			if (index_ < 0)
				index_ = files_.length - 1;
		}
		getThumbImage();
		imgButton_.setImage(thumbImage_);
		shell_.setText(getDialogTitle());
		adjustShellBounds();
	}

	private void getThumbImage() {
		if (thumbImage_ != null && !thumbImage_.isDisposed())
			thumbImage_.dispose();

		Image orgThumb = ImageService.getImage(display_, files_[index_].getPath());
		ImageData orgThumbData = orgThumb.getImageData();
		int width = orgThumbData.width;
		int height = orgThumbData.height;

		double factor = Math.min(Math.min((double)(imgBounds_.width - BUTTON_BORDER_WIDTH) / width, (double)(imgBounds_.height - BUTTON_BORDER_HEIGHT) / height),
			SettingsService.getInstance().getIntValue("gui", "screenshotsmaxzoompercentage") / 100d);
		if (width * factor > SettingsService.getInstance().getIntValue("gui", "screenshotsmaxwidth"))
			factor = (double)SettingsService.getInstance().getIntValue("gui", "screenshotsmaxwidth") / width;
		if (height * factor > SettingsService.getInstance().getIntValue("gui", "screenshotsmaxheight"))
			factor = (double)SettingsService.getInstance().getIntValue("gui", "screenshotsmaxheight") / height;
		thumbImage_ = ImageService.createScaledImage(orgThumb, width, height, (int)(width * factor), (int)(height * factor), false, null);

		orgThumb.dispose();
	}
}