/*
 *  Copyright (C) 2006-2021  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util.archive;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import org.dbgl.gui.interfaces.ProgressNotifyable;
import org.dbgl.service.TextService;
import SevenZip.HRESULT;
import SevenZip.Archive.IArchiveExtractCallback;
import SevenZip.Archive.IInArchive;
import SevenZip.Archive.SevenZipEntry;


public class SevenzipExtractFilesCallback implements IArchiveExtractCallback {

	private ProgressNotifyable prog_;
	private IInArchive archiveHandler_;
	private Map<Integer, File> dstFileMap_;

	public SevenzipExtractFilesCallback(ProgressNotifyable prog, IInArchive archiveHandler, Map<Integer, File> dstFileMap) {
		prog_ = prog;
		archiveHandler_ = archiveHandler;
		dstFileMap_ = dstFileMap;
	}

	@Override
	public int SetTotal(long size) {
		return HRESULT.S_OK;
	}

	@Override
	public int SetCompleted(long completeValue) {
		prog_.setProgress(completeValue);
		return HRESULT.S_OK;
	}

	@Override
	public int PrepareOperation(int askExtractMode) {
		return HRESULT.S_OK;
	}

	@Override
	public int SetOperationResult(int operationResult) throws java.io.IOException {
		switch (operationResult) {
			case IInArchive.NExtract_NOperationResult_kOK:
				return HRESULT.S_OK;
			case IInArchive.NExtract_NOperationResult_kUnSupportedMethod:
				throw new IOException("Unsupported Method");
			case IInArchive.NExtract_NOperationResult_kCRCError:
				throw new IOException("CRC Failed");
			case IInArchive.NExtract_NOperationResult_kDataError:
				throw new IOException("Data Error");
			default:
				throw new IOException("Unknown Error");
		}
	}

	@Override
	public int GetStream(int index, OutputStream[] outStream, int askExtractMode) throws java.io.IOException {
		outStream[0] = null;

		if (askExtractMode == IInArchive.NExtract_NAskMode_kExtract) {
			SevenZipEntry entry = archiveHandler_.getEntry(index);
			File dstFile = dstFileMap_.get(index);

			if (entry.isDirectory())
				return dstFile.isDirectory() || dstFile.mkdirs() ? HRESULT.S_OK: HRESULT.S_FALSE;

			File dirAbove = dstFile.getParentFile();
			if ((dirAbove != null) && !dirAbove.isDirectory() && !dirAbove.mkdirs())
				return HRESULT.S_FALSE;
			if (dstFile.exists())
				throw new IOException(TextService.getInstance().get("general.error.filetobeextractedexists", new Object[] {dstFile}));
			outStream[0] = new FileOutputStream(dstFile);
		}

		return HRESULT.S_OK;
	}
}
