/*
 *  Copyright (C) 2006-2021  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.controls;

import org.eclipse.nebula.widgets.gallery.AbstractGalleryItemRenderer;
import org.eclipse.nebula.widgets.gallery.GalleryItem;
import org.eclipse.nebula.widgets.gallery.RendererHelper;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;


public class GalleryItemRenderer extends AbstractGalleryItemRenderer {

	private static final int LINE_WIDTH = 2;
	private static final int SELECTION_RADIUS = 8;

	private static final String ELLIPSIS = "...";

	private static final Color foregroundColor_ = Display.getDefault().getSystemColor(SWT.COLOR_LIST_FOREGROUND);
	private static final Color backgroundColor_ = Display.getDefault().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
	private static final Color selectionForegroundColor_ = Display.getDefault().getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT);
	private static final Color selectionBackgroundColor_ = Display.getDefault().getSystemColor(SWT.COLOR_LIST_SELECTION);

	private final String truncPosition_;

	public GalleryItemRenderer(String truncPosition) {
		truncPosition_ = truncPosition;
	}

	@Override
	public void draw(GC gc, GalleryItem item, int index, int x, int y, int width, int height) {
		// Set up the GC
		gc.setFont(getFont(item));

		// Create some room for the label
		int fontHeight = gc.getFontMetrics().getHeight();

		// Checks if background has to be drawn
		boolean drawBackground = selected;
		Color drawBackgroundColor = null;
		if (!drawBackground && item.getBackground(true) != null) {
			drawBackgroundColor = getBackground(item);
			if (!RendererHelper.isColorsEquals(drawBackgroundColor, backgroundColor_))
				drawBackground = true;
		}

		// Draw background (rounded rectangles)
		if (drawBackground) {
			if (selected) {
				gc.setBackground(selectionBackgroundColor_);
				gc.setForeground(selectionBackgroundColor_);
			} else if (drawBackgroundColor != null) {
				gc.setBackground(drawBackgroundColor);
			}

			gc.setLineStyle(SWT.LINE_DOT);
			gc.setLineWidth(LINE_WIDTH);
			gc.drawRoundRectangle(x + (LINE_WIDTH / 2), y + (LINE_WIDTH / 2), width - LINE_WIDTH, height - LINE_WIDTH, SELECTION_RADIUS, SELECTION_RADIUS);
			gc.fillRoundRectangle(x + (LINE_WIDTH / 2), y + height - fontHeight - (LINE_WIDTH / 2) - 1, width - LINE_WIDTH, fontHeight + 1, SELECTION_RADIUS, SELECTION_RADIUS);
		}

		// Draw image
		Image drawImage = item.getImage();
		if (drawImage != null) {
			Rectangle itemImageBounds = drawImage.getBounds();
			int xShift = RendererHelper.getShift(width - (LINE_WIDTH * 2), itemImageBounds.width);
			gc.drawImage(drawImage, x + LINE_WIDTH + xShift, y + LINE_WIDTH);
			drawAllOverlays(gc, item, x, y, new Point(itemImageBounds.width, itemImageBounds.height), xShift, LINE_WIDTH);
		}

		// Set colors for label
		if (selected) {
			gc.setForeground(selectionForegroundColor_);
			gc.setBackground(selectionBackgroundColor_);
		} else {
			// Not selected, use item values or defaults
			gc.setBackground(drawBackgroundColor != null ? drawBackgroundColor: backgroundColor_);
			Color drawForegroundColor = getForeground(item);
			gc.setForeground(drawForegroundColor != null ? drawForegroundColor: foregroundColor_);
		}

		// Create label
		String text;
		if (truncPosition_.equalsIgnoreCase("end"))
			text = createLabelTruncAtEnd(item.getText(), gc, width - SELECTION_RADIUS);
		else // middle
			text = RendererHelper.createLabel(item.getText(), gc, width - SELECTION_RADIUS);

		// Draw centered text
		gc.drawText(text, x + RendererHelper.getShift(width, gc.textExtent(text).x), y + height - fontHeight, true);
	}

	@Override
	public void dispose() {
		// nothing to dispose
	}

	private static String createLabelTruncAtEnd(String text, GC gc, int width) {
		if (text != null) {
			int extent = gc.textExtent(text).x;
			if (extent > width) {
				int w = gc.textExtent(ELLIPSIS).x;
				if (width > w) {
					int l = text.length();
					while (extent > width)
						extent = gc.textExtent(text.substring(0, --l)).x + w;
					return text.substring(0, l) + ELLIPSIS;
				}
			}
		}
		return text;
	}
}
