/* 
 * Created on Jul 23, 2006 by ronald.blankendaal
 * 
 * @file $RCSfile: FileUtils.java,v $
 * @version $Revision: 1.8 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2006/10/08 18:57:34 $ (UTC date of last check in)
 */

package com.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import com.gui.StreamGobbler;
import com.model.Autoexec;
import com.model.Configuration;
import com.model.DosboxVersion;
import com.model.Profile;
import com.model.Settings;


public abstract class FileUtils {

	public static final String CAPTURES_DIR = "captures";
  public static final String TEMPLATES_DIR = "templates";
  public static final String PROFILES_DIR = "profiles";
  public static final String CONF_EXT = ".conf";
  public static final String DOSBOX_CONF = "dosbox" + CONF_EXT;
  public static final String SETUP_CONF = "setup" + CONF_EXT;
  public static final String DFEND_PATH = "C:/Program Files/D-Fend";
  public static final String DFEND_PROFILES = "Profiles.dat";
  
  
  public static String getCwd() {
    try {
      return new File(".").getCanonicalPath();
    } catch(IOException ex) {
      System.err.println("Couldn't get current working directory!!\n");
    }
    return null; // this shouldn't ever fail
  }
  
  public static String getCanonicalPath(String path) {
  	try {
  		return new File(path).getCanonicalPath();
  	} catch (IOException e) {
  		return new File(path).getAbsolutePath();
  	}
  }
  
  public static void createDir(String dir) {
    if (!new File(dir).exists() && !new File(dir).mkdirs())
      System.err.println("Couldn't create " + dir + " directory!!\n");
  }
  
  public static void copyFiles(String srcDir, String dstDir) {
    File[] srcFiles = new File(srcDir).listFiles();
    if (srcFiles != null)
      for (File src: srcFiles)
        if (src.isFile()) FileUtils.copy(src, new File(dstDir + src.getName()));
  }
  
  /** Fast & simple file copy. */
  public static void copy(File source, File dest) {
       FileChannel in = null, out = null;
       try {          
         in = new FileInputStream(source).getChannel();
         out = new FileOutputStream(dest).getChannel();
         in.transferTo(0, in.size(), out);
       } catch (IOException e) {
         System.err.println("Couldn't copy file " + source + " to directory " + dest + "!!\n");
       } finally {
         if (in != null) try {in.close();} catch (IOException ex) {}
         if (out != null) try {out.close();} catch (IOException ex) {}
       }
  }
  
  public static void removeFile(String filename) {
    File file = new File(filename);
    if (! (file.isFile() && file.delete()) )
      System.err.println("Couldn't delete file " + filename);
  }
  
  public static void removeFilesInDirAndDir(String dirname) {
  	File[] files = new File(dirname).listFiles();
  	if (files != null) {
  		for (File file: files)
  			if (file.isDirectory()) {
  				System.err.println("Directory found inside " + dirname + ", aborting removal");
  				return;
  			}
  		for (File file: files)
  			if (!file.delete())
  	      System.err.println("Couldn't delete file " + file);
  	}
  	File dir = new File(dirname);
    if (! (dir.isDirectory() && dir.delete()) )
    	System.err.println("Couldn't delete directory " + dirname);
  }
  
  public static boolean isExistingReadableFile(String filename) {
    File file = new File(filename);
    return file.isFile() && file.canRead();
  }
  
  public static void doRunDosbox(DosboxVersion dbversion) {
	  List<String> parameters = new ArrayList<String>();
	  parameters.add("-conf");
	  parameters.add(DOSBOX_CONF);
		doRunDosbox(dbversion, parameters, false);
  }
  
  public static void doCreateDosboxConf(DosboxVersion dbversion) {
  	List<String> parameters = new ArrayList<String>();
		parameters.add("-c");
		parameters.add("config -writeconf " + FileUtils.DOSBOX_CONF);
		parameters.add("-c");
		parameters.add("exit");
		FileUtils.doRunDosbox(dbversion, parameters, true);
  }
  
  private static void doRunDosbox(DosboxVersion dbversion, List<String> parameters, boolean waitFor) {
  	try {
  		List<String> commandItems = new ArrayList<String>();
  		commandItems.add(dbversion.getExecutable());
  		commandItems.addAll(parameters);
  		executeCommand(commandItems, dbversion.getCanonicalPath(), waitFor);
  	} catch (IOException e) {
  		System.err.println("Dosbox could not be started!!\n");
  	}
  }
  
  public static void doRunProfile(Profile prof, List<DosboxVersion> dbversionsList, boolean setup) {
    try {
      int dbversionIndex = ListUtils.findTIDosboxVersion(dbversionsList, prof.getDbversionId());
      DosboxVersion dbversion = dbversionsList.get(dbversionIndex);
      List<String> confs = new ArrayList<String>();
      confs.add(dbversion.getExecutable());
      if (dbversion.isMultiConfig()) {
        // selected default dosbox config file
      	confs.add("-conf");
        confs.add(DOSBOX_CONF); 
      }
      // add profile-specific settings
      String filename = prof.getId() + CONF_EXT;
      
      if (setup && prof.hasSetupBoolean()) {
	      Configuration conf = new Configuration(PROFILES_DIR + File.separatorChar + prof.getId() + CONF_EXT);
	      Autoexec autoexec = new Autoexec(conf.getValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC));
	      autoexec.setSetup(prof.getSetup());
	      autoexec.setSetupParameters(prof.getSetupParameters());
	      conf.setValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC, autoexec.toString(true));
	      conf.saveToFile(PROFILES_DIR + File.separatorChar + SETUP_CONF);
	      filename = SETUP_CONF;
      }
      
      confs.add("-conf");
      confs.add(FileUtils.getProfileConfig(filename));
      FileUtils.executeCommand(confs, dbversion.getCanonicalPath(), false);
    } catch (IOException e) {
      System.err.println("Profile could not be started!!\n");
    }
  }
  
  private static void executeCommand(List<String> execCommands, String cwd, 
  		boolean waitFor) throws IOException {
  	if (Settings.getInstance().getBooleanValue("dosbox", "hideconsole")) {
  		execCommands.add("-noconsole");
  	}
  	String[] rtCmds = new String[execCommands.size()];
  	for (int i = 0; i < execCommands.size(); i++) {
    	rtCmds[i] = execCommands.get(i);
    	System.out.print(rtCmds[i] + " ");
    }
    System.out.println();
    File dir = (cwd != null)? new File(cwd): null;
  	Process proc = Runtime.getRuntime().exec(rtCmds, null, dir);
    StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "DOSBox stderr");            
    StreamGobbler outputGobbler = new StreamGobbler(proc.getInputStream(), "DOSBox stdout");
    outputGobbler.start();
    errorGobbler.start();
    if (waitFor)
    	try {	proc.waitFor();} catch (InterruptedException e) {}
  }
  
  public static void setCapturesDir(Configuration profileConf, int profileId, String srcCap) {
    String capturesDir = getCwd();
    if (capturesDir != null) {
      capturesDir += File.separatorChar + CAPTURES_DIR + File.separatorChar + profileId;
      profileConf.setValue("dosbox", "captures", capturesDir);
      
      // Create directory and optionally copy files from src to dst
      createDir(capturesDir);
      if (srcCap != null)
        copyFiles(srcCap, CAPTURES_DIR + File.separatorChar + profileId + File.separatorChar);
    }
  }
  
  public static String getProfileConfig(String filename) {
  	String result = getCwd();
  	if (result != null)
  		result += File.separatorChar + PROFILES_DIR + File.separatorChar + filename;
  	return result;
  }
}
