/* 
 * Created on 30-jun-2006 by ronald.blankendaal
 * 
 * @file $RCSfile: Database.java,v $
 * @version $Revision: 1.21 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2006/10/25 19:43:45 $ (UTC date of last check in)
 */

package com.db;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import com.model.Developer;
import com.model.DosboxVersion;
import com.model.Genre;
import com.model.Profile;
import com.model.Publisher;
import com.model.Status;
import com.model.Template;
import com.model.Year;
import com.util.FileUtils;


public class Database {

  private Connection con = null;
  
  private static final String GET_IDENTITY_QUERY = "CALL IDENTITY()";
  
  private static final String GAME_LIST_QUERY = "SELECT GAM.ID, GAM.TITLE, DEV.NAME, PUBL.NAME, GEN.NAME, YR.YEAR, GAM.DBVERSION_ID, GAM.SETUP, GAM.SETUP_PARAMS, GAM.NOTES, GAM.LINK1, GAM.LINK2, GAM.FAVORITE, STAT.STAT, GAM.CONFFILE, GAM.CAPTURES FROM GAMES GAM, DEVELOPERS DEV, PUBLISHERS PUBL, GENRES GEN, PUBLYEARS YR, STATUS STAT WHERE GAM.DEV_ID=DEV.ID AND GAM.PUBL_ID=PUBL.ID AND GAM.GENRE_ID=GEN.ID AND GAM.YEAR_ID=YR.ID AND GAM.STAT_ID=STAT.ID ORDER BY ";
  private static final String[] GAME_LIST_ORDER = {"GAM.TITLE", "GAM.SETUP", "DEV.NAME", "PUBL.NAME", "GEN.NAME", "YR.YEAR", "STAT.STAT", "GAM.FAVORITE", "GAM.ID"};
  private static final String DEVELOPER_LIST_QUERY = "SELECT ID, NAME FROM DEVELOPERS ORDER BY NAME";
  private static final String PUBLISHER_LIST_QUERY = "SELECT ID, NAME FROM PUBLISHERS ORDER BY NAME";
  private static final String GENRE_LIST_QUERY = "SELECT ID, NAME FROM GENRES ORDER BY NAME";
  private static final String PUBLYEAR_LIST_QUERY = "SELECT ID, YEAR FROM PUBLYEARS ORDER BY YEAR";
  private static final String DOSBOXVERSIONS_LIST_QUERY = "SELECT ID, TITLE, PATH, MULTICONF, DEFAULT FROM DOSBOXVERSIONS ORDER BY TITLE";
  private static final String TEMPLATES_LIST_QUERY = "SELECT ID, TITLE, DBVERSION_ID, DEFAULT FROM TEMPLATES ORDER BY TITLE";
  private static final String STATUS_LIST_QUERY = "SELECT ID, STAT FROM STATUS ORDER BY STAT";
  
  private static final String ADD_GAME_QUERY = "INSERT INTO GAMES(TITLE, DEV_ID, PUBL_ID, GENRE_ID, YEAR_ID, DBVERSION_ID, STAT_ID, SETUP, SETUP_PARAMS, NOTES, LINK1, LINK2, FAVORITE, CONFFILE, CAPTURES) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NULL, NULL)";
  private static final String ADD_DEVELOPER_QUERY = "INSERT INTO DEVELOPERS(NAME) VALUES (?)";
  private static final String ADD_PUBLISHER_QUERY = "INSERT INTO PUBLISHERS(NAME) VALUES (?)";
  private static final String ADD_GENRE_QUERY = "INSERT INTO GENRES(NAME) VALUES (?)";
  private static final String ADD_YEAR_QUERY = "INSERT INTO PUBLYEARS(YEAR) VALUES (?)";
  private static final String ADD_DOSBOXVERSION_QUERY = "INSERT INTO DOSBOXVERSIONS(TITLE, PATH, MULTICONF, DEFAULT) VALUES (?, ?, ?, ?)";
  private static final String ADD_TEMPLATE_QUERY = "INSERT INTO TEMPLATES(TITLE, DBVERSION_ID, DEFAULT) VALUES (?, ?, ?)";
  private static final String ADD_STATUS_QUERY = "INSERT INTO STATUS(STAT) VALUES (?)";
  
  private static final String UPDATE_GAME_QUERY = "UPDATE GAMES SET TITLE = ?, DEV_ID = ?, PUBL_ID = ?, GENRE_ID = ?, YEAR_ID = ?, DBVERSION_ID = ?, STAT_ID = ?, SETUP = ?, SETUP_PARAMS = ?, NOTES = ?, LINK1 = ?, LINK2 = ?, FAVORITE = ? WHERE ID = ?";
  private static final String UPDATE_GAME_CONF_QUERY = "UPDATE GAMES SET CONFFILE = ?, CAPTURES = ? WHERE ID = ?";
  private static final String UPDATE_DOSBOXVERSION_NODEFAULT = "UPDATE DOSBOXVERSIONS SET DEFAULT = FALSE";
  private static final String UPDATE_DOSBOXVERSION_QUERY = "UPDATE DOSBOXVERSIONS SET TITLE = ?, PATH = ?, MULTICONF = ?, DEFAULT = ? WHERE ID = ?";
  private static final String UPDATE_TEMPLATE_QUERY = "UPDATE TEMPLATES SET TITLE = ?, DBVERSION_ID = ?, DEFAULT = ? WHERE ID = ?";
  private static final String UPDATE_TEMPLATE_NODEFAULT = "UPDATE TEMPLATES SET DEFAULT = FALSE";
  
  private static final String REMOVE_GAME_QUERY = "DELETE FROM GAMES WHERE ID = ?";
  private static final String REMOVE_DOSBOXVERSION_QUERY = "DELETE FROM DOSBOXVERSIONS WHERE ID = ?";
  private static final String REMOVE_TEMPLATE_QUERY = "DELETE FROM TEMPLATES WHERE ID = ?";

  private static final String TOGGLE_FAVORITE_QUERY = "UPDATE GAMES SET FAVORITE = NOT FAVORITE WHERE ID = ?";
  
  private static final String GET_VERSION = "SELECT MAJORVERSION, MINORVERSION FROM VERSION";
  private static final String UPGRADE_TO_V050_QUERY = 
  	"ALTER TABLE GAMES ADD COLUMN CONFFILE VARCHAR(256);" +
  	"ALTER TABLE GAMES ADD COLUMN CAPTURES VARCHAR(256);" +
  	"CREATE TABLE VERSION(MAJORVERSION INTEGER NOT NULL, MINORVERSION INTEGER NOT NULL);" +
  	"INSERT INTO VERSION VALUES(0, 50);"+
  	"UPDATE GAMES SET" +
  	" CAPTURES = '" + FileUtils.CAPTURES_DIR + "' || GAMES.ID," +
  	" CONFFILE = '" + FileUtils.PROFILES_DIR + "' || GAMES.ID || '" + FileUtils.CONF_EXT + "';";

  
  private Database() {
    init();
  }
  
  private static class DatabaseHolder {
    private static Database instance = new Database();
  } 
  
  public static Database getInstance() {
    return DatabaseHolder.instance;
  }
  
  private int[] getVersion() {
  	int[] result = new int[2];
  	try {
    	Statement stmt = con.createStatement();
    	ResultSet rs = stmt.executeQuery(GET_VERSION);
    	if (rs != null && rs.next()) {
    		result[0] = rs.getInt(1); // major
    		result[1] = rs.getInt(2); // minor
    	}
    	stmt.close();
  	} catch (SQLException e) {
  		// assume version < 0.50 (0.0)
    }
  	return result;
  }
  
  private void upgradeToV050() {
  	System.out.println("upgrading database to v0.50");
  	try {
  		Statement stmt = con.createStatement();
    	stmt.executeQuery(UPGRADE_TO_V050_QUERY);
    	stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'upgrade to database v0.50' failed!!\n");
      e.printStackTrace();
    }
  }
  
  public void upgradeIfNecessary() {
  	int[] version = getVersion(); 
  	if (version[0] <= 0 && version[1] < 50) {
  		upgradeToV050();
  	}
  }
  
  private void init() {
    System.out.println("startup");
    try {
      // Register the JDBC driver for dBase
      Class.forName("org.hsqldb.jdbcDriver");
      con = DriverManager.getConnection("jdbc:hsqldb:" + "./db/database", "sa", "");
    } catch (SQLException e) {
      System.err.println("JDBC Connection could not be established!!\n");
      e.printStackTrace();
      try {
        if (con != null)
        	con.close();
      } catch (SQLException exc) {
        System.err.println("JDBC Connection could not be closed properly!!\n");
        exc.printStackTrace();
      }
      System.exit(1);
    } catch (ClassNotFoundException e) {
      System.err.println("JDBC Driver could not be registered!!\n");
      e.printStackTrace();
      System.exit(1);
    }
  }
  
  public void shutdown() {
    System.out.println("shutdown");
    try {
      Statement st = con.createStatement();
      st.execute("SHUTDOWN");
      con.close();
    } catch (SQLException e) {
      System.err.println("JDBC Connection could not be closed properly!!\n");
      e.printStackTrace();
    }
  }
  
  public List<Profile> readProfilesList(int column, boolean ascending) {
	  List<Profile> profilesList = new ArrayList<Profile>();
	  try {
		  Statement stmt = con.createStatement();
		  String query = GAME_LIST_QUERY + GAME_LIST_ORDER[column] + (ascending? " ASC": " DESC");
		  ResultSet rs = stmt.executeQuery(query);
		  if (rs != null) {
			  while (rs.next()) {
				  profilesList.add(new Profile(rs.getInt(1), rs.getString(2),
						  rs.getString(3), rs.getString(4), rs.getString(5),
						  rs.getString(6), rs.getInt(7), rs.getString(8),
						  rs.getString(9), rs.getString(10), rs.getString(11),
						  rs.getString(12), rs.getBoolean(13), rs.getString(14),
						  rs.getString(15), rs.getString(16)));
			  }
		  }
		  stmt.close();
	  } catch (SQLException e) {
		  System.err.println("JDBC execute query 'read profiles' failed!!\n");
		  e.printStackTrace();
	  }
	  return profilesList;
  }
  
  public List<Developer> readDevelopersList() {
    List<Developer> developersList = new ArrayList<Developer>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(DEVELOPER_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          developersList.add(new Developer(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read developers' failed!!\n");
      e.printStackTrace();
    }
    return developersList;
  }
  
  public List<Publisher> readPublishersList() {
    List<Publisher> publishersList = new ArrayList<Publisher>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(PUBLISHER_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          publishersList.add(new Publisher(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read publishers' failed!!\n");
      e.printStackTrace();
    }
    return publishersList;
  }
  
  public List<Genre> readGenresList() {
    List<Genre> genresList = new ArrayList<Genre>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(GENRE_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          genresList.add(new Genre(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read genres' failed!!\n");
      e.printStackTrace();
    }
    return genresList;
  }
  
  public List<Year> readYearsList() {
    List<Year> yearsList = new ArrayList<Year>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(PUBLYEAR_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          yearsList.add(new Year(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read years' failed!!\n");
      e.printStackTrace();
    }
    return yearsList;
  }
  
  public List<Status> readStatusList() {
    List<Status> statusList = new ArrayList<Status>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(STATUS_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          statusList.add(new Status(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read status' failed!!\n");
      e.printStackTrace();
    }
    return statusList;
  }
  
  public List<DosboxVersion> readDosboxVersionsList() {
    List<DosboxVersion> dosboxVersionsList = new ArrayList<DosboxVersion>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(DOSBOXVERSIONS_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          dosboxVersionsList.add(new DosboxVersion(
          		rs.getInt(1), rs.getString(2), rs.getString(3), rs.getBoolean(4), rs.getBoolean(5)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read dosbox versions' failed!!\n");
      e.printStackTrace();
    }
    return dosboxVersionsList;
  }

  public List<Template> readTemplatesList() {
    List<Template> templatesList = new ArrayList<Template>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(TEMPLATES_LIST_QUERY);
      if (rs != null)
        while (rs.next())
          templatesList.add(new Template(rs.getInt(1), rs.getString(2), rs.getInt(3), rs.getBoolean(4)));
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'read templates' failed!!\n");
      e.printStackTrace();
    }
    return templatesList;
  }

  public void removeProfile(int gameId) {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_GAME_QUERY);
      stmt.setInt(1, gameId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'remove profile' failed!!\n");
      e.printStackTrace();
    }
  }
  
  public void removeDosboxVersion(int dbversionId) {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_DOSBOXVERSION_QUERY);
      stmt.setInt(1, dbversionId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'remove dosbox version' failed!!\n");
      e.printStackTrace();
    }
  }
  
  public void removeTemplate(int templateId) {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_TEMPLATE_QUERY);
      stmt.setInt(1, templateId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'remove template' failed!!\n");
      e.printStackTrace();
    }
  }

  public int addOrEditProfile(String title, String developer, String publisher, String genre, String year,
          String status, String setup, String setupParams, String notes, String link1, String link2, boolean favorite,
          int devId, int publId, int genId, int yrId, int statId, int dbversionId, int profileId) {
    int result = -1;
    try {
      if (devId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_DEVELOPER_QUERY);
        pstmt.setString(1, developer);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          devId = rs.getInt(1);
        stmt.close();
      }
      
      if (publId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_PUBLISHER_QUERY);
        pstmt.setString(1, publisher);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          publId = rs.getInt(1);
        stmt.close();
      }
      
      if (genId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_GENRE_QUERY);
        pstmt.setString(1, genre);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          genId = rs.getInt(1);
        stmt.close();
      }
      
      if (yrId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_YEAR_QUERY);
        pstmt.setString(1, year);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          yrId = rs.getInt(1);
        stmt.close();
      }
      
      if (statId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_STATUS_QUERY);
        pstmt.setString(1, status);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          statId = rs.getInt(1);
        stmt.close();
      }
      
      if (profileId == -1) {
        // create new profile
        PreparedStatement pstmt = con.prepareStatement(ADD_GAME_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, devId);
        pstmt.setInt(3, publId);
        pstmt.setInt(4, genId);
        pstmt.setInt(5, yrId);
        pstmt.setInt(6, dbversionId);
        pstmt.setInt(7, statId);
        pstmt.setString(8, setup);
        pstmt.setString(9, setupParams);
        pstmt.setString(10, notes);
        pstmt.setString(11, link1);
        pstmt.setString(12, link2);
        pstmt.setBoolean(13, favorite);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing profile
        PreparedStatement pstmt = con.prepareStatement(UPDATE_GAME_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, devId);
        pstmt.setInt(3, publId);
        pstmt.setInt(4, genId);
        pstmt.setInt(5, yrId);
        pstmt.setInt(6, dbversionId);
        pstmt.setInt(7, statId);
        pstmt.setString(8, setup);
        pstmt.setString(9, setupParams);
        pstmt.setString(10, notes);
        pstmt.setString(11, link1);
        pstmt.setString(12, link2);
        pstmt.setBoolean(13, favorite);
        pstmt.setInt(14, profileId);
        pstmt.executeUpdate();
        pstmt.close();
        result = profileId;
      }
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'add/edit profile' failed!!\n");
      e.printStackTrace();
    }
    return result;
  }
  
  public void updateProfileConf(String confFile, String captures, int profileId) {
    try {
    	PreparedStatement pstmt = con.prepareStatement(UPDATE_GAME_CONF_QUERY);
    	pstmt.setString(1, confFile);
      pstmt.setString(2, captures);
      pstmt.setInt(3, profileId);
    	pstmt.executeUpdate();
    	pstmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'update profile conf' failed!!\n");
      e.printStackTrace();
    }
  }
  
  public void toggleFavorite(int profileId) {
    try {
    	PreparedStatement pstmt = con.prepareStatement(TOGGLE_FAVORITE_QUERY);
    	pstmt.setInt(1, profileId);
    	pstmt.executeUpdate();
    	pstmt.close();
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'toggle favorite' failed!!\n");
      e.printStackTrace();
    }
  }
  
  private void resetDosboxVersionDefault() throws SQLException {
  	Statement stmt = con.createStatement();
  	stmt.executeQuery(UPDATE_DOSBOXVERSION_NODEFAULT);
  }
  
  private void resetTemplateDefault() throws SQLException {
  	Statement stmt = con.createStatement();
  	stmt.executeQuery(UPDATE_TEMPLATE_NODEFAULT);
  }
  
  public int addOrEditDosboxVersion(String title, String path, boolean multiConfig, boolean defaultVersion, int dbversionId) {
    int result = -1;
    try {
    	if (defaultVersion)
    		resetDosboxVersionDefault();
    	
      if (dbversionId == -1) {
        // create new dosbox version
        PreparedStatement pstmt = con.prepareStatement(ADD_DOSBOXVERSION_QUERY);
        pstmt.setString(1, title);
        pstmt.setString(2, path);
        pstmt.setBoolean(3, multiConfig);
        pstmt.setBoolean(4, defaultVersion);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing dosbox version
        PreparedStatement pstmt = con.prepareStatement(UPDATE_DOSBOXVERSION_QUERY);
        pstmt.setString(1, title);
        pstmt.setString(2, path);
        pstmt.setBoolean(3, multiConfig);
        pstmt.setBoolean(4, defaultVersion);
        pstmt.setInt(5, dbversionId);
        pstmt.executeUpdate();
        pstmt.close();
        result = dbversionId;
      }
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'add/edit dosbox version' failed!!\n");
      e.printStackTrace();
    }
    return result;
  }

  public int addOrEditTemplate(String title, int dbversionId, boolean defaultVersion, int templateId) {
    int result = -1;
    try {
    	if (defaultVersion)
    		resetTemplateDefault();
    	
      if (templateId == -1) {
        // create new template
        PreparedStatement pstmt = con.prepareStatement(ADD_TEMPLATE_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, dbversionId);
        pstmt.setBoolean(3, defaultVersion);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing template
        PreparedStatement pstmt = con.prepareStatement(UPDATE_TEMPLATE_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, dbversionId);
        pstmt.setBoolean(3, defaultVersion);
        pstmt.setInt(4, templateId);
        pstmt.executeUpdate();
        pstmt.close();
        result = templateId;
      }
    } catch (SQLException e) {
      System.err.println("JDBC execute query 'add/edit template' failed!!\n");
      e.printStackTrace();
    }
    return result;
  }
}
