/* 
 * Created on Jul 23, 2006 by ronald.blankendaal
 * 
 * @file $RCSfile: Mount.java,v $
 * @version $Revision: 1.9 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2006/09/12 18:14:15 $ (UTC date of last check in)
 */

package com.model;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import com.util.StringUtils;


public class Mount {

  public enum MountingType { DIR, IMAGE, PHYSFS };
  
  private MountingType mountingType;
  private String mountAs;
  private char driveletter;
  private String label;
  private String path;
  private boolean ioctl;
  private String write;
  
  
  public Mount(MountingType mountingType, String mountAs, String driveletter, String path,
          String label, boolean ioctl, String write) {
    this.mountingType = mountingType;
    this.mountAs = mountAs.toLowerCase();
    this.driveletter = Character.toLowerCase(driveletter.charAt(0));
    this.path = StringUtils.fixDir(path);
    this.label = label;
    this.ioctl = ioctl;
    this.write = StringUtils.fixDir(write);
  }

  private void init() {
    mountingType = MountingType.DIR;
    mountAs = "";
    driveletter = '\0';
    label = "";
    path = "";
    ioctl = false;
    write = "";
  }
  
  public Mount(boolean floppy, String hostFileLocation, String[] existingMounts) {
  	init();
  	if (hostFileLocation.equals(""))
      return;
    
  	List<Character> freeDriveletters = getFreeDriveletters(floppy, existingMounts);
  	if (freeDriveletters.size() < 1) // no free driveletters left, give up
  		return;
  	
    driveletter = freeDriveletters.get(0);
    int lastSlashIndex = hostFileLocation.lastIndexOf(File.separatorChar);
    path = StringUtils.fixDir(hostFileLocation.substring(0, lastSlashIndex + 1));
  }
  
  public Mount(String mount) {
    init();
    if (mount.equals(""))
      return;
    
    int spaceIndex = mount.indexOf(' ');
    driveletter = Character.toLowerCase(mount.charAt(spaceIndex + 1));
    
    int firstQuoteIndex = mount.indexOf('"');
    int secondQuoteIndex = mount.lastIndexOf('"');
    if (firstQuoteIndex == -1 && secondQuoteIndex == -1) {
    	// not using quotes around path
    	firstQuoteIndex = mount.indexOf(' ', spaceIndex + 1);
    	secondQuoteIndex = mount.indexOf(' ', firstQuoteIndex + 1);
    	if (secondQuoteIndex == -1) secondQuoteIndex = mount.length();
    }
    path = mount.substring(firstQuoteIndex + 1, secondQuoteIndex);
    
    if (mount.toLowerCase().startsWith("mount ")) {
      int typeIndex = mount.indexOf("-t ");
      if (typeIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", typeIndex + 3);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        mountAs = mount.substring(typeIndex + 3, spaceIndex2);
      }
      int labelIndex = mount.indexOf("-label ");
      if (labelIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", labelIndex + 7);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        label = mount.substring(labelIndex + 7, spaceIndex2);
      }
      if (mount.indexOf("-ioctl") != -1) {
        ioctl = true;
      }
      
      // now, lets see if its a physfs mount (zipfile)
      if (path.toLowerCase().endsWith(".zip:\\")) {
        mountingType = MountingType.PHYSFS;
        int colonIndex1 = path.indexOf(":");
        if (colonIndex1 == 1)
          colonIndex1 = path.indexOf(":", colonIndex1 + 1);
        int colonIndex2 = path.lastIndexOf(":");
        if (colonIndex1 != colonIndex2) {
          write = StringUtils.fixDir(path.substring(0, colonIndex1));
          path = path.substring(colonIndex1 + 1, colonIndex2);
        } else {
          path = path.substring(0, colonIndex2);
        }
      }
    } else if (mount.toLowerCase().startsWith("imgmount ")) {
      mountingType = MountingType.IMAGE;
      mountAs = "iso";
    }
    
    if (mountingType != MountingType.PHYSFS) {
      path = StringUtils.fixDir(path);
    }
  }
  
  public static List<Character> getFreeDriveletters(boolean floppy, String[] existingMounts) {
  	Mount[] mnts = new Mount[existingMounts.length];
  	for (int i = 0; i < mnts.length; i++)
  		mnts[i] = new Mount(existingMounts[i]);
  	List<Character> freeDriveletters = new ArrayList<Character>();
  	char start = floppy? 'a': 'c';
  	for (char i = start; i < 'z'; i++)
    	freeDriveletters.add(i);
  	if (!floppy) {
  		freeDriveletters.add('a');
  		freeDriveletters.add('b');
  	}
  	for (int i = 0; i < mnts.length; i++)
  		freeDriveletters.remove(new Character(mnts[i].getDriveletter()));
  	return freeDriveletters;
  }

  public String toString() {
    StringBuffer result = new StringBuffer();
    switch (mountingType) {
      case DIR: 
        result.append("mount " + driveletter + " \"" + path + File.separatorChar + "\"");
        if (ioctl) result.append(" -ioctl");
        break;
      case PHYSFS:
        result.append("mount " + driveletter + " \"");
        if (!write.equals("")) result.append(write + "\\:");
        result.append(path + ":\\\"");
        break;
      case IMAGE:
        result.append("imgmount " + driveletter + " \"" + path + "\"");
        break;
    }
    if (!label.equals("")) result.append(" -label " + label);
    if (!mountAs.equals("")) result.append(" -t " + mountAs);
    return result.toString();
  }

  public char getDriveletter() {
    return driveletter;
  }

  public String getDriveletterString() {
    return String.valueOf(driveletter);
  }

  public boolean isIoctl() {
    return ioctl;
  }

  public String getLabel() {
    return label;
  }

  public String getPath() {
    return path;
  }

  public MountingType getMountingType() {
    return mountingType;
  }
  
  public String getMountAs() {
    return mountAs;
  }
  
  public String getWrite() {
    return write;
  }
}
