/* 
 * Created on 30-jun-2006 by ronald.blankendaal
 * 
 * @file $RCSfile: Database.java,v $
 * @version $Revision: 1.26 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2007/01/10 11:47:06 $ (UTC date of last check in)
 */

package com.db;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import com.model.Developer;
import com.model.DosboxVersion;
import com.model.Genre;
import com.model.Profile;
import com.model.Publisher;
import com.model.Status;
import com.model.Template;
import com.model.Year;
import com.util.FileUtils;
import com.gui.GeneralPurposeDialogs;
import org.eclipse.swt.widgets.Shell;


public class Database {

  private Connection con = null;
  
  private static final String GET_IDENTITY_QUERY = "CALL IDENTITY()";
  
  private static final String GAME_LIST_QUERY = "SELECT GAM.ID, GAM.TITLE, DEV.NAME, PUBL.NAME, GEN.NAME, YR.YEAR, GAM.DBVERSION_ID, GAM.SETUP, GAM.SETUP_PARAMS, GAM.NOTES, GAM.LINK1, GAM.LINK2, GAM.FAVORITE, STAT.STAT, GAM.CONFFILE, GAM.CAPTURES FROM GAMES GAM, DEVELOPERS DEV, PUBLISHERS PUBL, GENRES GEN, PUBLYEARS YR, STATUS STAT WHERE GAM.DEV_ID=DEV.ID AND GAM.PUBL_ID=PUBL.ID AND GAM.GENRE_ID=GEN.ID AND GAM.YEAR_ID=YR.ID AND GAM.STAT_ID=STAT.ID ORDER BY ";
  private static final String GAME_QUERY = "SELECT TITLE, DEV_ID, PUBL_ID, GENRE_ID, YEAR_ID, DBVERSION_ID, STAT_ID, SETUP, SETUP_PARAMS, NOTES, LINK1, LINK2, FAVORITE, CONFFILE, CAPTURES FROM GAMES WHERE ID = ?";
  
  private static final String DEVELOPER_LIST_QUERY = "SELECT ID, NAME FROM DEVELOPERS ORDER BY NAME";
  private static final String PUBLISHER_LIST_QUERY = "SELECT ID, NAME FROM PUBLISHERS ORDER BY NAME";
  private static final String GENRE_LIST_QUERY = "SELECT ID, NAME FROM GENRES ORDER BY NAME";
  private static final String PUBLYEAR_LIST_QUERY = "SELECT ID, YEAR FROM PUBLYEARS ORDER BY YEAR";
  private static final String DOSBOXVERSIONS_LIST_QUERY = "SELECT ID, TITLE, PATH, MULTICONF, DEFAULT, PARAMETERS FROM DOSBOXVERSIONS ORDER BY TITLE";
  private static final String TEMPLATES_LIST_QUERY = "SELECT ID, TITLE, DBVERSION_ID, DEFAULT FROM TEMPLATES ORDER BY TITLE";
  private static final String STATUS_LIST_QUERY = "SELECT ID, STAT FROM STATUS ORDER BY STAT";
  
  private static final String ADD_GAME_QUERY = "INSERT INTO GAMES(TITLE, DEV_ID, PUBL_ID, GENRE_ID, YEAR_ID, DBVERSION_ID, STAT_ID, SETUP, SETUP_PARAMS, NOTES, LINK1, LINK2, FAVORITE, CONFFILE, CAPTURES) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NULL, NULL)";
  private static final String ADD_DEVELOPER_QUERY = "INSERT INTO DEVELOPERS(NAME) VALUES (?)";
  private static final String ADD_PUBLISHER_QUERY = "INSERT INTO PUBLISHERS(NAME) VALUES (?)";
  private static final String ADD_GENRE_QUERY = "INSERT INTO GENRES(NAME) VALUES (?)";
  private static final String ADD_YEAR_QUERY = "INSERT INTO PUBLYEARS(YEAR) VALUES (?)";
  private static final String ADD_DOSBOXVERSION_QUERY = "INSERT INTO DOSBOXVERSIONS(TITLE, PATH, MULTICONF, DEFAULT, PARAMETERS) VALUES (?, ?, ?, ?, ?)";
  private static final String ADD_TEMPLATE_QUERY = "INSERT INTO TEMPLATES(TITLE, DBVERSION_ID, DEFAULT) VALUES (?, ?, ?)";
  private static final String ADD_STATUS_QUERY = "INSERT INTO STATUS(STAT) VALUES (?)";
  
  private static final String UPDATE_GAME_QUERY = "UPDATE GAMES SET TITLE = ?, DEV_ID = ?, PUBL_ID = ?, GENRE_ID = ?, YEAR_ID = ?, DBVERSION_ID = ?, STAT_ID = ?, SETUP = ?, SETUP_PARAMS = ?, NOTES = ?, LINK1 = ?, LINK2 = ?, FAVORITE = ? WHERE ID = ?";
  private static final String UPDATE_GAME_CONF_QUERY = "UPDATE GAMES SET CONFFILE = ?, CAPTURES = ? WHERE ID = ?";
  private static final String UPDATE_DOSBOXVERSION_NODEFAULT = "UPDATE DOSBOXVERSIONS SET DEFAULT = FALSE";
  private static final String UPDATE_DOSBOXVERSION_QUERY = "UPDATE DOSBOXVERSIONS SET TITLE = ?, PATH = ?, MULTICONF = ?, DEFAULT = ?, PARAMETERS = ? WHERE ID = ?";
  private static final String UPDATE_TEMPLATE_QUERY = "UPDATE TEMPLATES SET TITLE = ?, DBVERSION_ID = ?, DEFAULT = ? WHERE ID = ?";
  private static final String UPDATE_TEMPLATE_NODEFAULT = "UPDATE TEMPLATES SET DEFAULT = FALSE";
  
  private static final String REMOVE_GAME_QUERY = "DELETE FROM GAMES WHERE ID = ?";
  private static final String REMOVE_DOSBOXVERSION_QUERY = "DELETE FROM DOSBOXVERSIONS WHERE ID = ?";
  private static final String REMOVE_TEMPLATE_QUERY = "DELETE FROM TEMPLATES WHERE ID = ?";

  private static final String TOGGLE_FAVORITE_QUERY = "UPDATE GAMES SET FAVORITE = NOT FAVORITE WHERE ID = ?";
  
  private static final String GET_VERSION = "SELECT MAJORVERSION, MINORVERSION FROM VERSION";
  private static final String UPGRADE_TO_V050_QUERY = 
  	"ALTER TABLE GAMES ADD COLUMN CONFFILE VARCHAR(256);" +
  	"ALTER TABLE GAMES ADD COLUMN CAPTURES VARCHAR(256);" +
  	"CREATE TABLE VERSION(MAJORVERSION INTEGER NOT NULL, MINORVERSION INTEGER NOT NULL);" +
  	"INSERT INTO VERSION VALUES(0, 50);"+
  	"UPDATE GAMES SET" +
  	" CAPTURES = '" + FileUtils.CAPTURES_DIR + "' || GAMES.ID," +
  	" CONFFILE = '" + FileUtils.PROFILES_DIR + "' || GAMES.ID || '" + FileUtils.CONF_EXT + "';";
  private static final String UPGRADE_TO_V051_QUERY = 
  	"ALTER TABLE DOSBOXVERSIONS ADD COLUMN PARAMETERS VARCHAR(256) DEFAULT '';" +
  	"UPDATE VERSION SET MINORVERSION = 51;";

  
  private Database() {
    init();
  }
  
  private static class DatabaseHolder {
    private static Database instance = new Database();
  } 
  
  public static Database getInstance() {
    return DatabaseHolder.instance;
  }
  
  private int[] getVersion() {
  	int[] result = new int[2];
  	try {
    	Statement stmt = con.createStatement();
    	ResultSet rs = stmt.executeQuery(GET_VERSION);
    	if (rs != null && rs.next()) {
    		result[0] = rs.getInt(1); // major
    		result[1] = rs.getInt(2); // minor
    	}
    	stmt.close();
  	} catch (SQLException e) {
  		// assume version < 0.50 (0.0)
    }
  	return result;
  }
  
  private void upgradeToV050() throws SQLException {
  	System.out.println("upgrading database to v0.50");
  	try {
  		Statement stmt = con.createStatement();
    	stmt.executeQuery(UPGRADE_TO_V050_QUERY);
    	stmt.close();
    } catch (SQLException e) {
      e.printStackTrace();
    	throw new SQLException("Could not upgrade the database to v0.50");
    }
  }
  
  private void upgradeToV051() throws SQLException {
  	System.out.println("upgrading database to v0.51");
  	try {
  		Statement stmt = con.createStatement();
    	stmt.executeQuery(UPGRADE_TO_V051_QUERY);
    	stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Could not upgrade the database to v0.51");
    }
  }
  
  private void upgradeIfNecessary() throws SQLException {
  	int[] version = getVersion(); 
  	if (version[0] <= 0 && version[1] < 50) {
  		upgradeToV050();
  	}
  	if (version[0] <= 0 && version[1] < 51) {
  		upgradeToV051();
  	}
  }
  
  private void init() {
    System.out.println("startup");
    FileUtils.renameDatabaseIfNecessary();
    try {
      // Register the JDBC driver for dBase
      Class.forName("org.hsqldb.jdbcDriver");
      con = DriverManager.getConnection("jdbc:hsqldb:" + FileUtils.DBFILE, "sa", "");
      upgradeIfNecessary();
    } catch (SQLException e) {
      e.printStackTrace();
      try {
        if (con != null)
        	con.close();
      } catch (SQLException exc) {
        System.err.println("JDBC Connection could not be closed properly!!\n");
        exc.printStackTrace();
      }
      Shell shell = new Shell();
      GeneralPurposeDialogs.fatalMessage(shell, "There was a problem initializing the database connection!\n\nTechnical information:\n" + e.toString());
      System.exit(1);
    } catch (ClassNotFoundException e) {
      e.printStackTrace();
      Shell shell = new Shell();
      GeneralPurposeDialogs.fatalMessage(shell, "There was a problem registering the database JDBC driver!\n\nTechnical information:\n" + e.toString());
      System.exit(1);
    }
  }
  
  public void shutdown() throws SQLException {
    System.out.println("shutdown");
    try {
      Statement st = con.createStatement();
      st.execute("SHUTDOWN");
      con.close();
    } catch (SQLException e) {
      e.printStackTrace();
    	throw new SQLException("Database connection could not be shutdown properly");
    }
  }
  
  public List<Profile> readProfilesList(String orderingClause) throws SQLException {
	  List<Profile> profilesList = new ArrayList<Profile>();
	  try {
		  Statement stmt = con.createStatement();
		  ResultSet rs = stmt.executeQuery(GAME_LIST_QUERY + orderingClause);
		  if (rs != null) {
			  while (rs.next()) {
				  profilesList.add(new Profile(rs.getInt(1), rs.getString(2),
						  rs.getString(3), rs.getString(4), rs.getString(5),
						  rs.getString(6), rs.getInt(7), rs.getString(8),
						  rs.getString(9), rs.getString(10), rs.getString(11),
						  rs.getString(12), rs.getBoolean(13), rs.getString(14),
						  rs.getString(15), rs.getString(16)));
			  }
		  }
		  stmt.close();
	  } catch (SQLException e) {
	  	e.printStackTrace();
    	throw new SQLException("Database query 'read profiles' failed");
	  }
	  return profilesList;
  }
  
  public List<Developer> readDevelopersList() throws SQLException {
    List<Developer> developersList = new ArrayList<Developer>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(DEVELOPER_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          developersList.add(new Developer(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read developers' failed");
    }
    return developersList;
  }
  
  public List<Publisher> readPublishersList() throws SQLException {
    List<Publisher> publishersList = new ArrayList<Publisher>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(PUBLISHER_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          publishersList.add(new Publisher(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read publishers' failed");
    }
    return publishersList;
  }
  
  public List<Genre> readGenresList() throws SQLException {
    List<Genre> genresList = new ArrayList<Genre>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(GENRE_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          genresList.add(new Genre(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read genres' failed");
    }
    return genresList;
  }
  
  public List<Year> readYearsList() throws SQLException {
    List<Year> yearsList = new ArrayList<Year>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(PUBLYEAR_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          yearsList.add(new Year(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read years' failed");
    }
    return yearsList;
  }
  
  public List<Status> readStatusList() throws SQLException {
    List<Status> statusList = new ArrayList<Status>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(STATUS_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          statusList.add(new Status(rs.getInt(1), rs.getString(2)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read status' failed");
    }
    return statusList;
  }
  
  public List<DosboxVersion> readDosboxVersionsList() throws SQLException {
    List<DosboxVersion> dosboxVersionsList = new ArrayList<DosboxVersion>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(DOSBOXVERSIONS_LIST_QUERY);
      if (rs != null) {
        while (rs.next()) {
          dosboxVersionsList.add(new DosboxVersion(
          		rs.getInt(1), rs.getString(2), rs.getString(3), rs.getBoolean(4), rs.getBoolean(5),
          		rs.getString(6)));
        }
      }
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read dosboxversions' failed");
    }
    return dosboxVersionsList;
  }

  public List<Template> readTemplatesList() throws SQLException {
    List<Template> templatesList = new ArrayList<Template>();
    try {
      Statement stmt = con.createStatement();
      ResultSet rs = stmt.executeQuery(TEMPLATES_LIST_QUERY);
      if (rs != null)
        while (rs.next())
          templatesList.add(new Template(rs.getInt(1), rs.getString(2), rs.getInt(3), rs.getBoolean(4)));
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'read templates' failed");
    }
    return templatesList;
  }

  public void removeProfile(int gameId) throws SQLException {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_GAME_QUERY);
      stmt.setInt(1, gameId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'remove profile' failed");
    }
  }
  
  public void removeDosboxVersion(int dbversionId) throws SQLException {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_DOSBOXVERSION_QUERY);
      stmt.setInt(1, dbversionId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'remove dosboxversion' failed");
    }
  }
  
  public void removeTemplate(int templateId) throws SQLException {
    try {
      PreparedStatement stmt = con.prepareStatement(REMOVE_TEMPLATE_QUERY);
      stmt.setInt(1, templateId);
      stmt.executeUpdate();
      stmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'remove template' failed");
    }
  }

  public int addOrEditProfile(String title, String developer, String publisher, String genre, String year,
          String status, String setup, String setupParams, String notes, String link1, String link2, boolean favorite,
          int devId, int publId, int genId, int yrId, int statId, int dbversionId, int profileId) throws SQLException {
    int result = -1;
    try {
      if (devId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_DEVELOPER_QUERY);
        pstmt.setString(1, developer);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          devId = rs.getInt(1);
        stmt.close();
      }
      
      if (publId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_PUBLISHER_QUERY);
        pstmt.setString(1, publisher);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          publId = rs.getInt(1);
        stmt.close();
      }
      
      if (genId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_GENRE_QUERY);
        pstmt.setString(1, genre);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          genId = rs.getInt(1);
        stmt.close();
      }
      
      if (yrId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_YEAR_QUERY);
        pstmt.setString(1, year);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          yrId = rs.getInt(1);
        stmt.close();
      }
      
      if (statId == -1) {
        PreparedStatement pstmt = con.prepareStatement(ADD_STATUS_QUERY);
        pstmt.setString(1, status);
        pstmt.executeUpdate();
        pstmt.close();
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          statId = rs.getInt(1);
        stmt.close();
      }
      
      if (profileId == -1) {
        // create new profile
        PreparedStatement pstmt = con.prepareStatement(ADD_GAME_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, devId);
        pstmt.setInt(3, publId);
        pstmt.setInt(4, genId);
        pstmt.setInt(5, yrId);
        pstmt.setInt(6, dbversionId);
        pstmt.setInt(7, statId);
        pstmt.setString(8, setup);
        pstmt.setString(9, setupParams);
        pstmt.setString(10, notes);
        pstmt.setString(11, link1);
        pstmt.setString(12, link2);
        pstmt.setBoolean(13, favorite);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing profile
        PreparedStatement pstmt = con.prepareStatement(UPDATE_GAME_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, devId);
        pstmt.setInt(3, publId);
        pstmt.setInt(4, genId);
        pstmt.setInt(5, yrId);
        pstmt.setInt(6, dbversionId);
        pstmt.setInt(7, statId);
        pstmt.setString(8, setup);
        pstmt.setString(9, setupParams);
        pstmt.setString(10, notes);
        pstmt.setString(11, link1);
        pstmt.setString(12, link2);
        pstmt.setBoolean(13, favorite);
        pstmt.setInt(14, profileId);
        pstmt.executeUpdate();
        pstmt.close();
        result = profileId;
      }
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'add/edit profile' failed");
    }
    return result;
  }
  
  public void updateProfileConf(String confFile, String captures, int profileId) throws SQLException {
    try {
    	PreparedStatement pstmt = con.prepareStatement(UPDATE_GAME_CONF_QUERY);
    	pstmt.setString(1, confFile);
      pstmt.setString(2, captures);
      pstmt.setInt(3, profileId);
    	pstmt.executeUpdate();
    	pstmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'update profile conf' failed");
    }
  }
  
  public void toggleFavorite(int profileId) throws SQLException {
    try {
    	PreparedStatement pstmt = con.prepareStatement(TOGGLE_FAVORITE_QUERY);
    	pstmt.setInt(1, profileId);
    	pstmt.executeUpdate();
    	pstmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'toggle favorite' failed");
    }
  }
  
  public int duplicateProfile(int profileId) throws SQLException {
  	int result = -1;
  	try {
    	PreparedStatement pstmt = con.prepareStatement(GAME_QUERY);
    	pstmt.setInt(1, profileId);
    	ResultSet rs = pstmt.executeQuery();
    	if (rs != null && rs.next()) {
    		// create duplicate profile
        PreparedStatement pstmt2 = con.prepareStatement(ADD_GAME_QUERY);
        pstmt2.setString(1, rs.getString(1));
        pstmt2.setInt(2, rs.getInt(2));
        pstmt2.setInt(3, rs.getInt(3));
        pstmt2.setInt(4, rs.getInt(4));
        pstmt2.setInt(5, rs.getInt(5));
        pstmt2.setInt(6, rs.getInt(6));
        pstmt2.setInt(7, rs.getInt(7));
        pstmt2.setString(8, rs.getString(8));
        pstmt2.setString(9, rs.getString(9));
        pstmt2.setString(10, rs.getString(10));
        pstmt2.setString(11, rs.getString(11));
        pstmt2.setString(12, rs.getString(12));
        pstmt2.setBoolean(13, rs.getBoolean(13));
        pstmt2.executeUpdate();
        pstmt2.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs2 = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs2 != null && rs2.next())
          result = rs2.getInt(1);
        stmt.close();
    	}
    	pstmt.close();
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'duplicate profile' failed");
    }
    return result;
  }
  
  private void resetDosboxVersionDefault() throws SQLException {
  	Statement stmt = con.createStatement();
  	stmt.executeQuery(UPDATE_DOSBOXVERSION_NODEFAULT);
  }
  
  private void resetTemplateDefault() throws SQLException {
  	Statement stmt = con.createStatement();
  	stmt.executeQuery(UPDATE_TEMPLATE_NODEFAULT);
  }
  
  public int addOrEditDosboxVersion(String title, String path, boolean multiConfig,
  		boolean defaultVersion, String parameters, int dbversionId) throws SQLException {
    int result = -1;
    try {
    	if (defaultVersion)
    		resetDosboxVersionDefault();
    	
      if (dbversionId == -1) {
        // create new dosbox version
        PreparedStatement pstmt = con.prepareStatement(ADD_DOSBOXVERSION_QUERY);
        pstmt.setString(1, title);
        pstmt.setString(2, path);
        pstmt.setBoolean(3, multiConfig);
        pstmt.setBoolean(4, defaultVersion);
        pstmt.setString(5, parameters);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing dosbox version
        PreparedStatement pstmt = con.prepareStatement(UPDATE_DOSBOXVERSION_QUERY);
        pstmt.setString(1, title);
        pstmt.setString(2, path);
        pstmt.setBoolean(3, multiConfig);
        pstmt.setBoolean(4, defaultVersion);
        pstmt.setString(5, parameters);
        pstmt.setInt(6, dbversionId);
        pstmt.executeUpdate();
        pstmt.close();
        result = dbversionId;
      }
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'add/edit dosboxversion' failed");
    }
    return result;
  }

  public int addOrEditTemplate(String title, int dbversionId, boolean defaultVersion, int templateId) throws SQLException {
    int result = -1;
    try {
    	if (defaultVersion)
    		resetTemplateDefault();
    	
      if (templateId == -1) {
        // create new template
        PreparedStatement pstmt = con.prepareStatement(ADD_TEMPLATE_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, dbversionId);
        pstmt.setBoolean(3, defaultVersion);
        pstmt.executeUpdate();
        pstmt.close();
        
        Statement stmt = con.createStatement();
        ResultSet rs = stmt.executeQuery(GET_IDENTITY_QUERY);
        if (rs != null && rs.next())
          result = rs.getInt(1);
        stmt.close();
      } else {
        // update existing template
        PreparedStatement pstmt = con.prepareStatement(UPDATE_TEMPLATE_QUERY);
        pstmt.setString(1, title);
        pstmt.setInt(2, dbversionId);
        pstmt.setBoolean(3, defaultVersion);
        pstmt.setInt(4, templateId);
        pstmt.executeUpdate();
        pstmt.close();
        result = templateId;
      }
    } catch (SQLException e) {
    	e.printStackTrace();
    	throw new SQLException("Database query 'add/edit template' failed");
    }
    return result;
  }
}
