/* 
 * Created on Jul 23, 2006 by ronald.blankendaal
 * 
 * @file $RCSfile: Mount.java,v $
 * @version $Revision: 1.11 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2007/01/22 20:52:27 $ (UTC date of last check in)
 */

package com.model;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import com.util.FileUtils;
import com.util.StringUtils;


public class Mount {

  public enum MountingType { DIR, IMAGE, PHYSFS };
  
  private MountingType mountingType;
  private String mountAs;
  private char driveletter;
  private String label;
  private String path;
  private boolean ioctl;
  private String write;
  private String freesize;
  
  
  public Mount(MountingType mountingType, String mountAs, String driveletter, String path,
          String label, boolean ioctl, String write, String freesize) {
    this.mountingType = mountingType;
    this.mountAs = mountAs.toLowerCase();
    this.driveletter = Character.toUpperCase(driveletter.charAt(0));
    while (path.endsWith("\n"))
    	path = path.substring(0, path.length() - 1);
    this.path = StringUtils.fixDir(path);
    this.label = label;
    this.ioctl = ioctl;
    this.write = StringUtils.fixDir(write);
    this.freesize = freesize;
  }

  private void init() {
    mountingType = MountingType.DIR;
    mountAs = "";
    driveletter = '\0';
    label = "";
    path = "";
    ioctl = false;
    write = "";
    freesize = "";
  }
  
  private void initForPhysFS(String physFSPath) {
    mountingType = MountingType.PHYSFS;
    int colonIndex1 = physFSPath.indexOf(":");
    if (colonIndex1 == 1)
      colonIndex1 = physFSPath.indexOf(":", colonIndex1 + 1);
    int colonIndex2 = physFSPath.lastIndexOf(":");
    if (colonIndex1 != colonIndex2) {
      write = StringUtils.fixDir(physFSPath.substring(0, colonIndex1));
      path = physFSPath.substring(colonIndex1 + 1, colonIndex2);
    } else {
      path = physFSPath.substring(0, colonIndex2);
    }
  }
  
  public Mount(boolean floppy, String hostFileLocation, String[] existingMounts) {
  	init();
  	if (hostFileLocation.equals(""))
      return;
    
  	List<Character> freeDriveletters = getFreeDriveletters(floppy, existingMounts);
  	if (freeDriveletters.size() < 1) // no free driveletters left, give up
  		return;
  	
    driveletter = freeDriveletters.get(0);
    
    if (FileUtils.containsPhysFS(hostFileLocation)) {
    	initForPhysFS(hostFileLocation);
    } else {
    	int lastSlashIndex = hostFileLocation.lastIndexOf(File.separatorChar);
    	path = StringUtils.fixDir(hostFileLocation.substring(0, lastSlashIndex + 1));
    }
  }
  
  public Mount(String mount) {
    init();
    if (mount.equals(""))
      return;
    
    int spaceIndex = mount.indexOf(' ');
    driveletter = Character.toUpperCase(mount.charAt(spaceIndex + 1));
    
    int firstQuoteIndex = mount.indexOf('"');
    int secondQuoteIndex = mount.lastIndexOf('"');
    if (firstQuoteIndex == -1 && secondQuoteIndex == -1) {
    	// not using quotes around path
    	firstQuoteIndex = mount.indexOf(' ', spaceIndex + 1);
    	secondQuoteIndex = mount.indexOf(' ', firstQuoteIndex + 1);
    	if (secondQuoteIndex == -1) secondQuoteIndex = mount.length();
    }
    path = mount.substring(firstQuoteIndex + 1, secondQuoteIndex);
    
    if (mount.toLowerCase().startsWith("mount ")) {
      int typeIndex = mount.indexOf("-t ");
      if (typeIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", typeIndex + 3);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        mountAs = mount.substring(typeIndex + 3, spaceIndex2);
      }
      int labelIndex = mount.indexOf("-label ");
      if (labelIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", labelIndex + 7);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        label = mount.substring(labelIndex + 7, spaceIndex2);
      }
      if (mount.indexOf("-ioctl") != -1) {
        ioctl = true;
      }
      int freesizeIndex = mount.indexOf("-freesize ");
      if (freesizeIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", freesizeIndex + 10);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        freesize = mount.substring(freesizeIndex + 10, spaceIndex2);
      }
      
      if (FileUtils.isPhysFS(path))
      	initForPhysFS(path);
      
    } else if (mount.toLowerCase().startsWith("imgmount ")) {
      mountingType = MountingType.IMAGE;
      path = path.replaceAll("\" \"", "\n");
      mountAs = "iso";
    }
    
    if (mountingType == MountingType.DIR)
      path = StringUtils.fixDir(path);
  }
  
  public static List<Character> getFreeDriveletters(boolean floppy, String[] existingMounts) {
  	Mount[] mnts = new Mount[existingMounts.length];
  	for (int i = 0; i < mnts.length; i++)
  		mnts[i] = new Mount(existingMounts[i]);
  	List<Character> freeDriveletters = new ArrayList<Character>();
  	char start = floppy? 'A': 'C';
  	for (char i = start; i < 'Z'; i++)
    	freeDriveletters.add(i);
  	if (!floppy) {
  		freeDriveletters.add('A');
  		freeDriveletters.add('B');
  	}
  	for (int i = 0; i < mnts.length; i++)
  		freeDriveletters.remove(new Character(mnts[i].getDriveletter()));
  	return freeDriveletters;
  }

  public String toString() {
    StringBuffer result = new StringBuffer();
    switch (mountingType) {
      case DIR: 
        result.append("mount " + driveletter + " \"" + path + File.separatorChar + "\"");
        if (ioctl) result.append(" -ioctl");
        break;
      case PHYSFS:
        result.append("mount " + driveletter + " \"");
        if (!write.equals("")) result.append(write + "\\:");
        result.append(path + ":\\\"");
        break;
      case IMAGE:
      	result.append("imgmount " + driveletter + " \"" + path.replaceAll("\n", "\" \"") + "\"");
        break;
    }
    if (!label.equals("")) result.append(" -label " + label);
    if (!mountAs.equals("")) result.append(" -t " + mountAs);
    if (!freesize.equals("")) result.append(" -freesize " + freesize);
    return result.toString();
  }

  public char getDriveletter() {
    return driveletter;
  }

  public String getDriveletterString() {
    return String.valueOf(driveletter);
  }

  public boolean isIoctl() {
    return ioctl;
  }

  public String getLabel() {
    return label;
  }

  public String getPath() {
    return path;
  }
  
  public String getHostPath() {
  	return (mountingType == MountingType.PHYSFS)? path + ':': path;
  }

  public MountingType getMountingType() {
    return mountingType;
  }
  
  public String getMountAs() {
    return mountAs;
  }
  
  public String getWrite() {
    return write;
  }
  
  public String getFreesize() {
  	return freesize;
  }
}
