/* 
 * Created on Jul 23, 2006 by ronald.blankendaal
 * 
 * @file $RCSfile: FileUtils.java,v $
 * @version $Revision: 1.13 $ 
 * @author $Author: ronald $ (last checked in by) 
 * @date $Date: 2007/01/10 11:47:06 $ (UTC date of last check in)
 */

package com.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import SevenZip.MyRandomAccessFile;
import SevenZip.Archive.IInArchive;
import SevenZip.Archive.SevenZipEntry;
import SevenZip.Archive.SevenZip.Handler;
import com.gui.StreamGobbler;
import com.model.Autoexec;
import com.model.Configuration;
import com.model.DosboxVersion;
import com.model.Profile;
import com.model.Settings;


public abstract class FileUtils {

	public static final String DBFILE = "./db/database";
	public static final String[] DBFILES = {"db/database.script", "db/database.properties"};
	public static final String[] DEFDBFILES = {"db/initial.script",  "db/initial.properties"};
	public static final String DFEND_PATH = "C:\\Program Files\\D-Fend\\";
  public static final String DFEND_PROFILES = "Profiles.dat";
  public static final String DOSBOX_CONF = "dosbox.conf";
  public static final String CAPTURES_DIR = "captures" + File.separatorChar;
  public static final String PROFILES_DIR = "profiles" + File.separatorChar;
  public static final String CONF_EXT = ".conf";
  public static final String[] EXECUTABLES = {".exe", ".com", ".bat"};
  public static final String[] ARCHIVES = {".zip", ".7z"};
  public static final String[] BOOTERIMAGES = {".cp2", ".dcf", ".img", ".jrc", ".td0"};
  public static final String[] CDIMAGES = {".iso", ".cue"};
  
  public static final String CNF_FILTER = "*.conf;*.CONF";
  public static final String EXE_FILTER = "*.com;*.COM;*.exe;*.EXE;*.bat;*.BAT";
  public static final String ARC_FILTER = "*.zip;*.ZIP;*.7z;*.7Z";
  public static final String BTR_FILTER = "*.cp2;*.CP2;*.dcf;*.DCF;*.img;*.IMG;*.jrc;*.JRC;*.td0;*.TD0";
  public static final String CDI_FILTER = "*.iso;*.ISO;*.cue;*.CUE";
  public static final String ALL_FILTER = "*.*";
  
  private static final String TEMPLATES_DIR = "templates" + File.separatorChar;
  private static final String SETUP_CONF = "setup.conf";
  private static String CWD;
  
  static {
  	try {
  		CWD = new File(".").getCanonicalPath() + File.separatorChar;
  	} catch(IOException ex) {
      System.err.println("FATAL ERROR: Couldn't get current working directory\n");
      System.exit(1);
    } 
  }
  
  private static String getCanonicalPath(String path) {
  	try {
  		return new File(path).getCanonicalPath();
  	} catch (IOException e) {
  		return new File(path).getAbsolutePath();
  	}
  }

  private static boolean isExistingFile(String filename) {
    File file = new File(filename);
    return file.isFile() && file.exists();
  }

  /** Fast & simple file copy. */
  private static void copy(File source, File dest) {
  	FileChannel in = null, out = null;
  	try {          
  		in = new FileInputStream(source).getChannel();
  		out = new FileOutputStream(dest).getChannel();
  		in.transferTo(0, in.size(), out);
  	} catch (IOException e) {
  		System.err.println("Couldn't copy file " + source + " to directory " + dest + "!!\n");
  	} finally {
  		if (in != null) try {in.close();} catch (IOException ex) {}
  		if (out != null) try {out.close();} catch (IOException ex) {}
  	}
  }

  private static void executeCommand(List<String> execCommands, String cwd, boolean waitFor)
  		throws IOException {
  	StringBuffer cmd = new StringBuffer();
  	try {
			if (Settings.getInstance().getBooleanValue("dosbox", "hideconsole"))
				execCommands.add("-noconsole");
			String[] rtCmds = new String[execCommands.size()];
			for (int i = 0; i < execCommands.size(); i++) {
				rtCmds[i] = execCommands.get(i);
				cmd.append(rtCmds[i] + " ");
			}
			System.out.println(cmd);
			File dir = (cwd != null)? new File(cwd): null;
			Process proc = Runtime.getRuntime().exec(rtCmds, null, dir);
			StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "DOSBox stderr");            
			StreamGobbler outputGobbler = new StreamGobbler(proc.getInputStream(), "DOSBox stdout");
			outputGobbler.start();
			errorGobbler.start();
			if (waitFor)
				try {	proc.waitFor();} catch (InterruptedException e) {}
		} catch (IOException e) {
			e.printStackTrace();
    	throw new IOException("Dosbox could not be started with the following command:\n" + cmd);
		}
  }

  private static void doRunDosbox(DosboxVersion dbversion,
  		List<String> parameters, boolean waitFor) throws IOException {
  	List<String> commandItems = new ArrayList<String>();
  	commandItems.add(dbversion.getExecutable());
  	commandItems.addAll(parameters);
  	if (dbversion.getParameters().length() > 0)
  		for (String p: dbversion.getParameters().split(" "))
  			commandItems.add(p);
  	executeCommand(commandItems, getCanonicalPath(dbversion.getPath()), waitFor);
  }
  
  public static void createDir(String dir) {
    if (!new File(dir).exists() && !new File(dir).mkdirs())
      System.err.println("Couldn't create " + dir + " directory!!\n");
  }
  
  public static void copyFiles(String srcDir, String dstDir) {
    File[] srcFiles = new File(srcDir).listFiles();
    if (srcFiles != null)
      for (File src: srcFiles)
        if (src.isFile()) copy(src, new File(dstDir + File.separatorChar + src.getName()));
  }

  public static void removeFile(String filename) {
    File file = new File(filename);
    if (! (file.isFile() && file.delete()) )
      System.err.println("Couldn't delete file " + filename);
  }
  
  public static void removeFilesInDirAndDir(String dirname) {
  	File[] files = new File(dirname).listFiles();
  	if (files != null) {
  		for (File file: files)
  			if (file.isDirectory()) {
  				System.err.println("Directory found inside " + dirname + ", aborting removal");
  				return;
  			}
  		for (File file: files)
  			if (!file.delete())
  	      System.err.println("Couldn't delete file " + file);
  	}
  	File dir = new File(dirname);
    if (! (dir.isDirectory() && dir.delete()) )
    	System.err.println("Couldn't delete directory " + dirname);
  }
  
  public static boolean isExistingReadableFile(String filename) {
    File file = new File(filename);
    return file.isFile() && file.canRead();
  }
  
  public static String constructCanonicalProfileConfig(String configPathAndFilename) {
  	return getCanonicalPath(configPathAndFilename);
  }
  
  public static String constructCapturesDir(int profileId) {
  	return CAPTURES_DIR + profileId;
  }
  
  public static String constructCanonicalCapturesDir(int profileId) {
  	return CWD + CAPTURES_DIR + profileId;
  }
  
  public static String constructCanonicalDBConfLocation(String path) {
  	return getCanonicalPath(path + DOSBOX_CONF);
  }
  
  public static String constructCanonicalDBExeLocation(String path) {
  	return getCanonicalPath(path + PlatformUtils.DB_EXECUTABLE);
  }
  
  public static String constructTemplateFileLocation(int templateId) {
  	return TEMPLATES_DIR + templateId + CONF_EXT;
  }
  
  public static void doRunDosbox(DosboxVersion dbversion) throws IOException {
	  List<String> parameters = new ArrayList<String>();
	  parameters.add("-conf");
	  parameters.add(DOSBOX_CONF);
		doRunDosbox(dbversion, parameters, false);
  }
  
  public static void doCreateDosboxConf(DosboxVersion dbversion) throws IOException {
  	List<String> parameters = new ArrayList<String>();
		parameters.add("-c");
		parameters.add("config -writeconf " + DOSBOX_CONF);
		parameters.add("-c");
		parameters.add("exit");
		doRunDosbox(dbversion, parameters, true);
  }
  
  public static void doRunProfile(Profile prof, List<DosboxVersion> dbversionsList,
  		boolean setup) throws IOException {
  	int dbversionIndex = ListUtils.findTIDosboxVersion(dbversionsList, prof.getDbversionId());
  	DosboxVersion dbversion = dbversionsList.get(dbversionIndex);
  	doRunProfile(prof, dbversion, setup);
  }
  
  public static void doRunProfile(Profile prof, DosboxVersion dbversion,
  		boolean setup) throws IOException {
  	List<String> confs = new ArrayList<String>();
  	confs.add(dbversion.getExecutable());
  	if (dbversion.isMultiConfig()) {
  		// selected default dosbox config file
  		confs.add("-conf");
  		confs.add(DOSBOX_CONF); 
  	}
  	// add profile-specific settings
  	String filename = prof.getCanonicalConfPathAndFile();
  	if (setup && prof.hasSetupBoolean()) {
  		Configuration conf = new Configuration(prof.getConfPathAndFile());
  		Autoexec autoexec = new Autoexec(conf.getValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC));
  		autoexec.setSetup(prof.getSetup());
  		autoexec.setSetupParameters(prof.getSetupParameters());
  		conf.setValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC, autoexec.toString(true));
  		conf.saveToFile(PROFILES_DIR + SETUP_CONF);
  		filename = CWD + PROFILES_DIR + SETUP_CONF;
  	}
  	confs.add("-conf");
  	confs.add(filename);
  	if (dbversion.getParameters().length() > 0)
  		for (String p: dbversion.getParameters().split(" "))
  			confs.add(p);
  	executeCommand(confs, getCanonicalPath(dbversion.getPath()), false);
  }
  
  public static String constructUniqueConfigFile(int profileId, String profileTitle,
  		String gameLocation) {
  	Settings set = Settings.getInstance();
  	String candidateFilename = "";
  	int candidateNr = 1;
  	boolean unique = false;
  	while (!unique) {
  		candidateFilename = PROFILES_DIR;
    	if ((set.getIntValue("profiledefaults", "confpath") == 1) && (gameLocation != null))
    		candidateFilename = gameLocation;
    	
	  	if (set.getIntValue("profiledefaults", "conffile") == 1) {
	  		String candidateTitle = profileTitle;
	  		if (candidateNr > 1)
	  			candidateTitle += "(" + candidateNr + ")";
	  		candidateFilename += candidateTitle.replaceAll("[^a-zA-Z_0-9()]", "") + CONF_EXT;
	  	} else {
	  		if (candidateNr > 1)
	  			return null; // id files should be unique, else give up
	  		candidateFilename += profileId + CONF_EXT;
	  	}
	  	candidateNr++;
	  	unique = !isExistingFile(candidateFilename);
  	}
  	return candidateFilename;
  }

  public static void renameDatabaseIfNecessary() {
  	if (!isExistingFile(DBFILES[0]) && !isExistingFile(DBFILES[1]) && 
  			isExistingReadableFile(DEFDBFILES[0]) && isExistingReadableFile(DEFDBFILES[1])) {
  		new File(DEFDBFILES[0]).renameTo(new File(DBFILES[0]));
  		new File(DEFDBFILES[1]).renameTo(new File(DBFILES[1]));
  	}
  }
  
  public static boolean isExecutable(String filename) {
  	for (String ext: EXECUTABLES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isArchive(String filename) {
  	for (String ext: ARCHIVES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isPhysFS(String mountPath) {
  	for (String ext: ARCHIVES)
  		if (mountPath.toLowerCase().endsWith(ext + ":\\")) return true;
  	return false;
  }
  
  public static boolean containsPhysFS(String mountPath) {
  	for (String ext: ARCHIVES)
  		if (mountPath.toLowerCase().contains(ext + ":\\")) return true;
  	return false;
  }
  
  public static boolean isBooterImage(String filename) {
  	for (String ext: BOOTERIMAGES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isConfFile(String filename) {
  	return (filename.toLowerCase().endsWith(CONF_EXT));
  }
  
  public static String[] getExecutablesInZip(String archive) throws IOException {
  	List<String> result = new ArrayList<String>();
  	File arcFile = new File(archive);
		
  	if (archive.toLowerCase().endsWith(ARCHIVES[0])) { // zip
  		ZipFile zf = new ZipFile(arcFile);
  		for (Enumeration entries = zf.entries(); entries.hasMoreElements();) {
  			ZipEntry entry = (ZipEntry)entries.nextElement();
  			String name = entry.getName();
  			if (!entry.isDirectory() && isExecutable(name))
  				result.add(PlatformUtils.archiveToNativePath(name));
  		}
  	} else if (archive.toLowerCase().endsWith(ARCHIVES[1])) { //7-zip
  		MyRandomAccessFile istream = new MyRandomAccessFile(archive, "r");
      IInArchive zf = new Handler();
      if (zf.Open(istream) != 0)
          throw new IOException("Failure opening 7-zip archive");
      for(int i = 0; i < zf.size() ; i++) {
        SevenZipEntry entry = zf.getEntry(i);
  			String name = entry.getName();
  			if (!entry.isDirectory() && isExecutable(name))
  				result.add(PlatformUtils.archiveToNativePath(name));
  		}
  	}
		
  	Collections.sort(result,
  			new Comparator<String>() {
	  			public int compare(String s1, String s2) {
	  				int count1 = StringUtils.countCharacters(s1, '\\');
	  				int count2 = StringUtils.countCharacters(s2, '\\');
	  				if (count1 != count2)
	  					return count1 - count2;
	  				else
	  					return s1.compareTo(s2);
	  			}
	  		} );
  	
		return result.toArray(new String[0]);
  }
}
