package com.model;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Set;
import java.util.TreeMap;
import com.util.StringUtils;


public class Configuration {

	public static final String AUTOEXEC = "autoexec";
  
	private java.util.HashMap<String, Section> sections;
  private StringBuffer parsingProblems;
  

  private final class Section {
  	protected java.util.HashMap<String, String> items;
    
    private Section() {
      items = new java.util.HashMap<String, String>();
    }
    
    private Section(Section sec) {
      this();
      for (String key: sec.items.keySet())
        items.put(new String(key), new String(sec.items.get(key)));
    }
    
    private void addAutoexecItem(String value) {
      if (items.containsKey(AUTOEXEC))
        items.put(AUTOEXEC, items.get(AUTOEXEC) + "\n" + value);
      else
        items.put(AUTOEXEC, value); 
    }
    
    private void substract(Section sec) {
      for (String key: sec.items.keySet()) {
        if (items.containsKey(key)) {
          String value = sec.items.get(key);
          String thisvalue = items.get(key);
          if (thisvalue.equals(value))
            items.remove(key);
        }
      }
    }

    private String toString(boolean ordered) {
      StringBuffer result = new StringBuffer();
      for (String key: ordered? new TreeMap<String, String>(items).keySet(): items.keySet()) {
        if (!key.equals(AUTOEXEC))
        	result.append(key + "=");
        result.append(items.get(key) + "\n");
      }
      return result.toString();
    }
  }
  
  public Configuration() {
  	parsingProblems = new StringBuffer();
    sections = new java.util.HashMap<String, Section>();
  }
  
  public Configuration(Configuration conf) {
    this();
    for (String key: conf.sections.keySet())
      sections.put(key, new Section(conf.sections.get(key)));
  }
  
  public Configuration(File file) throws IOException {
    this(file, true);
  }
  
  public Configuration(File file, boolean toLowerCase) throws IOException {
    this();
    parseConfigurationFile(file, toLowerCase);
  }
  
  public Configuration emergencyProfileInit() {
  	createSection("dosbox"); // for 'captures' value
		createSection(AUTOEXEC);
		return this;
  }

  public Set<String> getSectionKeys() {
    return sections.keySet();
  }
  
  public void saveToFile(File file) throws IOException {
	  saveToFile(file, false);
  }
  
  protected void saveToFile(File file, boolean ordered) throws IOException {
    try {
      BufferedWriter configData = new BufferedWriter(new FileWriter(file));
      configData.write(toString(ordered));
      configData.flush();
      configData.close();
    } catch (IOException e) {
    	e.printStackTrace();
    	throw new IOException("Something went wrong while trying to save " + file);
    }
  }

	public void parseConfigurationFile(File file) throws IOException {
    parseConfigurationFile(file, true);
  }
  
  public void parseConfigurationFile(File file, boolean toLowerCase) throws IOException {
    try {
      BufferedReader configData = new BufferedReader(new FileReader(file));
      String orgTextLine;
      String currentSectionTitle = null;
      Section currentSection = null;

      while ((orgTextLine = configData.readLine()) != null) {
      	orgTextLine = orgTextLine.trim();
        String textLine = toLowerCase? orgTextLine.toLowerCase(): new String(orgTextLine);
        if ((textLine.length() > 0) && !textLine.startsWith("#")) {
          if (textLine.startsWith("[")) { // a new section starts here
            int start = textLine.indexOf(("["));
            int end = textLine.lastIndexOf("]");
            if (end == -1) {
            	parsingProblems.append("Parsing error in " + file + ", line: " + textLine + "\n");
            } else {
              currentSectionTitle = textLine.substring(start + 1, end);
              if (sections.containsKey(currentSectionTitle)) {
                currentSection = sections.get(currentSectionTitle);
              } else {
                currentSection = new Section();
                sections.put(currentSectionTitle, currentSection);
              }
            }
          } else { // an item starts here
            if (currentSectionTitle != null) {
            	if (currentSectionTitle.equals(AUTOEXEC)) { // autoexec config item
            		currentSection.addAutoexecItem(orgTextLine);
	            } else {                                    // normal config item
	              int end = textLine.indexOf("=");
	              if (end == -1) {
	              	parsingProblems.append("Parsing error in " + file + ", line: " + textLine + "\n");
	              } else {
	                String name = textLine.substring(0, end).trim();
	                String value = textLine.substring(end + 1).trim();
	                currentSection.items.put(name, value);
	              }
	            }
            } else { // value before section
            	parsingProblems.append("Parsing error (no section) in " + file + ", line: " + textLine + "\n");
            }
          }
        }
      }
      configData.close();
    }
    catch (FileNotFoundException e) {
    	e.printStackTrace();
    	throw new IOException("Could not read " + e.getMessage());
    }
    catch (IOException e) {
    	e.printStackTrace();
    	String msg = "Something went wrong while trying to read " + file;
    	if (parsingProblems.length() > 0)
      	msg += "\n" + getParsingProblems();
    	throw new IOException(msg);
    }
  }
  
  public boolean hasValue(String sectionTitle, String sectionItem) {
		return sections.containsKey(sectionTitle) &&
			sections.get(sectionTitle).items.containsKey(sectionItem);
	}
	
	public String getValue(String sectionTitle, String sectionItem) {
	  String result = null;
	  if (sections.containsKey(sectionTitle))
	    result = sections.get(sectionTitle).items.get(sectionItem);
	  if (result == null)
	    return ""; // in case the item was not found
	  return result;
	}

	public int getIntValue(String sectionTitle, String sectionItem) {
	  try {
		  return Integer.parseInt(getValue(sectionTitle, sectionItem));
	  } catch (NumberFormatException e) {
		  return -1; // value is not a number
	  }
  }
  
  public boolean getBooleanValue(String sectionTitle, String sectionItem) {
	  return Boolean.valueOf(getValue(sectionTitle, sectionItem));
  }
  
  public void createSection(String sectionTitle) {
	  if (!sections.containsKey(sectionTitle))
		  sections.put(sectionTitle, new Section());
  }
  
  public void setValue(String sectionTitle, String sectionItem, String value) {
    if (sections.containsKey(sectionTitle))
      sections.get(sectionTitle).items.put(sectionItem, value);
  }
  
  public void setIntValue(String sectionTitle, String sectionItem, int value) {
	  setValue(sectionTitle, sectionItem, String.valueOf(value));
  }
  
  public void setBooleanValue(String sectionTitle, String sectionItem, boolean value) {
	  setValue(sectionTitle, sectionItem, String.valueOf(value));
  }
  
  public void updateValue(String sectionTitle, String sectionItem, String value) {
    if (sections.containsKey(sectionTitle))
    	if (sections.get(sectionTitle).items.containsKey(sectionItem))
    		sections.get(sectionTitle).items.put(sectionItem, value);
  }
	
  public void removeSection(String title) {
    sections.remove(title);
  }
  
  public void removeValue(String sectionTitle, String sectionItem) {
    if (sections.containsKey(sectionTitle)) {
      Section sec = sections.get(sectionTitle);
      sec.items.remove(sectionItem);
      if (sec.items.isEmpty())
        sections.remove(sectionTitle);
    }
  }
  
  public void substract(Configuration conf) {
    for (String key: conf.sections.keySet()) {
      if (sections.containsKey(key)) {
        Section mysec = sections.get(key);
        mysec.substract(conf.sections.get(key));
        if (mysec.items.isEmpty())
          sections.remove(key);
      }
    }
  }
  
  public String[] getItems(String section) {
		return new TreeMap<String, String>(sections.get(section).items).keySet().toArray(new String[0]);
	}

  public String[] getValues(String sectionTitle, String sectionItem) {
  	String val = getValue(sectionTitle, sectionItem);
  	if (val.length() <= 0)
  		return new String[0];
  	String[] res = val.split(" ");
  	for (int i = 0; i < res.length; i++)
  		res[i] = res[i].replaceAll("<space>", " ");
  	return res;
	}
  
  public String getMultilineValues(String sectionTitle, String sectionItem) {
		return StringUtils.StringArrayToString(getValues(sectionTitle, sectionItem));
	}
  
  public void setMultilineValues(String sectionTitle, String sectionItem, String values, String delimiter) {
		setValue(sectionTitle, sectionItem, values.replaceAll(" ", "<space>").replace(delimiter, " ").trim());
	}
  
  public int[] getIntValues(String sectionTitle, String sectionItem) {
		return StringUtils.stringToIntArray(getValue(sectionTitle, sectionItem));
	}
	
  public void setIntValues(String sectionTitle, String sectionItem, int[] values) {
		setValue(sectionTitle, sectionItem, StringUtils.intArrayToString(values));
	}
	
  public boolean[] getBooleanValues(String sectionTitle, String sectionItem) {
		return StringUtils.stringToBooleanArray(getValue(sectionTitle, sectionItem));
	}
	
	public void setBooleanValues(String sectionTitle, String sectionItem, boolean[] values) {
		setValue(sectionTitle, sectionItem, StringUtils.booleanArrayToString(values));
	}
	
	public void updateWithValuesFrom(Configuration conf) {
		for (String sectionTitle: conf.sections.keySet())
			for (String sectionItem: conf.getItems(sectionTitle))
				this.setValue(sectionTitle, sectionItem, conf.getValue(sectionTitle, sectionItem));
	}
  
  public String toString(boolean ordered) {
    StringBuffer result = new StringBuffer();
    StringBuffer autoexec = new StringBuffer();
    for (String key: sections.keySet()) {
      if (!key.equals(AUTOEXEC)) {
      	result.append("[" + key + "]\n");
      	result.append((sections.get(key)).toString(ordered) + "\n");
      } else {
      	autoexec.append("[" + key + "]\n");
        autoexec.append((sections.get(key)).toString(false));
      }
    }
    return result.toString() + autoexec.toString();
  }

  public boolean hasParsingProblems() {
  	return parsingProblems.length() > 0;
  }
  
  public String getParsingProblems() {
  	String result = parsingProblems.toString();
  	parsingProblems = new StringBuffer();
  	return result;
  }
}
