package com.model;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import com.util.FileUtils;
import com.util.StringUtils;


public class Mount {

  public enum MountingType { DIR, IMAGE, PHYSFS };
  
  private MountingType mountingType;
  private String mountAs;
  private char driveletter;
  private String label;
  private File[] path;
  private File write;
  private boolean ioctl;
  private String freesize;
  
  
  public Mount(MountingType mountingType, String mountAs, String driveletter, String[] paths,
          String label, boolean ioctl, String write, String freesize) {
    this.mountingType = mountingType;
    this.mountAs = mountAs.toLowerCase();
    this.driveletter = Character.toUpperCase(driveletter.charAt(0));
    this.path = new File[paths.length];
    for (int i = 0; i < paths.length; i++)
    	this.path[i] = new File(paths[i]);
    this.write = new File(write);
    this.label = label;
    this.ioctl = ioctl;
    this.freesize = freesize;
  }

  private void init() {
    mountingType = MountingType.DIR;
    mountAs = "";
    driveletter = '\0';
    label = "";
    path = null;
    write = null;
    ioctl = false;
    freesize = "";
  }
  
  private void initForPhysFS(String physFSPath) {
    mountingType = MountingType.PHYSFS;
    int colonIndex1 = physFSPath.indexOf(":");
    if (colonIndex1 == 1)
      colonIndex1 = physFSPath.indexOf(":", colonIndex1 + 1);
    int colonIndex2 = physFSPath.lastIndexOf(":");
    path = new File[1];
    if (colonIndex1 != colonIndex2) {
      write = new File(physFSPath.substring(0, colonIndex1));
      path[0] = new File(physFSPath.substring(colonIndex1 + 1, colonIndex2));
    } else {
      path[0] = new File(physFSPath.substring(0, colonIndex2));
    }
  }
  
  public Mount(boolean floppy, String hostFileLocation, String[] existingMounts) {
  	init();
  	if (hostFileLocation.equals(""))
      return;
    
  	List<Character> freeDriveletters = getFreeDriveletters(floppy, existingMounts);
  	if (freeDriveletters.size() < 1) // no free driveletters left, give up
  		return;
    driveletter = freeDriveletters.get(0);
    
    if (FileUtils.containsPhysFS(hostFileLocation)) {
    	initForPhysFS(hostFileLocation);
    } else {
    	File file = new File(hostFileLocation);
    	path = new File[1];
    	path[0] = file.getParentFile();
    	if (path[0] == null)
    		path[0] = new File(".");
    }
  }
  
  public Mount(String mount) {
    init();
    if (mount.equals(""))
      return;
    
    int spaceIndex = mount.indexOf(' ');
    driveletter = Character.toUpperCase(mount.charAt(spaceIndex + 1));
    
    int firstQuoteIndex = mount.indexOf('"');
    int secondQuoteIndex = mount.lastIndexOf('"');
    if (firstQuoteIndex == -1 && secondQuoteIndex == -1) {
    	// not using quotes around path
    	firstQuoteIndex = mount.indexOf(' ', spaceIndex + 1);
    	secondQuoteIndex = mount.indexOf(' ', firstQuoteIndex + 1);
    	if (secondQuoteIndex == -1) secondQuoteIndex = mount.length();
    }
    String mountLocation = mount.substring(firstQuoteIndex + 1, secondQuoteIndex);
    String[] paths = StringUtils.mountToStringArray(mountLocation);
    path = new File[paths.length];
    for (int i = 0; i < paths.length; i++)
    	path[i] = FileUtils.makeRelativeToDosroot(new File(paths[i]));
    
    if (mount.toLowerCase().startsWith("mount ")) {
    	if (FileUtils.isPhysFS(mountLocation))
      	initForPhysFS(mountLocation);
    	
      int typeIndex = mount.indexOf("-t ");
      if (typeIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", typeIndex + 3);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        mountAs = mount.substring(typeIndex + 3, spaceIndex2);
      }
      int labelIndex = mount.indexOf("-label ");
      if (labelIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", labelIndex + 7);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        label = mount.substring(labelIndex + 7, spaceIndex2);
      }
      if (mount.indexOf("-ioctl") != -1) {
        ioctl = true;
      }
      int freesizeIndex = mount.indexOf("-freesize ");
      if (freesizeIndex != -1) {
        int spaceIndex2 = mount.indexOf(" ", freesizeIndex + 10);
        if (spaceIndex2 == -1) spaceIndex2 = mount.length();
        freesize = mount.substring(freesizeIndex + 10, spaceIndex2);
      }
      
    } else if (mount.toLowerCase().startsWith("imgmount ")) {
      mountingType = MountingType.IMAGE;
      mountAs = "iso";
    }
  }
  
  public static List<Character> getFreeDriveletters(boolean floppy, String[] existingMounts) {
  	Mount[] mnts = new Mount[existingMounts.length];
  	for (int i = 0; i < mnts.length; i++)
  		mnts[i] = new Mount(existingMounts[i]);
  	List<Character> freeDriveletters = new ArrayList<Character>();
  	char start = floppy? 'A': 'C';
  	for (char i = start; i < 'Z'; i++)
    	freeDriveletters.add(i);
  	if (!floppy) {
  		freeDriveletters.add('A');
  		freeDriveletters.add('B');
  	}
  	for (int i = 0; i < mnts.length; i++)
  		freeDriveletters.remove(new Character(mnts[i].getDriveletter()));
  	return freeDriveletters;
  }

  public String toString() {
    StringBuffer result = new StringBuffer();
    switch (mountingType) {
      case DIR: 
        result.append("mount " + driveletter + " \"" + getPathAsString() + "\"");
        if (ioctl) result.append(" -ioctl");
        break;
      case PHYSFS:
        result.append("mount " + driveletter + " \"");
        if (!getWriteAsString().equals("")) result.append(getWriteAsString() + "\\:");
        result.append(path[0].getPath() + ":\\\"");
        break;
      case IMAGE:
      	result.append("imgmount " + driveletter);
      	for (int i = 0; i < path.length; i++)
      		result.append(" \"" + path[i].getPath() + "\"");
        break;
    }
    if (!label.equals("")) result.append(" -label " + label);
    if (!mountAs.equals("")) result.append(" -t " + mountAs);
    if (!freesize.equals("")) result.append(" -freesize " + freesize);
    return result.toString();
  }

  public char getDriveletter() {
    return driveletter;
  }

  public String getDriveletterString() {
    return String.valueOf(driveletter);
  }

  public boolean isIoctl() {
    return ioctl;
  }

  public String getLabel() {
    return label;
  }

  public String getPathAsString() {
  	if (path.length <= 0)
  		return "";
  	return path[0].getPath();
	}
  
  public char getDriveletterFromPath() {
  	return Character.toUpperCase(path[0].getAbsolutePath().charAt(0));
	}
  	
  public String getImgMountAsString() {
  	String[] paths = new String[path.length];
  	for (int i = 0; i < path.length; i++)
  		paths[i] = path[i].getPath();
  	return StringUtils.StringArrayToString(paths);
	}
  
  public String getHostPathAsString() {
  	return (mountingType == MountingType.PHYSFS)? getPathAsString() + ':': getPathAsString();
  }

  public MountingType getMountingType() {
    return mountingType;
  }
  
  public String getMountAs() {
    return mountAs;
  }
  
  public String getWriteAsString() {
  	if (write == null)
  		return "";
  	return write.getPath();
	}
  	
  public String getFreesize() {
  	return freesize;
  }
  
  public File canBeUsedFor(File hostFile) {
  	hostFile = FileUtils.canonicalToDosroot(hostFile.getPath());
  	if (FileUtils.areRelated(FileUtils.canonicalToDosroot(getHostPathAsString()), hostFile))
  		return FileUtils.makeRelativeTo(hostFile, FileUtils.canonicalToDosroot(getHostPathAsString()));
  	return null;
  }
  
  public void makeRelative(File basePath) {
  	for (int i = 0; i < path.length; i++)
  		path[i] = FileUtils.makeRelativeTo(path[i], basePath);
  	if (write != null)
  		write = FileUtils.makeRelativeTo(write, basePath);
  }
}
