package com.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import SevenZip.MyRandomAccessFile;
import SevenZip.Archive.IInArchive;
import SevenZip.Archive.SevenZipEntry;
import SevenZip.Archive.SevenZip.Handler;
import com.gui.StreamGobbler;
import com.model.Autoexec;
import com.model.Configuration;
import com.model.DosboxVersion;
import com.model.Profile;
import com.model.Settings;


public final class FileUtils {

	private static final String DOSBOX_CONF = "dosbox.conf";
	private static final String[] DBFILES = {"database.script", "database.properties"};
	private static final String[] DEFDBFILES = {"initial.script",  "initial.properties"};
	private static final File DATA_DIR_FILE;
	private static final File DOSROOT_DIR_FILE;
	private static final File DOSBOX_DIR_FILE;
  private static final String DATABASE_DIR = "db" + File.separatorChar;
  private static final String TEMPLATES_DIR = "templates" + File.separatorChar;
  private static final String SETUP_CONF = "setup.conf";
  
  public static final String DOSROOT_DIR = "dosroot" + File.separatorChar;
  public static final String CAPTURES_DIR = "captures" + File.separatorChar;
  public static final String PROFILES_DIR = "profiles" + File.separatorChar;
  public static final String CONF_EXT = ".conf";
  public static final String[] EXECUTABLES = {".exe", ".com", ".bat"};
  public static final String[] ARCHIVES = {".zip", ".7z"};
  public static final String[] BOOTERIMAGES = {".cp2", ".dcf", ".img", ".jrc", ".td0"};
  
  public static final String CNF_FILTER = "*.conf;*.CONF";
  public static final String EXE_FILTER = "*.com;*.COM;*.exe;*.EXE;*.bat;*.BAT";
  public static final String ARC_FILTER = "*.zip;*.ZIP;*.7z;*.7Z";
  public static final String BTR_FILTER = "*.cp2;*.CP2;*.dcf;*.DCF;*.img;*.IMG;*.jrc;*.JRC;*.td0;*.TD0";
  public static final String CDI_FILTER = "*.iso;*.ISO;*.cue;*.CUE";
  public static final String ALL_FILTER = "*";
  
  
  static {
  	Settings settings = Settings.getInstance();
		String DATADIR = getCanonicalPath(settings.getValue("directory", "data"));
		DATA_DIR_FILE = new File(DATADIR);
		DOSROOT_DIR_FILE = new File(DATADIR, DOSROOT_DIR);
		String DOSBOXDIR = getCanonicalPath(settings.getValue("directory", "dosbox"));
		DOSBOX_DIR_FILE = new File(DOSBOXDIR);
		
		createDosrootIfNecessary();
  }
  
  private static String getCanonicalPath(String path) {
		if (!PlatformUtils.IS_LINUX || PlatformUtils.IS_OSX) {
			//Linux and OSX have ~/ for homedirectory. Allow single ~ as well
			if(path.startsWith("~/") || (path.length() == 1 && path.charAt(0) == '~'))
				path = path.replaceAll("^~", System.getProperty("user.home"));
		}
  	try {
  		return new File(path).getCanonicalPath();
  	} catch (IOException e) {
  		return new File(path).getAbsolutePath();
  	}
  }

  private static boolean isExistingFile(File file) {
    return file.isFile() && file.exists();
  }

  private static void copy(File source, File dest) {
  	FileChannel in = null, out = null;
  	try {          
  		in = new FileInputStream(source).getChannel();
  		out = new FileOutputStream(dest).getChannel();
  		in.transferTo(0, in.size(), out);
  	} catch (IOException e) {
  		System.err.println("Couldn't copy file " + source + " to directory " + dest + "!!\n");
  	} finally {
  		if (in != null) try {in.close();} catch (IOException ex) {}
  		if (out != null) try {out.close();} catch (IOException ex) {}
  	}
  }

  private static void executeCommand(List<String> execCommands, File cwd, boolean waitFor)
  		throws IOException {
  	StringBuffer cmd = new StringBuffer();
  	try {
			if (Settings.getInstance().getBooleanValue("dosbox", "hideconsole"))
				execCommands.add("-noconsole");
			String[] rtCmds = new String[execCommands.size()];
			for (int i = 0; i < execCommands.size(); i++) {
				rtCmds[i] = execCommands.get(i);
				cmd.append(rtCmds[i] + " ");
			}
			System.out.println(cmd);
			File dir = (cwd != null)? cwd: DOSROOT_DIR_FILE;
			Process proc = Runtime.getRuntime().exec(rtCmds, null, dir);
			StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "DOSBox stderr");            
			StreamGobbler outputGobbler = new StreamGobbler(proc.getInputStream(), "DOSBox stdout");
			outputGobbler.start();
			errorGobbler.start();
			if (waitFor)
				try {	proc.waitFor();} catch (InterruptedException e) {}
		} catch (IOException e) {
			e.printStackTrace();
    	throw new IOException("Dosbox could not be started with the following command:\n" + cmd);
		}
  }

  private static void doRunDosbox(DosboxVersion dbversion,
  		List<String> parameters, File cwd, boolean waitFor) throws IOException {
  	List<String> commandItems = new ArrayList<String>();
  	commandItems.add(dbversion.getCanonicalExecutable().getPath());
  	commandItems.addAll(parameters);
  	if (dbversion.getParameters().length() > 0)
  		for (String p: dbversion.getParameters().split(" "))
  			commandItems.add(p);
  	executeCommand(commandItems, cwd, waitFor);
  }
  
  private static File canonicalTo(File base, String path) {
  	File f = new File(path);
  	if (f.isAbsolute())
  		return f;
		else
			try {
				return new File(base, f.getPath()).getCanonicalFile();
			} catch (IOException e) {
				return new File(base, f.getPath()).getAbsoluteFile();
			}
  }
  
  public static boolean areRelated(File parent, File child) {
  	File remainder = child.getParentFile();
  	while (remainder != null) {
  		if (parent.equals(remainder))
  			return true;
  		remainder = remainder.getParentFile();
  	}
  	return false;
  }
  
  public static String getDosRoot() {
  	return DOSROOT_DIR_FILE.getPath();
  }

  public static File makeRelativeTo(File file, File basePath) {
  	if (!file.isAbsolute())
  		return file;
  	if (file.equals(basePath))
			return new File(".");
  	File remainder = new File(file.getName());
  	File parent = file.getParentFile();
  	while (parent != null) {
  		if (parent.equals(basePath))
  			return remainder;
  		remainder = new File(parent.getName(), remainder.getPath());
  		parent = parent.getParentFile();
  	}
  	return file;
  }
  
  public static File makeRelativeToData(File file) {
  	return makeRelativeTo(file, DATA_DIR_FILE);
  }
  
  public static File makeRelativeToDosroot(File file) {
  	return makeRelativeTo(file, DOSROOT_DIR_FILE);
  }
  
  public static File makeRelativeToDosbox(File file) {
  	return makeRelativeTo(file, DOSBOX_DIR_FILE);
  }

  public static File canonicalToData(String path) {
  	return canonicalTo(DATA_DIR_FILE, path);
  }
  
  public static File canonicalToDosbox(String path) {
  	return canonicalTo(DOSBOX_DIR_FILE, path);
  }
  
  public static File canonicalToDosroot(String path) {
  	return canonicalTo(DOSROOT_DIR_FILE, path);
  }
  
  public static String sanitizeToDosroot(String path) {
  	return makeRelativeToDosroot(canonicalToDosroot(path)).getPath();
  }
  
  public static File constructCanonicalDBConfLocation(String path) {
  	return canonicalToDosbox(new File(path, DOSBOX_CONF).getPath());
  }
  
  public static File constructCanonicalDBExeLocation(String path) {
  	return canonicalToDosbox(new File(path, PlatformUtils.DB_EXECUTABLE).getPath());
  }
  
  public static String constructCapturesDir(int profileId) {
  	return CAPTURES_DIR + profileId;
  }
  
  public static String constructRelativeCapturesDir(int profileId) {
  	return ".." + File.separatorChar + constructCapturesDir(profileId);
  }
  
  public static File constructCanonicalTemplateFileLocation(int templateId) {
  	return canonicalToData(TEMPLATES_DIR + templateId + CONF_EXT);
  }

  public static String constructUniqueConfigFileString(int profileId, String profileTitle,
  		Configuration gameConf) {
  	Settings set = Settings.getInstance();
  	String candidateFilename = "";
  	int candidateNr = 1;
  	boolean unique = false;
  	while (!unique) {
    	if (set.getIntValue("profiledefaults", "confpath") == 1) {
    		Autoexec autoexec = new Autoexec(gameConf.getValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC));
      	String loc = autoexec.isBooter()? autoexec.getImg1(): autoexec.getMain();
      	int index = loc.lastIndexOf(File.separatorChar);
      	candidateFilename = loc.substring(0, index + 1);
    	} else {
    		candidateFilename = PROFILES_DIR;
    	}
    	
	  	if (set.getIntValue("profiledefaults", "conffile") == 1) {
	  		String candidateTitle = profileTitle;
	  		if (candidateNr > 1)
	  			candidateTitle += "(" + candidateNr + ")";
	  		candidateFilename += candidateTitle.replaceAll("[^a-zA-Z_0-9()]", "") + CONF_EXT;
	  	} else {
	  		if (candidateNr > 1)
	  			return null; // id files should be unique, else give up
	  		candidateFilename += profileId + CONF_EXT;
	  	}
	  	candidateNr++;
	  	unique = !isExistingFile(new File(candidateFilename));
  	}
  	return candidateFilename;
  }

  public static void doRunDosbox(DosboxVersion dbversion) throws IOException {
	  List<String> parameters = new ArrayList<String>();
	  parameters.add("-conf");
	  parameters.add(dbversion.getCanonicalConfFile().getPath());
		doRunDosbox(dbversion, parameters, null, false);
  }
  
  public static void doCreateDosboxConf(DosboxVersion dbversion) throws IOException {
  	List<String> parameters = new ArrayList<String>();
		parameters.add("-c");
		parameters.add("config -writeconf " + DOSBOX_CONF);
		parameters.add("-c");
		parameters.add("exit");
		doRunDosbox(dbversion, parameters, dbversion.getCanonicalExecutable().getParentFile(), true);
  }
  
  public static void doRunProfile(Profile prof, List<DosboxVersion> dbversions,
  		boolean setup) throws IOException {
  	DosboxVersion db = dbversions.get(DosboxVersion.findById(dbversions, prof.getDbversionId()));
  	doRunProfile(prof, db, setup);
  }
  
  public static void doRunProfile(Profile prof, DosboxVersion dbversion,
  		boolean setup) throws IOException {
  	List<String> confs = new ArrayList<String>();
  	confs.add(dbversion.getCanonicalExecutable().getPath());
  	if (dbversion.isMultiConfig()) {
  		// selected default dosbox config file
  		confs.add("-conf");
  		confs.add(dbversion.getCanonicalConfFile().getPath()); 
  	}
  	// add profile-specific settings
  	File file = prof.getCanonicalConfFile();
  	if (setup && prof.hasSetup()) {
  		Configuration conf = new Configuration(file);
  		Autoexec autoexec = new Autoexec(conf.getValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC));
  		autoexec.setSetup(prof.getSetup());
  		autoexec.setSetupParameters(prof.getSetupParameters());
  		conf.setValue(Configuration.AUTOEXEC, Configuration.AUTOEXEC, autoexec.toString(true));
  		file = canonicalToData(PROFILES_DIR + SETUP_CONF);
  		conf.saveToFile(file);
  	}
  	confs.add("-conf");
  	confs.add(file.getPath());
  	if (dbversion.getParameters().length() > 0)
  		for (String p: dbversion.getParameters().split(" "))
  			confs.add(p);
  	executeCommand(confs, null, false);
  }
  
  public static void createDir(File dir) {
    if (!dir.exists() && !dir.mkdirs())
      System.err.println("Couldn't create " + dir + " directory!!\n");
  }
  
  public static void copyFiles(File srcDir, File dstDir) {
    File[] srcFiles = srcDir.listFiles();
    if (srcFiles != null)
      for (File src: srcFiles)
        if (src.isFile()) copy(src, new File(dstDir, src.getName()));
  }

  public static void removeFile(File file) {
    if (! (file.isFile() && file.delete()) )
      System.err.println("Couldn't delete file " + file.getPath());
  }
  
  public static void removeFilesInDirAndDir(File dir) {
  	File[] files = dir.listFiles();
  	if (files != null) {
  		for (File file: files)
  			if (file.isDirectory()) {
  				System.err.println("Directory found inside " + dir.getPath() + ", aborting removal");
  				return;
  			}
  		for (File file: files)
  			if (!file.delete())
  	      System.err.println("Couldn't delete file " + file);
  	}
  	if (! (dir.isDirectory() && dir.delete()) )
    	System.err.println("Couldn't delete directory " + dir.getPath());
  }
  
  public static boolean isReadableFile(File file) {
    return file.isFile() && file.canRead();
  }

  public static void renameDatabaseIfNecessary() {
  	File db0 = canonicalToData(DATABASE_DIR + DBFILES[0]);
  	File db1 = canonicalToData(DATABASE_DIR + DBFILES[1]);
  	File def0 = canonicalToData(DATABASE_DIR + DEFDBFILES[0]);
  	File def1 = canonicalToData(DATABASE_DIR + DEFDBFILES[1]);
  	if (!isExistingFile(db0) && !isExistingFile(db1) && isReadableFile(def0) && isReadableFile(def1)) {
  		def0.renameTo(db0);
  		def1.renameTo(db1);
  	}
  }
  
  public static void createDosrootIfNecessary() {
  	if (!DOSROOT_DIR_FILE.isDirectory()) {
  		System.out.print("Creating new directory '" + DOSROOT_DIR_FILE.getPath() + "' ... ");
  		if (!DOSROOT_DIR_FILE.mkdirs())
  			System.out.println("Failure");
  		else
  			System.out.println("Ok");
  	}
  }
  
  public static boolean isExecutable(String filename) {
  	for (String ext: EXECUTABLES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isArchive(String filename) {
  	for (String ext: ARCHIVES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isPhysFS(String mountPath) {
  	for (String ext: ARCHIVES)
  		if (mountPath.toLowerCase().endsWith(ext + ':' + File.separatorChar)) return true;
  	return false;
  }
  
  public static boolean containsPhysFS(String mountPath) {
  	for (String ext: ARCHIVES)
  		if (mountPath.toLowerCase().contains(ext + ':' + File.separatorChar)) return true;
  	return false;
  }
  
  public static boolean isBooterImage(String filename) {
  	for (String ext: BOOTERIMAGES)
  		if (filename.toLowerCase().endsWith(ext)) return true;
  	return false;
  }
  
  public static boolean isConfFile(String filename) {
  	return (filename.toLowerCase().endsWith(CONF_EXT));
  }
  
  public static String[] getExecutablesInZip(String archive) throws IOException {
  	List<String> result = new ArrayList<String>();
  	File arcFile = new File(archive);
		
  	if (archive.toLowerCase().endsWith(ARCHIVES[0])) { // zip
  		ZipFile zf = new ZipFile(arcFile);
  		for (Enumeration<? extends ZipEntry> entries = zf.entries(); entries.hasMoreElements();) {
  			ZipEntry entry = entries.nextElement();
  			String name = entry.getName();
  			if (!entry.isDirectory() && isExecutable(name))
  				result.add(PlatformUtils.archiveToNativePath(name));
  		}
  	} else if (archive.toLowerCase().endsWith(ARCHIVES[1])) { // 7-zip
  		MyRandomAccessFile istream = new MyRandomAccessFile(archive, "r");
      IInArchive zf = new Handler();
      if (zf.Open(istream) != 0)
          throw new IOException("Failure opening 7-zip archive");
      for(int i = 0; i < zf.size() ; i++) {
        SevenZipEntry entry = zf.getEntry(i);
  			String name = entry.getName();
  			if (!entry.isDirectory() && isExecutable(name))
  				result.add(PlatformUtils.archiveToNativePath(name));
  		}
  	}
		
  	Collections.sort(result,
  			new Comparator<String>() {
	  			public int compare(String s1, String s2) {
	  				int count1 = StringUtils.countCharacters(s1, '\\');
	  				int count2 = StringUtils.countCharacters(s2, '\\');
	  				if (count1 != count2)
	  					return count1 - count2;
	  				else
	  					return s1.compareTo(s2);
	  			}
	  		} );
  	
		return result.toArray(new String[0]);
  }
}
