package com.gui;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import swing2swt.layout.BorderLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import com.connect.Messaging;
import com.db.Database;
import com.model.DosboxVersion;
import com.model.OrderingVector;
import com.model.Profile;
import com.model.Settings;
import com.model.Template;
import com.swtdesigner.SWTImageManager;
import com.util.FileUtils;
import com.util.PlatformUtils;
import com.util.StringUtils;


public final class MainWindow {

	public static final String PROGRAM_NAME = "DOSBox Game Launcher v0.62";
  public static String[] COLUMNNAMES;
  public static final int NON_EDITABLE_COLUMN_NAMES = 10;
	public static final int EDITABLE_COLUMN_NAMES = 10;
  
  private Display display;
  private Shell shell;
  private OrderingVector orderingVector = null;
  private java.util.List<Profile> profilesList;
  private java.util.List<DosboxVersion> dbversionsList;
  private java.util.List<Template> templatesList;
  private Database db = null;
  private Settings settings = null;
  private File currentThumbFile = null;
  private int thumbHeight;
  
  private int[] columnIds;
  private Table template_table;
  private Table dosboxversion_table;
  private Table profile_table;
  private Text notesField;
  private ToolBar thumbsToolBar;
  private Link[] link;
  private Menu runWithSubMenu;
  private Menu setupWithSubMenu;
  
	
  public static void main(String[] args) {
    try {
      MainWindow window = new MainWindow();
      window.open();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  public void open() {
    db = Database.getInstance();
    settings = Settings.getInstance();

    display = Display.getDefault();
    createContents();
    
    Messaging mess = null;
    if (settings.getBooleanValue("communication", "port_enabled")) {
    	mess = new Messaging(settings.getIntValue("communication", "port"), this);
    	mess.start();
    }
    
    shell.open();
    shell.layout();
    while (!shell.isDisposed()) {
      if (!display.readAndDispatch())
        display.sleep();
    }
    SWTImageManager.dispose();
    display.dispose();

    if (mess != null)
    	mess.close();

    try {
			settings.save();
		} catch (IOException e) {
			GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
		}
    try { db.shutdown(); } catch (SQLException e) {}
  }
  
  private void createContents() {
  	initColumnIds();
    orderingVector = new OrderingVector(settings.getIntValues("gui", "sortcolumn"),
    		settings.getBooleanValues("gui", "sortascending"));
    
  	shell = new Shell();
    try {
			profilesList = db.readProfilesList(orderingVector.toClause());
			dbversionsList = db.readDosboxVersionsList();
			templatesList = db.readTemplatesList();
    } catch (SQLException e) {
    	GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
		}
    shell.addControlListener(new ControlAdapter() {
    	public void controlResized(final ControlEvent e) {
    		boolean isMaximized = shell.getMaximized();
    		if (!isMaximized) {
    			Rectangle rec = shell.getBounds();
        		settings.setIntValue("gui", "width", rec.width);
        		settings.setIntValue("gui", "height", rec.height);
        		settings.setIntValue("gui", "x", rec.x);
        		settings.setIntValue("gui", "y", rec.y);
    		}
    		settings.setBooleanValue("gui", "maximized", isMaximized);
    	} });
    shell.addControlListener(new ControlAdapter() {
    	public void controlMoved(final ControlEvent e) {
    		if (!shell.getMaximized()) {
    			Rectangle rec = shell.getBounds();
    			settings.setIntValue("gui", "x", rec.x);
    			settings.setIntValue("gui", "y", rec.y);
    		}
    	} });
    shell.setLayout(new BorderLayout(0, 0));
    shell.setMaximized(settings.getBooleanValue("gui", "maximized"));
    shell.setLocation(settings.getIntValue("gui", "x"), settings.getIntValue("gui", "y"));
    shell.setSize(settings.getIntValue("gui", "width"), settings.getIntValue("gui", "height"));
    shell.setText(PROGRAM_NAME);

    final Menu menu = new Menu(shell, SWT.BAR);
    shell.setMenuBar(menu);

    final MenuItem fileMenuItem = new MenuItem(menu, SWT.CASCADE);
    fileMenuItem.setText("File");

    final Menu menu_1 = new Menu(fileMenuItem);
    fileMenuItem.setMenu(menu_1);

    if (PlatformUtils.IS_WINDOWS) {
	    final MenuItem dfendImportMenuItem = new MenuItem(menu_1, SWT.NONE);
	    dfendImportMenuItem.addSelectionListener(new SelectionAdapter() {
	      public void widgetSelected(final SelectionEvent e) {
	        doDFendImport();
	      }
	    });
	    dfendImportMenuItem.setText("Import D-Fend Profiles");
	
	    new MenuItem(menu_1, SWT.SEPARATOR);
    }
    
    final MenuItem migrateMenuItem = new MenuItem(menu_1, SWT.NONE);
    migrateMenuItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doMigrate();
      }
    });
    migrateMenuItem.setText("Migrate Profiles");
    
    final MenuItem confImportMenuItem = new MenuItem(menu_1, SWT.NONE);
    confImportMenuItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doImportConfigfile();
      }
    });
    confImportMenuItem.setText("Import Profile");
    new MenuItem(menu_1, SWT.SEPARATOR);
    
    final MenuItem settingsMenuItem = new MenuItem(menu_1, SWT.NONE);
    settingsMenuItem.setText("Adjust Settings");
    settingsMenuItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doOpenSettingsDialog();
      }
    });

    final MenuItem exitMenuItem = new MenuItem(menu_1, SWT.NONE);
    exitMenuItem.setText("Exit");
    exitMenuItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        shell.close();
      }
    });
    
    final TabFolder tabFolder = new TabFolder(shell, SWT.NONE);

    final TabItem profilesTabItem = new TabItem(tabFolder, SWT.NONE);
    profilesTabItem.setText("Profiles");

    final Composite composite = new Composite(tabFolder, SWT.NONE);
    composite.setLayout(new BorderLayout(0, 0));
    profilesTabItem.setControl(composite);
    
    final ToolBar toolBar = new ToolBar(composite, SWT.NONE);
    toolBar.setLayoutData(BorderLayout.NORTH);

    final ToolItem newItemToolItem = new ToolItem(toolBar, SWT.PUSH);
    newItemToolItem.setText("Add profile");
    newItemToolItem.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doAddProfile();
      }
    });
    
    final ToolItem newItemToolItem_2 = new ToolItem(toolBar, SWT.PUSH);
    newItemToolItem_2.setText("Edit Profile");
    newItemToolItem_2.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doEditProfile();
      }
    });
    
    final ToolItem newItemToolItem_1 = new ToolItem(toolBar, SWT.PUSH);
    newItemToolItem_1.setText("Remove profile");
    newItemToolItem_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doRemoveProfile();
      }
    });

    final SashForm sashForm = new SashForm(composite, SWT.NONE);
    
    profile_table = new Table(sashForm, SWT.FULL_SELECTION | SWT.BORDER);
    
    final Composite informationGroup = new Composite(sashForm, SWT.NONE);
    informationGroup.addControlListener(new ControlAdapter() {
    	public void controlResized(final ControlEvent e) {
    		settings.setIntValues("gui", "sashweights", sashForm.getWeights());
    	}
    });
    informationGroup.setLayout(new BorderLayout(0, 0));
    notesField = new Text(informationGroup, SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER | SWT.WRAP);

    final Composite composite_2 = new Composite(informationGroup, SWT.NONE);
    final GridLayout gridLayout = new GridLayout();
    gridLayout.marginWidth = 0;
    gridLayout.marginHeight = 2;
    gridLayout.verticalSpacing = 1;
    composite_2.setLayout(gridLayout);
    composite_2.setLayoutData(BorderLayout.SOUTH);

    link = new Link[4];
    for (int i = 0; i < link.length; i++) {
	    link[i] = new Link(composite_2, SWT.NONE);
	    link[i].setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
	    link[i].setText(" ");
	    link[i].addSelectionListener(new SelectionAdapter() {
	      public void widgetSelected(final SelectionEvent e) {
	        PlatformUtils.openForBrowsing(e.text);
	      }
	    });
    }

    thumbHeight = settings.getIntValue("gui", "screenshotsheight");
    final ScrolledComposite scrolledComposite = new ScrolledComposite(composite, SWT.BORDER | SWT.H_SCROLL);
    scrolledComposite.setMinHeight(thumbHeight + 20);
    scrolledComposite.setLayoutData(BorderLayout.SOUTH);

    thumbsToolBar = new ToolBar(scrolledComposite, SWT.NONE);
    thumbsToolBar.addMouseListener(new MouseAdapter() {
    	public void mouseDown(final MouseEvent e) {
    		currentThumbFile = new File(thumbsToolBar.getItem(new Point(e.x, e.y)).getToolTipText());
    	}
    });
    scrolledComposite.setContent(thumbsToolBar);
    
    final Menu menu_thumb = new Menu(thumbsToolBar);
    thumbsToolBar.setMenu(menu_thumb);

    final MenuItem thumbMenuItem1 = new MenuItem(menu_thumb, SWT.NONE);
    thumbMenuItem1.setText("Remove");
    thumbMenuItem1.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRemoveThumb();
    	}
    });
    new MenuItem(menu_thumb, SWT.SEPARATOR);
    
    final MenuItem thumbMenuItem2 = new MenuItem(menu_thumb, SWT.NONE);
    thumbMenuItem2.setText("Open folder");
    thumbMenuItem2.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		PlatformUtils.openDirForViewing(currentThumbFile.getParentFile());
    	}
    });
    
    final MenuItem thumbMenuItem3 = new MenuItem(menu_thumb, SWT.NONE);
    thumbMenuItem3.setText("Refresh");
    thumbMenuItem3.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		SWTImageManager.dispose();
    		displayProfileInformation();
    	}
    });

    profile_table.addKeyListener(new KeyAdapter() {
    	public void keyPressed(final KeyEvent e) {
    		if (e.keyCode == SWT.DEL) {
        	doRemoveProfile();
        } else if (e.keyCode == SWT.INSERT) {
        	doAddProfile();
        } else if (e.keyCode == SWT.F2) {
        	doEditProfile(true);
        } else if (e.stateMask == SWT.CTRL && (Character.toLowerCase(e.keyCode) == 'm')) {
        	doToggleFavoriteProfile();
        } else if (e.stateMask == SWT.CTRL && (Character.toLowerCase(e.keyCode) == 'd')) {
        	doDuplicateProfile();
        }
    	}
    });

    profile_table.setLayoutData(BorderLayout.WEST);
    profile_table.addTraverseListener(new TraverseListener() {
      public void keyTraversed(final TraverseEvent e) {
        if ((e.stateMask == SWT.CTRL) && (e.detail == SWT.TRAVERSE_RETURN)) {
          doEditProfile();
        } else if ((e.stateMask == SWT.SHIFT) && (e.detail == SWT.TRAVERSE_RETURN)) {
          doRunProfile(true);
        } else if (e.detail == SWT.TRAVERSE_RETURN) {
          doRunProfile(false);
        } 
      }
    });
    profile_table.addMouseListener(new MouseAdapter() {
      public void mouseDoubleClick(final MouseEvent e) {
        doRunProfile(false);
      }
    });

    profile_table.setLinesVisible(true);
    profile_table.setHeaderVisible(true);
    
    for (int i = 0; i < columnIds.length; i++)
    	addProfileColumn(COLUMNNAMES[columnIds[i]], i);

    profile_table.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        displayProfileInformation();
      } });

    final Menu menu_2 = new Menu(profile_table);
    profile_table.setMenu(menu_2);

    final MenuItem runMenuItem = new MenuItem(menu_2, SWT.NONE);
    runMenuItem.setText("Run");
    runMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRunProfile(false);
    	}
    });

    final MenuItem runWithMenuItem = new MenuItem(menu_2, SWT.CASCADE);
    runWithMenuItem.setText("Run with");

    runWithSubMenu = new Menu(runWithMenuItem);
    runWithMenuItem.setMenu(runWithSubMenu);
    updateProfileDbversionsSubmenu();

    final MenuItem setupMenuItem = new MenuItem(menu_2, SWT.NONE);
    setupMenuItem.setText("Setup");
    setupMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRunProfile(true);
    	}
    });
    
    final MenuItem setupWithMenuItem = new MenuItem(menu_2, SWT.CASCADE);
    setupWithMenuItem.setText("Setup with");

    setupWithSubMenu = new Menu(setupWithMenuItem);
    setupWithMenuItem.setMenu(setupWithSubMenu);
    updateSetupDbversionsSubmenu();

    new MenuItem(menu_2, SWT.SEPARATOR);

    final MenuItem addMenuItem = new MenuItem(menu_2, SWT.NONE);
    addMenuItem.setText("Add");
    addMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doAddProfile();
    	}
    });
    
    final MenuItem editMenuItem = new MenuItem(menu_2, SWT.NONE);
    editMenuItem.setText("Edit");
    editMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doEditProfile();
    	}
    });

    final MenuItem duplicateMenuItem = new MenuItem(menu_2, SWT.NONE);
    duplicateMenuItem.setText("Duplicate");
    duplicateMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doDuplicateProfile();
    	}
    });

    final MenuItem removeMenuItem = new MenuItem(menu_2, SWT.NONE);
    removeMenuItem.setText("Remove");
    removeMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRemoveProfile();
    	}
    });
    
    new MenuItem(menu_2, SWT.SEPARATOR);
    
    final MenuItem toggleMenuItem = new MenuItem(menu_2, SWT.NONE);
    toggleMenuItem.setText("Toggle favorite");
    toggleMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doToggleFavoriteProfile();
    	}
    });
    
    if (PlatformUtils.IS_WINDOWS) {
  	  new MenuItem(menu_2, SWT.SEPARATOR);
	    final MenuItem shortcutMenuItem = new MenuItem(menu_2, SWT.NONE);
	    shortcutMenuItem.setText("Create shortcut");
	    shortcutMenuItem.addSelectionListener(new SelectionAdapter() {
	    	public void widgetSelected(final SelectionEvent e) {
	    		doCreateShortcut();
	    	}
	    });
    }
    
    menu_2.addMenuListener(new MenuAdapter() {
    	public void menuShown(final MenuEvent e) {
    		if (profile_table.getSelectionIndex() != -1) {
    			boolean hasSetup = profilesList.get(profile_table.getSelectionIndex()).hasSetup();
    			setupMenuItem.setEnabled(hasSetup);
    			setupWithMenuItem.setEnabled(hasSetup);
    		}
    	}
    });
    
    sashForm.setWeights(settings.getIntValues("gui", "sashweights"));
    
    final TabItem dosboxTabItem = new TabItem(tabFolder, SWT.NONE);
    dosboxTabItem.setText("DOSBox versions");

    final Composite composite_1 = new Composite(tabFolder, SWT.NONE);
    composite_1.setLayout(new BorderLayout(0, 0));
    dosboxTabItem.setControl(composite_1);
    
    final ToolBar toolBar_1 = new ToolBar(composite_1, SWT.NONE);
    toolBar_1.setLayoutData(BorderLayout.NORTH);

    final ToolItem newItemToolItem_3 = new ToolItem(toolBar_1, SWT.PUSH);
    newItemToolItem_3.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doAddDosboxVersion();
      }
    });
    newItemToolItem_3.setText("Add version");

    final ToolItem newItemToolItem_2_1 = new ToolItem(toolBar_1, SWT.PUSH);
    newItemToolItem_2_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doEditDosboxVersion();
      }
    });
    newItemToolItem_2_1.setText("Edit version");

    final ToolItem newItemToolItem_1_1 = new ToolItem(toolBar_1, SWT.PUSH);
    newItemToolItem_1_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doRemoveDosboxVersion();
      }
    });
    newItemToolItem_1_1.setText("Remove version");

    dosboxversion_table = new Table(composite_1, SWT.FULL_SELECTION | SWT.BORDER);
    dosboxversion_table.addKeyListener(new KeyAdapter() {
    	public void keyPressed(final KeyEvent e) {
    		if (e.keyCode == SWT.DEL) {
        	doRemoveDosboxVersion();
        } else if (e.keyCode == SWT.INSERT) {
        	doAddDosboxVersion();
        } else if (e.stateMask == SWT.CTRL && (Character.toLowerCase(e.keyCode) == 'm')) {
        	doToggleDefaultVersion();
        }
    	}
    });
    dosboxversion_table.addTraverseListener(new TraverseListener() {
      public void keyTraversed(final TraverseEvent e) {
      	if ((e.stateMask == SWT.CTRL) && (e.detail == SWT.TRAVERSE_RETURN))
          doEditDosboxVersion();
      	else if (e.detail == SWT.TRAVERSE_RETURN)
          doRunDosbox();
      }
    });
    dosboxversion_table.addMouseListener(new MouseAdapter() {
      public void mouseDoubleClick(final MouseEvent e) {
        doRunDosbox();
      }
    });
    
    dosboxversion_table.setLinesVisible(true);
    dosboxversion_table.setHeaderVisible(true);

    addDBColumn("Title", 0);
    addDBColumn("Version", 1);
    addDBColumn("Path", 2);
    addDBColumn("Default", 3);
    addDBColumn("ID", 4);
    
    final Menu menu_4 = new Menu(dosboxversion_table);
    dosboxversion_table.setMenu(menu_4);

    final MenuItem runDBMenuItem = new MenuItem(menu_4, SWT.NONE);
    runDBMenuItem.setText("Run");
    runDBMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRunDosbox();
    	}
    });

    new MenuItem(menu_4, SWT.SEPARATOR);

    final MenuItem addDBMenuItem = new MenuItem(menu_4, SWT.NONE);
    addDBMenuItem.setText("Add");
    addDBMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doAddDosboxVersion();
    	}
    });
    
    final MenuItem editDBMenuItem = new MenuItem(menu_4, SWT.NONE);
    editDBMenuItem.setText("Edit");
    editDBMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doEditDosboxVersion();
    	}
    });

    final MenuItem removeDBMenuItem = new MenuItem(menu_4, SWT.NONE);
    removeDBMenuItem.setText("Remove");
    removeDBMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRemoveDosboxVersion();
    	}
    });
    
    new MenuItem(menu_4, SWT.SEPARATOR);
    
    final MenuItem toggleDefaultMenuItem = new MenuItem(menu_4, SWT.NONE);
    toggleDefaultMenuItem.setText("Toggle default");
    toggleDefaultMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doToggleDefaultVersion();
    	}
    });

    new MenuItem(menu_4, SWT.SEPARATOR);
    
    final MenuItem openFolderMenuItem = new MenuItem(menu_4, SWT.NONE);
    openFolderMenuItem.setText("Open folder");
    openFolderMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		int index = dosboxversion_table.getSelectionIndex();
        if (index != -1)
        	PlatformUtils.openDirForViewing(dbversionsList.get(index).getCanonicalConfFile().getParentFile());
    	}
    });
    
    final TabItem templatesTabItem = new TabItem(tabFolder, SWT.NONE);
    templatesTabItem.setText("Templates");

    final Composite composite_1_1 = new Composite(tabFolder, SWT.NONE);
    composite_1_1.setLayout(new BorderLayout(0, 0));
    templatesTabItem.setControl(composite_1_1);
    
    final ToolBar toolBar_1_1 = new ToolBar(composite_1_1, SWT.NONE);
    toolBar_1_1.setLayoutData(BorderLayout.NORTH);

    final ToolItem newItemToolItem_3_1 = new ToolItem(toolBar_1_1, SWT.PUSH);
    newItemToolItem_3_1.setText("Add template");
    newItemToolItem_3_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doAddTemplate();
      }
    });

    final ToolItem newItemToolItem_2_1_1 = new ToolItem(toolBar_1_1, SWT.PUSH);
    newItemToolItem_2_1_1.setText("Edit template");
    newItemToolItem_2_1_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doEditTemplate();
      }
    });

    final ToolItem newItemToolItem_1_1_1 = new ToolItem(toolBar_1_1, SWT.PUSH);
    newItemToolItem_1_1_1.setText("Remove template");
    newItemToolItem_1_1_1.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(final SelectionEvent e) {
        doRemoveTemplate();
      }
    });

    template_table = new Table(composite_1_1, SWT.FULL_SELECTION | SWT.BORDER);
    template_table.addKeyListener(new KeyAdapter() {
    	public void keyPressed(final KeyEvent e) {
    		if (e.keyCode == SWT.DEL) {
        	doRemoveTemplate();
        } else if (e.keyCode == SWT.INSERT) {
        	doAddTemplate();
        } else if (e.stateMask == SWT.CTRL && (Character.toLowerCase(e.keyCode) == 'm')) {
        	doToggleDefaultTemplate();
        }
    	}
    });
    template_table.addTraverseListener(new TraverseListener() {
      public void keyTraversed(final TraverseEvent e) {
      	if ((e.stateMask == SWT.CTRL) && (e.detail == SWT.TRAVERSE_RETURN))
          doEditTemplate();
      }
    });
    
    template_table.setLinesVisible(true);
    template_table.setHeaderVisible(true);
    
    addTemplateColumn("Title", 0);
    addTemplateColumn("Default", 1);
    
    final Menu menu_3 = new Menu(template_table);
    template_table.setMenu(menu_3);

    final MenuItem addTemplateMenuItem = new MenuItem(menu_3, SWT.NONE);
    addTemplateMenuItem.setText("Add");
    addTemplateMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doAddTemplate();
    	}
    });
    
    final MenuItem editTemplateMenuItem = new MenuItem(menu_3, SWT.NONE);
    editTemplateMenuItem.setText("Edit");
    editTemplateMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doEditTemplate();
    	}
    });

    final MenuItem removeTemplateMenuItem = new MenuItem(menu_3, SWT.NONE);
    removeTemplateMenuItem.setText("Remove");
    removeTemplateMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doRemoveTemplate();
    	}
    });
    
    new MenuItem(menu_3, SWT.SEPARATOR);
    
    final MenuItem toggleTemplateMenuItem = new MenuItem(menu_3, SWT.NONE);
    toggleTemplateMenuItem.setText("Toggle default");
    toggleTemplateMenuItem.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doToggleDefaultTemplate();
    	}
    });
    
    // init values
    for (Profile prof: profilesList) addProfileToTable(prof);
    for (DosboxVersion dbversion: dbversionsList) addDosboxVersionToTable(dbversion);
    for (Template template: templatesList) addTemplateToTable(template);
    
    profile_table.setSortColumn(profile_table.getColumn(findColumnById(settings.getIntValues("gui", "sortcolumn")[0])));
    profile_table.setSortDirection(settings.getBooleanValues("gui", "sortascending")[0]? SWT.UP: SWT.DOWN);
    profile_table.setColumnOrder(settings.getIntValues("gui", "columnorder"));
    profile_table.setSelection(settings.getIntValue("gui", "selectedprofile"));
    profile_table.setFocus();
    displayProfileInformation();
  }
  
  private void doReorderProfiles(int columnId, SelectionEvent e) {
	  TableColumn sortColumn = profile_table.getSortColumn();
	  TableColumn currentColumn = (TableColumn) e.widget;
	  int dir = profile_table.getSortDirection();
	  if (sortColumn == currentColumn) {
		  dir = dir == SWT.UP ? SWT.DOWN : SWT.UP;
	  } else {
		  profile_table.setSortColumn(currentColumn);
		  dir = SWT.UP;
	  }
	  int currentProfileId = -1;
	  if (profile_table.getSelectionIndex() != -1)
		  currentProfileId = profilesList.get(profile_table.getSelectionIndex()).getId();
	  try {
	  	orderingVector.addOrdering(columnIds[columnId], dir == SWT.UP);
			profilesList = db.readProfilesList(orderingVector.toClause());
		} catch (SQLException ex) {
			GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
		}
	  for (int i = 0; i < profilesList.size(); i++)
		  setTableItem(profile_table.getItem(i), profilesList.get(i));
	  profile_table.setSortDirection(dir);
	  if (currentProfileId != -1) {
		  profile_table.setSelection(Profile.findById(profilesList, currentProfileId));
	  }
	  settings.setIntValues("gui", "sortcolumn", orderingVector.getColumns());
	  settings.setBooleanValues("gui", "sortascending", orderingVector.getAscendings());
  }
  
  private void displayNotes(String notes) {
    notesField.setText(notes);
  }
  
  private void displayLinks(String[] p_link) {
    for (int i = 0; i < link.length; i++) {
    	String l = p_link[i];
    	String t = p_link[i];
	  	if (l != null && !l.equals("")) {
	  		if (!l.contains("://")) {
	  			l = "file://" + FileUtils.canonicalToData(l).getPath();
	  			t = FileUtils.makeRelativeToDosroot(FileUtils.canonicalToData(t)).getPath();
	  		}
	      link[i].setText("<a href=\"" + l + "\">" + t + "</a>");
	      link[i].setToolTipText(l);
	    } else {
	    	link[i].setText(" ");
	    	link[i].setToolTipText("");
	    }
    }
  }
  
  private void displayScreenshots(Profile prof) {
    ToolItem[] toolItems = thumbsToolBar.getItems();
    for (int i = 0; i < toolItems.length; i++)
      toolItems[i].dispose();
    if (prof != null) {
	    File path = prof.getCanonicalCaptures();
	    File[] files = path.listFiles();
	    if (files != null) {
	      for (File file: files) {
	        if (file.getName().toLowerCase().endsWith(".png")) {
	          final ToolItem toolItem = new ToolItem(thumbsToolBar, SWT.PUSH);
	          toolItem.setToolTipText(file.getPath());
	          toolItem.setText(file.getName());
	          toolItem.setImage(SWTImageManager.getResizedImage(display, thumbHeight, file.getPath()));
	          toolItem.addSelectionListener(new SelectionAdapter() {
	            public void widgetSelected(final SelectionEvent e) {
	              final Thumb thumbDialog = new Thumb(shell);
	              thumbDialog.setThumb(toolItem.getToolTipText());
	              thumbDialog.open();
	            }
	          });
	        }
	      }
	    }
    }
    thumbsToolBar.setVisible(thumbsToolBar.getItemCount() != 0);
    thumbsToolBar.pack();
  }
  
  private void doAddProfile() {
    doAddProfile(null);
  }
  
  private void doAddProfile(String file) {
    if (checkDefaultDBVersion() == null)
      return;
    final EditProfileDialog addProfileDialog = new EditProfileDialog(shell);
    if (file != null)
    	addProfileDialog.sendToProfile(file);
    updateWithAddedProfile((Profile)addProfileDialog.open());
  }
  
  public void addProfile(final String file) {
  	display.syncExec(new Runnable() {
  		public void run() {
  			doAddProfile(file);
  		}
    });
  }
  
  private void doAddDosboxVersion() {
    final EditDosboxVersionDialog addDosboxVersionDialog = new EditDosboxVersionDialog(shell);
    DosboxVersion dbversion = (DosboxVersion)addDosboxVersionDialog.open();
    if (dbversion != null)
    	updateDosboxVersionList(dbversion);
  }
  
  private void doAddTemplate() {
  	if (checkDefaultDBVersion() == null)
      return;
    final EditTemplateDialog addTemplateDialog = new EditTemplateDialog(shell);
    Template template = (Template)addTemplateDialog.open();
    if (template != null)
    	updateTemplateList(template);
  }
  
  private void doToggleFavoriteProfile() {
    int index = profile_table.getSelectionIndex();
    if (index != -1) {
    	Profile profile = profilesList.get(index);
    	try {
				db.toggleFavorite(profile.getId());
			} catch (SQLException e) {
				GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
			}
    	profile.toggleDefault();
    	profilesList.set(index, profile);
    	setTableItem(profile_table.getItem(index), profile);
    }
  }
  
  private void doToggleDefaultVersion() {
  	int index = dosboxversion_table.getSelectionIndex();
    if (index != -1) {
    	DosboxVersion ver = dbversionsList.get(index);
    	ver.toggleDefault();
    	try {
    	    db.addOrEditDosboxVersion(ver.getTitle(), ver.getPath(), ver.isMultiConfig(), ver.isDefault(),
                    ver.getParameters(), ver.getVersion(), ver.getId());
			} catch (SQLException e) {
				GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
			}
    	updateDosboxVersionList(ver);
    }
  }
  
  private void doToggleDefaultTemplate() {
  	int index = template_table.getSelectionIndex();
    if (index != -1) {
    	Template temp = templatesList.get(index);
    	temp.toggleDefault();
    	try {
				db.addOrEditTemplate(temp.getTitle(), temp.getdbversionId(), temp.isDefault(), temp.getId());
			} catch (SQLException e) {
				GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
			}
    	updateTemplateList(temp);
    }
  }
  
  private void doEditProfile() {
  	doEditProfile(false);
  }
  
  private void doEditProfile(boolean focusTitle) {
    int index = profile_table.getSelectionIndex();
    if (index != -1) {
      final EditProfileDialog editProfileDialog = new EditProfileDialog(shell);
      editProfileDialog.setProfile(profilesList.get(index));
      if (focusTitle)
      	editProfileDialog.focusTitle();
      Profile profile = (Profile)editProfileDialog.open();
      if (profile != null) {
      	boolean quickUpdate = true;
      	if (settings.getBooleanValue("gui", "autosortonupdate")) {
      		try {
      			profilesList = db.readProfilesList(orderingVector.toClause());
      			if (index != Profile.findById(profilesList, profile.getId()))
      				quickUpdate = false;
      		} catch (SQLException e) {
      			GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
      		}
      	}
      	if (quickUpdate) {
      		profilesList.set(index, profile);
      		setTableItem(profile_table.getItem(index), profile);
    		} else {
    			updateProfilesList(profile);
    		}
      	displayProfileInformation();
      }
    }
  }
  
  private void doDuplicateProfile() {
  	int index = profile_table.getSelectionIndex();
    if (index != -1) {
    	Profile orgProf = profilesList.get(index);
    	updateWithAddedProfile(EditProfileDialog.duplicateProfile(orgProf, db, shell));
    }
  }
  
  private void updateWithAddedProfile(Profile profile) {
  	if (profile != null) {
  		if (settings.getBooleanValue("gui", "autosortonupdate")) {
  			updateProfilesList(profile);
  		} else {
	  		profilesList.add(profile);
	      addProfileToTable(profile);
	      profile_table.setSelection(profile_table.getItemCount() - 1);
	      profile_table.setFocus();
  		}
  		displayProfileInformation();
  	}
  }
  
  private void doEditDosboxVersion() {
    int index = dosboxversion_table.getSelectionIndex();
    if (index != -1) {
      final EditDosboxVersionDialog editDosboxVersionDialog = new EditDosboxVersionDialog(shell);
      editDosboxVersionDialog.setDosboxVersion(dbversionsList.get(index));
      DosboxVersion dbversion = (DosboxVersion)editDosboxVersionDialog.open();
      if (dbversion != null)
      	updateDosboxVersionList(dbversion);
    }
  }
  
  private void updateProfilesList(Profile profile) {
  	try {
			profilesList = db.readProfilesList(orderingVector.toClause());
		} catch (SQLException e) {
			GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
		}
		profile_table.setRedraw(false);
    profile_table.removeAll();
    for (Profile prof: profilesList) addProfileToTable(prof);
    if (profile != null)
    	profile_table.setSelection(Profile.findById(profilesList, profile.getId()));
    profile_table.setRedraw(true);
    profile_table.setFocus();
  }
  
  private void updateDosboxVersionList(DosboxVersion dbversion) {
  	try {
			dbversionsList = db.readDosboxVersionsList();
		} catch (SQLException e) {
			GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
		}
    dosboxversion_table.removeAll();
    for (DosboxVersion version: dbversionsList) addDosboxVersionToTable(version);
    dosboxversion_table.setSelection(DosboxVersion.findById(dbversionsList, dbversion.getId()));
	  dosboxversion_table.setFocus();
	  
	  updateProfileDbversionsSubmenu();
	  updateSetupDbversionsSubmenu();
  }
  
  private void updateTemplateList(Template template) {
  	try {
			templatesList = db.readTemplatesList();
		} catch (SQLException e) {
			GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
		}
    template_table.removeAll();
    for (Template temp: templatesList) addTemplateToTable(temp);
    template_table.setSelection(Template.findById(templatesList, template.getId()));
	  template_table.setFocus();
  }
  
  private void doEditTemplate() {
    int index = template_table.getSelectionIndex();
    if (index != -1) {
      final EditTemplateDialog editTemplateDialog = new EditTemplateDialog(shell);
      editTemplateDialog.setTemplate(templatesList.get(index));
      Template template = (Template)editTemplateDialog.open();
      if (template != null)
      	updateTemplateList(template);
    }
  }
  
  private void doRemoveProfile() {
    int index = profile_table.getSelectionIndex();
    if (index != -1) {
      if (GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected profile?")) {
        Profile prof = profilesList.get(index);
        try {
					db.removeProfile(prof.getId());
					profile_table.remove(index);
	        profilesList.remove(index);
	        displayProfileInformation();
	        SWTImageManager.dispose();
	        if (GeneralPurposeDialogs.confirmMessage(shell,
	        		"Do you also want to remove the associated conf file " + prof.getConfPathAndFile() + " ?")) {
	          FileUtils.removeFile(prof.getCanonicalConfFile());
	        }
	        if (GeneralPurposeDialogs.confirmMessage(shell,
	        		"Do you also want to remove the associated captures directory " + prof.getCaptures() +
	        		" including its files?")) {
	        	FileUtils.removeFilesInDirAndDir(prof.getCanonicalCaptures());
	        }
				} catch (SQLException e) {
					GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
				}
      }
    }
  }

  private void doRemoveDosboxVersion() {
    int index = dosboxversion_table.getSelectionIndex();
    if (index != -1) {
      if (GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected DOSBox version?")) {
        try {
					db.removeDosboxVersion((dbversionsList.get(index)).getId());
					dosboxversion_table.remove(index);
	        dbversionsList.remove(index);
	        updateProfileDbversionsSubmenu();
	        updateSetupDbversionsSubmenu();
				} catch (SQLException e) {
					GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
				}
      }
    }
  }
  
  private void doRemoveTemplate() {
    int index = template_table.getSelectionIndex();
    if (index != -1) {
      if (GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected template?")) {
        int templateId = (templatesList.get(index)).getId();
        try {
					db.removeTemplate(templateId);
					template_table.remove(index);
	        templatesList.remove(index);
	        File conffile = FileUtils.constructCanonicalTemplateFileLocation(templateId);
	        if (GeneralPurposeDialogs.confirmMessage(shell,
	        		"Do you also want to remove the associated conf file " + conffile + " ?")) {
	        	FileUtils.removeFile(conffile);
	        }
				} catch (SQLException e) {
					GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
				}
      }
    }
  }
  
  private void addProfileToTable(Profile prof) {
    final TableItem newItemTableItem = new TableItem(profile_table, SWT.BORDER); 
    setTableItem(newItemTableItem, prof);
  }
  
  private void addDosboxVersionToTable(DosboxVersion dbversion) {
    final TableItem newItemTableItem = new TableItem(dosboxversion_table, SWT.BORDER); 
    setTableItem(newItemTableItem, dbversion);
  }
  
  private void addTemplateToTable(Template template) {
    final TableItem newItemTableItem = new TableItem(template_table, SWT.BORDER); 
    setTableItem(newItemTableItem, template);
  }
  
  private void setTableItem(TableItem newItemTableItem, Profile prof) {
  	for (int i = 0; i < columnIds.length; i++) {
  		String value;
  		switch (columnIds[i]) {
	  		case 0: value = prof.getTitle(); break;
	  		case 1: value = prof.hasSetupString(); break;
	  		case 2: value = prof.getDeveloperName(); break;
	  		case 3: value = prof.getPublisherName(); break;
	  		case 4: value = prof.getGenre(); break;
	  		case 5: value = prof.getYear(); break;
	  		case 6: value = prof.getStatus(); break;
	  		case 7: value = prof.isDefaultString(); break;
	  		case 8: value = String.valueOf(prof.getId()); break;
	  		case 9: value = String.valueOf(prof.getDbversionId()); break;
	  		case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
	  						value = prof.getCustomString(columnIds[i] - NON_EDITABLE_COLUMN_NAMES); break;
	  		case 18:value = prof.getCustomInt(0) + " %"; break;
	  		case 19:value = String.valueOf(prof.getCustomInt(1)); break;
	  		default:value = "";
  		}
  		newItemTableItem.setText(i, value);
  	}
  }
  
  private void setTableItem(TableItem newItemTableItem, DosboxVersion dbversion) {
      newItemTableItem.setText(0, dbversion.getTitle());
      newItemTableItem.setText(1, dbversion.getVersion());
      newItemTableItem.setText(2, dbversion.getPath());
      newItemTableItem.setText(3, dbversion.isDefaultString());
      newItemTableItem.setText(4, String.valueOf(dbversion.getId()));
  }
  
  private void setTableItem(TableItem newItemTableItem, Template template) {
    newItemTableItem.setText(0, template.getTitle());
    newItemTableItem.setText(1, template.isDefaultString());
  }
  
  private void doRunProfile(boolean setup) {
    int index = profile_table.getSelectionIndex();
    if (index != -1) {
    	Profile prof = profilesList.get(index);
    	if (!setup || prof.hasSetup())
				try {
					FileUtils.doRunProfile(prof, dbversionsList, setup);
				} catch (IOException e) {
					GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
				}
    }
  }
  
  private void doRunDosbox() {
    int index = dosboxversion_table.getSelectionIndex();
    if (index != -1)
			try {
				FileUtils.doRunDosbox(dbversionsList.get(index));
			} catch (IOException e) {
				GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
			}
  }
  
  private DosboxVersion checkDefaultDBVersion() {
  	int defaultDosboxVersionIndex = DosboxVersion.findDefault(dbversionsList);
    if (defaultDosboxVersionIndex == -1) {
      GeneralPurposeDialogs.infoMessage(shell, "You must first define a default DOSBox version!");
      return null;
    }
    return dbversionsList.get(defaultDosboxVersionIndex);
  }
  
  private void doDFendImport() {
  	DosboxVersion defaultDosboxVersion = checkDefaultDBVersion();
    if (defaultDosboxVersion == null)
      return;
    
    if (settings.getIntValue("profiledefaults", "confpath") == 1)
    	GeneralPurposeDialogs.infoMessage(shell, "Please realize that you have currently configured " +
    			"DBGL to store profile configuration files in the game directory. " +
    			"Therefore, if you continue and perform the import, all configuration files are " +
    			"saved to the various game directories configured in D-Fend.\n\n" +
    			"If you'd rather have those DOSBox config files stored in a central place, please open " +
    			"the Settings dialog first and change the default configuration-file location to '" +
    			SettingsDialog.confLocations[0] + "'!");

    DFendImportDialog importDialog = new DFendImportDialog(shell);
    importDialog.setDefaultDosboxVersion(defaultDosboxVersion);
    if (importDialog.open() != null) {
    	try {
				profilesList = db.readProfilesList(orderingVector.toClause());
				profile_table.removeAll();
	      for (Profile prof: profilesList) addProfileToTable(prof);
			} catch (SQLException e) {
				GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
			}
    }
  }
  
  private void doMigrate() {
  	GeneralPurposeDialogs.infoMessage(shell, "Please realize that you are about to alter all your profile data.\n\n" +
			"DBGL will change all absolute profile location information inside the .conf files to relative paths. " +
			"Therefore, please make sure that you have made a backup of your DBGL directory!\n\n" +
			"You will have to manually move your games, afterwards.");
  	String from = (String) new MigrateDialog(shell).open();
    if (from != null)
    	GeneralPurposeDialogs.infoMessage(shell, "Now close DBGL, move your game-related files and directories " +
    	"inside " + from + "\n" +
    	"to the new location " + FileUtils.getDosRoot() + "\n" + 
    	"and restart DBGL to complete the migration.");
  }
  
  private void displayProfileInformation() {
    int index = profile_table.getSelectionIndex();
    if (index != -1) {
      settings.setIntValue("gui", "selectedprofile", index);
      Profile prof = profilesList.get(index);
      displayScreenshots(prof);
      displayNotes(prof.getNotes());
      displayLinks(prof.getLinks());
    } else {
      displayScreenshots(null);
      displayNotes("");
      displayLinks(new String[] {null, null, null, null});
    }
  }
  
  private void doRemoveThumb() {
  	if (GeneralPurposeDialogs.confirmMessage(shell, "Would you like to remove the selected " +
  			"screenshot \"" + currentThumbFile + "\"?")) {
  		FileUtils.removeFile(currentThumbFile);
  		displayProfileInformation();
  	}
  	currentThumbFile = null;
  }
  
  private void doCreateShortcut() {
  	int index = profile_table.getSelectionIndex();
    if (index != -1)
      PlatformUtils.createShortcut(profilesList.get(index), dbversionsList);
  }
  
  private void addProfileColumn(String title, final int nr) {
  	final String width = "column" + (columnIds[nr] + 1) + "width";
  	final TableColumn column = new TableColumn(profile_table, SWT.NONE);
  	column.setWidth(settings.getIntValue("gui", width));
  	column.setMoveable(true);
  	column.setText(title);
  	if ((columnIds[nr] == 8) || (columnIds[nr] == 9) ||
  			(columnIds[nr] == 18) || (columnIds[nr] == 19)) // numeric values
  		column.setAlignment(SWT.RIGHT);
  	column.addSelectionListener(new SelectionAdapter() {
    	public void widgetSelected(final SelectionEvent e) {
    		doReorderProfiles(nr, e);
    	}
    });
  	column.addControlListener(new ControlAdapter() {
    	public void controlResized(final ControlEvent e) {
    		settings.setIntValue("gui", width, column.getWidth());
    	}
    	public void controlMoved(final ControlEvent e) {
    		settings.setIntValues("gui", "columnorder", profile_table.getColumnOrder());
    	}
    });
  }
  
  private void addDBColumn(String title, final int nr) {
  	final String width = "column2_" + (nr + 1) + "width";
  	final TableColumn column = new TableColumn(dosboxversion_table, SWT.NONE);
    column.setWidth(settings.getIntValue("gui", width));
    column.setText(title);
    column.addControlListener(new ControlAdapter() {
    	public void controlResized(final ControlEvent e) {
    		settings.setIntValue("gui", width, column.getWidth());
    	}
    });
  }
  
  private void addTemplateColumn(String title, final int nr) {
  	final String width = "column3_" + (nr + 1) + "width";
  	final TableColumn column = new TableColumn(template_table, SWT.NONE);
    column.setWidth(settings.getIntValue("gui", width));
    column.setText(title);
    column.addControlListener(new ControlAdapter() {
    	public void controlResized(final ControlEvent e) {
    		settings.setIntValue("gui", width, column.getWidth());
    	}
    });
  }
  
  private void doOpenSettingsDialog() {
  	SettingsDialog s = new SettingsDialog(shell);
  	if (((Boolean)s.open())) {
  		// first, store current table properties
  		int index = profile_table.getSelectionIndex();
  		int nrColumns = profile_table.getColumnCount();
  		String order = settings.getValue("gui", "columnorder");
  		
  		// then clean the table, reset column order and dispose all columns
  		profile_table.removeAll();
  		StringBuffer sb = new StringBuffer();
  		for (int i = 0; i < nrColumns; i++)
  			sb.append(i + " ");
  		profile_table.setColumnOrder(StringUtils.stringToIntArray(sb.toString()));
  		for (int i = nrColumns - 1; i >= 0; i--)
  			profile_table.getColumn(i).dispose();
  		
  		// next, recreate column ids and add selected columns to table
  		initColumnIds();
  		for (int i = 0; i < columnIds.length; i++)
      	addProfileColumn(COLUMNNAMES[columnIds[i]], i);

  		// redefine sort column in table
  		profile_table.setSortColumn(profile_table.getColumn(findColumnById(settings.getIntValues("gui", "sortcolumn")[0])));
      profile_table.setSortDirection(settings.getBooleanValues("gui", "sortascending")[0]? SWT.UP: SWT.DOWN);
      
      // restore saved columnorder since disposing and adding columns invalidates that
      settings.setValue("gui", "columnorder", order);
      
      // also reset orderingvector
      orderingVector = new OrderingVector(settings.getIntValues("gui", "sortcolumn"),
      		settings.getBooleanValues("gui", "sortascending"));
      
      // finally, refill table with data
  		Profile prof = (index != -1)? profilesList.get(index): null;
  		updateProfilesList(prof);
  	}
  }

  private void doImportConfigfile() {
  	FileDialog dialog = new FileDialog (shell, SWT.OPEN);
    dialog.setFilterNames (new String [] {
    		"DOSBox configuration files", "Executables, Booter Disk Images", FileUtils.ALL_FILTER});
    dialog.setFilterExtensions (new String [] {FileUtils.CNF_FILTER, 
    		FileUtils.EXE_FILTER + ";" + FileUtils.BTR_FILTER, FileUtils.ALL_FILTER});
    String result = dialog.open();
    if (result != null) {
      doAddProfile(result);
    }
  }
  
  private void initColumnIds() {
  	COLUMNNAMES = new String[NON_EDITABLE_COLUMN_NAMES + EDITABLE_COLUMN_NAMES];
  	COLUMNNAMES[0] = "Title";
  	COLUMNNAMES[1] = "Setup";
  	COLUMNNAMES[2] = "Developer";
  	COLUMNNAMES[3] = "Publisher";
  	COLUMNNAMES[4] = "Genre";
  	COLUMNNAMES[5] = "Year";
  	COLUMNNAMES[6] = "Status";
  	COLUMNNAMES[7] = "Favorite";
  	COLUMNNAMES[8] = "ID";
  	COLUMNNAMES[9] = "DB ID";
  	for (int i = 0; i < EDITABLE_COLUMN_NAMES; i++)
  		COLUMNNAMES[i + NON_EDITABLE_COLUMN_NAMES] = settings.getValue("gui", "custom" + (i + 1)); 
  	
  	int amount = 0;
  	for (int i = 0; i < COLUMNNAMES.length; i++)
  		if (settings.getBooleanValue("gui", "column" + (i + 1) + "visible"))
	  		amount++;
  	
  	int cNr = 0;
  	columnIds = new int[amount];
  	for (int i = 0; i < COLUMNNAMES.length; i++)
  		if (settings.getBooleanValue("gui", "column" + (i + 1) + "visible"))
	  		columnIds[cNr++] = i;
  }
  
  private int findColumnById(int id) {
  	for (int i = 0; i < columnIds.length; i++)
  		if (columnIds[i] == id)
	  		return i;
  	return -1;
  }
  
  private void updateProfileDbversionsSubmenu() {
  	MenuItem parent = runWithSubMenu.getParentItem();
  	runWithSubMenu.dispose();
  	runWithSubMenu = new Menu(parent);
  	parent.setMenu(runWithSubMenu);
  	for (int i = 0; i < dbversionsList.size(); i++) {
			final MenuItem runWithMenuItem = new MenuItem(runWithSubMenu, SWT.NONE);
			runWithMenuItem.setText(dbversionsList.get(i).getTitle());
			runWithMenuItem.addSelectionListener(new SelectionAdapter() {
	    	public void widgetSelected(final SelectionEvent e) {
	    		int index = profile_table.getSelectionIndex();
					if (index != -1) {
						Profile prof = profilesList.get(index);
						DosboxVersion dbversion = dbversionsList.get(runWithSubMenu.indexOf(runWithMenuItem));
						try {
							FileUtils.doRunProfile(prof, dbversion, false);
						} catch (IOException exc) {
							GeneralPurposeDialogs.warningMessage(shell, exc.getMessage());
						}
					}
	    	}
	    });
		}
  }
  
  private void updateSetupDbversionsSubmenu() {
  	MenuItem parent = setupWithSubMenu.getParentItem();
  	setupWithSubMenu.dispose();
  	setupWithSubMenu = new Menu(parent);
  	parent.setMenu(setupWithSubMenu);
  	for (int i = 0; i < dbversionsList.size(); i++) {
			final MenuItem setupWithMenuItem = new MenuItem(setupWithSubMenu, SWT.NONE);
			setupWithMenuItem.setText(dbversionsList.get(i).getTitle());
			setupWithMenuItem.addSelectionListener(new SelectionAdapter() {
	    	public void widgetSelected(final SelectionEvent e) {
	    		int index = profile_table.getSelectionIndex();
	    		if (index != -1) {
						Profile prof = profilesList.get(index);
						DosboxVersion dbversion = dbversionsList.get(setupWithSubMenu.indexOf(setupWithMenuItem));
						try {
							FileUtils.doRunProfile(prof, dbversion, true);
						} catch (IOException exc) {
							GeneralPurposeDialogs.warningMessage(shell, exc.getMessage());
						}
	    		}
	    	}
	    });
		}
  }
}
