package com.gui;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import swing2swt.layout.BorderLayout;
import org.eclipse.swt.widgets.*;
import com.connect.Messaging;
import com.db.Database;
import com.model.DosboxVersion;
import com.model.OrderingVector;
import com.model.Profile;
import com.model.Settings;
import com.model.Template;
import com.swtdesigner.SWTImageManager;
import com.util.FileUtils;
import com.util.PlatformUtils;
import com.util.StringUtils;


public final class MainWindow {

    public static final String PROGRAM_NAME = "DOSBox Game Launcher v0.63";
    private static final String REMOVE = "Remove";
    public static final int RO_COLUMN_NAMES = 10;
    public static final int EDIT_COLUMN_NAMES = 10;
    
    public static String[] columnNames;

    private Display display;
    private Shell shell;
    private OrderingVector orderingVector = null;
    private java.util.List<Profile> profilesList;
    private java.util.List<DosboxVersion> dbversionsList;
    private java.util.List<Template> templatesList;
    private Database dbase = null;
    private Settings settings = null;
    private File currentThumbFile = null;
    private int thumbHeight;

    private int[] columnIds;
    private Table template_table;
    private Table dbversion_table;
    private Table profile_table;
    private Text notesField;
    private ToolBar thumbsToolBar;
    private Link[] link;
    private Menu runWithSubMenu;
    private Menu setupWithSubMenu;
  
	
    public static void main(final String[] args) {
        try {
            MainWindow window = new MainWindow();
            window.open();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void open() {
        dbase = Database.getInstance();
        settings = Settings.getInstance();

        display = Display.getDefault();
        createContents();

        Messaging mess = null;
        if (settings.getBooleanValue("communication", "port_enabled")) {
            mess = new Messaging(settings.getIntValue("communication", "port"), this);
            mess.start();
        }

        shell.open();
        shell.layout();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
        SWTImageManager.dispose();
        display.dispose();

        if (mess != null) {
            mess.close();
        }

        try {
            settings.save();
        } catch (IOException e) {
            GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
        }
        try {
            dbase.shutdown();
        } catch (SQLException e) {
            // nothing we can do
        }
    }
  
    private void createContents() {
        initColumnIds();
        orderingVector = new OrderingVector(settings.getIntValues(Settings.GUI, Settings.SORTCOLUMN),
                settings.getBooleanValues(Settings.GUI, Settings.SORTASCENDING));

        shell = new Shell();
        try {
            profilesList = dbase.readProfilesList(orderingVector.toClause());
            dbversionsList = dbase.readDosboxVersionsList();
            templatesList = dbase.readTemplatesList();
        } catch (SQLException e) {
            GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
        }
        shell.addControlListener(new ControlAdapter() {
            public void controlResized(final ControlEvent event) {
                boolean isMaximized = shell.getMaximized();
                if (!isMaximized) {
                    Rectangle rec = shell.getBounds();
                    settings.setIntValue(Settings.GUI, Settings.WIDTH, rec.width);
                    settings.setIntValue(Settings.GUI, "height", rec.height);
                    settings.setIntValue(Settings.GUI, "x", rec.x);
                    settings.setIntValue(Settings.GUI, "y", rec.y);
                }
                settings.setBooleanValue(Settings.GUI, "maximized", isMaximized);
            } });
        shell.addControlListener(new ControlAdapter() {
            public void controlMoved(final ControlEvent event) {
                if (!shell.getMaximized()) {
                    Rectangle rec = shell.getBounds();
                    settings.setIntValue(Settings.GUI, "x", rec.x);
                    settings.setIntValue(Settings.GUI, "y", rec.y);
                }
            } });
        shell.setLayout(new BorderLayout(0, 0));
        shell.setMaximized(settings.getBooleanValue(Settings.GUI, "maximized"));
        shell.setLocation(settings.getIntValue(Settings.GUI, "x"), settings.getIntValue(Settings.GUI, "y"));
        shell.setSize(settings.getIntValue(Settings.GUI, Settings.WIDTH), settings.getIntValue(Settings.GUI, "height"));
        shell.setText(PROGRAM_NAME);

        final Menu menu = new Menu(shell, SWT.BAR);
        shell.setMenuBar(menu);

        final MenuItem fileMenuItem = new MenuItem(menu, SWT.CASCADE);
        fileMenuItem.setText("File");

        final Menu menu_1 = new Menu(fileMenuItem);
        fileMenuItem.setMenu(menu_1);

        if (PlatformUtils.IS_WINDOWS) {
            final MenuItem dfendImportItem = new MenuItem(menu_1, SWT.NONE);
            dfendImportItem.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(final SelectionEvent event) {
                    doDFendImport();
                }
            });
            dfendImportItem.setText("Import D-Fend Profiles");

            new MenuItem(menu_1, SWT.SEPARATOR);
        }

        final MenuItem migrateMenuItem = new MenuItem(menu_1, SWT.NONE);
        migrateMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doMigrate();
            }
        });
        migrateMenuItem.setText("Migrate Profiles");

        final MenuItem confImportItem = new MenuItem(menu_1, SWT.NONE);
        confImportItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doImportConfigfile();
            }
        });
        confImportItem.setText("Import Profile");
        new MenuItem(menu_1, SWT.SEPARATOR);

        final MenuItem settingsMenuItem = new MenuItem(menu_1, SWT.NONE);
        settingsMenuItem.setText("Adjust Settings");
        settingsMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doOpenSettingsDialog();
            }
        });

        final MenuItem exitMenuItem = new MenuItem(menu_1, SWT.NONE);
        exitMenuItem.setText("Exit");
        exitMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                shell.close();
            }
        });

        final TabFolder tabFolder = new TabFolder(shell, SWT.NONE);

        final TabItem profilesTabItem = new TabItem(tabFolder, SWT.NONE);
        profilesTabItem.setText("Profiles");

        final Composite composite = new Composite(tabFolder, SWT.NONE);
        composite.setLayout(new BorderLayout(0, 0));
        profilesTabItem.setControl(composite);

        final ToolBar toolBar = new ToolBar(composite, SWT.NONE);
        toolBar.setLayoutData(BorderLayout.NORTH);

        SelectionAdapter addProfileAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doAddProfile();
            }
        };
        SelectionAdapter editProfAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doEditProfile();
            }
        };
        SelectionAdapter removeProfAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRemoveProfile();
            }
        };

        final ToolItem newItemToolItem = new ToolItem(toolBar, SWT.PUSH);
        newItemToolItem.setText("Add profile");
        newItemToolItem.addSelectionListener(addProfileAdapter);

        final ToolItem newItemToolItem_2 = new ToolItem(toolBar, SWT.PUSH);
        newItemToolItem_2.setText("Edit Profile");
        newItemToolItem_2.addSelectionListener(editProfAdapter);

        final ToolItem newItemToolItem_1 = new ToolItem(toolBar, SWT.PUSH);
        newItemToolItem_1.setText("Remove profile");
        newItemToolItem_1.addSelectionListener(removeProfAdapter);

        final SashForm sashForm = new SashForm(composite, SWT.NONE);

        profile_table = new Table(sashForm, SWT.FULL_SELECTION | SWT.BORDER);

        final Composite informationGroup = new Composite(sashForm, SWT.NONE);
        informationGroup.addControlListener(new ControlAdapter() {
            public void controlResized(final ControlEvent event) {
                settings.setIntValues(Settings.GUI, "sashweights", sashForm.getWeights());
            }
        });
        informationGroup.setLayout(new BorderLayout(0, 0));
        notesField = new Text(informationGroup, SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER | SWT.WRAP);

        final Composite composite_2 = new Composite(informationGroup, SWT.NONE);
        final GridLayout gridLayout = new GridLayout();
        gridLayout.marginWidth = 0;
        gridLayout.marginHeight = 2;
        gridLayout.verticalSpacing = 1;
        composite_2.setLayout(gridLayout);
        composite_2.setLayoutData(BorderLayout.SOUTH);

        link = new Link[4];
        for (int i = 0; i < link.length; i++) {
            link[i] = new Link(composite_2, SWT.NONE);
            link[i].setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
            link[i].setText(" ");
            link[i].addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(final SelectionEvent event) {
                    PlatformUtils.openForBrowsing(event.text);
                }
            });
        }

        thumbHeight = settings.getIntValue(Settings.GUI, "screenshotsheight");
        final ScrolledComposite scrolledComposite = new ScrolledComposite(composite, SWT.BORDER | SWT.H_SCROLL);
        scrolledComposite.setMinHeight(thumbHeight + 20);
        scrolledComposite.setLayoutData(BorderLayout.SOUTH);

        thumbsToolBar = new ToolBar(scrolledComposite, SWT.NONE);
        thumbsToolBar.addMouseListener(new MouseAdapter() {
            public void mouseDown(final MouseEvent event) {
                currentThumbFile = new File(thumbsToolBar.getItem(new Point(event.x, event.y)).getToolTipText());
            }
        });
        scrolledComposite.setContent(thumbsToolBar);

        final Menu menu_thumb = new Menu(thumbsToolBar);
        thumbsToolBar.setMenu(menu_thumb);

        final MenuItem thumbMenuItem1 = new MenuItem(menu_thumb, SWT.NONE);
        thumbMenuItem1.setText(REMOVE);
        thumbMenuItem1.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRemoveThumb();
            }
        });
        new MenuItem(menu_thumb, SWT.SEPARATOR);

        final MenuItem thumbMenuItem2 = new MenuItem(menu_thumb, SWT.NONE);
        thumbMenuItem2.setText("Open folder");
        thumbMenuItem2.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                PlatformUtils.openDirForViewing(currentThumbFile.getParentFile());
            }
        });

        final MenuItem thumbMenuItem3 = new MenuItem(menu_thumb, SWT.NONE);
        thumbMenuItem3.setText("Refresh");
        thumbMenuItem3.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                SWTImageManager.dispose();
                displayProfileInformation();
            }
        });

        profile_table.addKeyListener(new KeyAdapter() {
            public void keyPressed(final KeyEvent event) {
                if (event.keyCode == SWT.DEL) {
                    doRemoveProfile();
                } else if (event.keyCode == SWT.INSERT) {
                    doAddProfile();
                } else if (event.keyCode == SWT.F2) {
                    doEditProfile(true);
                } else if (event.stateMask == SWT.CTRL && (Character.toLowerCase(event.keyCode) == 'm')) {
                    doToggleFavoriteProfile();
                } else if (event.stateMask == SWT.CTRL && (Character.toLowerCase(event.keyCode) == 'd')) {
                    doDuplicateProfile();
                }
            }
        });

        profile_table.setLayoutData(BorderLayout.WEST);
        profile_table.addTraverseListener(new TraverseListener() {
            public void keyTraversed(final TraverseEvent event) {
                if ((event.stateMask == SWT.CTRL) && (event.detail == SWT.TRAVERSE_RETURN)) {
                    doEditProfile();
                } else if ((event.stateMask == SWT.SHIFT) && (event.detail == SWT.TRAVERSE_RETURN)) {
                    doRunProfile(true);
                } else if (event.detail == SWT.TRAVERSE_RETURN) {
                    doRunProfile(false);
                } 
            }
        });
        profile_table.addMouseListener(new MouseAdapter() {
            public void mouseDoubleClick(final MouseEvent event) {
                doRunProfile(false);
            }
        });

        profile_table.setLinesVisible(true);
        profile_table.setHeaderVisible(true);

        for (int i = 0; i < columnIds.length; i++) {
            addProfileColumn(columnNames[columnIds[i]], i);
        }

        profile_table.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                displayProfileInformation();
            } });

        final Menu menu_2 = new Menu(profile_table);
        profile_table.setMenu(menu_2);

        final MenuItem runMenuItem = new MenuItem(menu_2, SWT.NONE);
        runMenuItem.setText("Run");
        runMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRunProfile(false);
            }
        });

        final MenuItem runWithMenuItem = new MenuItem(menu_2, SWT.CASCADE);
        runWithMenuItem.setText("Run with");

        runWithSubMenu = new Menu(runWithMenuItem);
        runWithMenuItem.setMenu(runWithSubMenu);
        updateProfileDbversionsSubmenu();

        final MenuItem setupMenuItem = new MenuItem(menu_2, SWT.NONE);
        setupMenuItem.setText("Setup");
        setupMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRunProfile(true);
            }
        });

        final MenuItem setupWithMenuItem = new MenuItem(menu_2, SWT.CASCADE);
        setupWithMenuItem.setText("Setup with");

        setupWithSubMenu = new Menu(setupWithMenuItem);
        setupWithMenuItem.setMenu(setupWithSubMenu);
        updateSetupDbversionsSubmenu();

        new MenuItem(menu_2, SWT.SEPARATOR);

        final MenuItem addMenuItem = new MenuItem(menu_2, SWT.NONE);
        addMenuItem.setText("Add");
        addMenuItem.addSelectionListener(addProfileAdapter);

        final MenuItem editMenuItem = new MenuItem(menu_2, SWT.NONE);
        editMenuItem.setText("Edit");
        editMenuItem.addSelectionListener(editProfAdapter);

        final MenuItem duplicateMenuItem = new MenuItem(menu_2, SWT.NONE);
        duplicateMenuItem.setText("Duplicate");
        duplicateMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doDuplicateProfile();
            }
        });

        final MenuItem removeMenuItem = new MenuItem(menu_2, SWT.NONE);
        removeMenuItem.setText(REMOVE);
        removeMenuItem.addSelectionListener(removeProfAdapter);

        new MenuItem(menu_2, SWT.SEPARATOR);

        final MenuItem toggleMenuItem = new MenuItem(menu_2, SWT.NONE);
        toggleMenuItem.setText("Toggle favorite");
        toggleMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doToggleFavoriteProfile();
            }
        });

        if (PlatformUtils.IS_WINDOWS) {
            new MenuItem(menu_2, SWT.SEPARATOR);
            final MenuItem shortcutMenuItem = new MenuItem(menu_2, SWT.NONE);
            shortcutMenuItem.setText("Create shortcut");
            shortcutMenuItem.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(final SelectionEvent event) {
                    doCreateShortcut();
                }
            });
        }

        menu_2.addMenuListener(new MenuAdapter() {
            public void menuShown(final MenuEvent event) {
                if (profile_table.getSelectionIndex() != -1) {
                    boolean hasSetup = profilesList.get(profile_table.getSelectionIndex()).hasSetup();
                    setupMenuItem.setEnabled(hasSetup);
                    setupWithMenuItem.setEnabled(hasSetup);
                }
            }
        });

        sashForm.setWeights(settings.getIntValues(Settings.GUI, "sashweights"));

        final TabItem dosboxTabItem = new TabItem(tabFolder, SWT.NONE);
        dosboxTabItem.setText("DOSBox versions");

        final Composite composite_1 = new Composite(tabFolder, SWT.NONE);
        composite_1.setLayout(new BorderLayout(0, 0));
        dosboxTabItem.setControl(composite_1);

        final ToolBar toolBar_1 = new ToolBar(composite_1, SWT.NONE);
        toolBar_1.setLayoutData(BorderLayout.NORTH);

        SelectionAdapter addDosboxAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doAddDosboxVersion();
            }
        };
        SelectionAdapter editDosboxAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doEditDosboxVersion();
            }
        };
        SelectionAdapter removeDbAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRemoveDosboxVersion();
            }
        };

        final ToolItem newItemToolItem_3 = new ToolItem(toolBar_1, SWT.PUSH);
        newItemToolItem_3.addSelectionListener(addDosboxAdapter);
        newItemToolItem_3.setText("Add version");

        final ToolItem newItemToolItem_4 = new ToolItem(toolBar_1, SWT.PUSH);
        newItemToolItem_4.addSelectionListener(editDosboxAdapter);
        newItemToolItem_4.setText("Edit version");

        final ToolItem newItemToolItem_5 = new ToolItem(toolBar_1, SWT.PUSH);
        newItemToolItem_5.addSelectionListener(removeDbAdapter);
        newItemToolItem_5.setText("Remove version");

        dbversion_table = new Table(composite_1, SWT.FULL_SELECTION | SWT.BORDER);
        dbversion_table.addKeyListener(new KeyAdapter() {
            public void keyPressed(final KeyEvent event) {
                if (event.keyCode == SWT.DEL) {
                    doRemoveDosboxVersion();
                } else if (event.keyCode == SWT.INSERT) {
                    doAddDosboxVersion();
                } else if (event.stateMask == SWT.CTRL && (Character.toLowerCase(event.keyCode) == 'm')) {
                    doToggleDefaultVersion();
                }
            }
        });
        dbversion_table.addTraverseListener(new TraverseListener() {
            public void keyTraversed(final TraverseEvent event) {
                if ((event.stateMask == SWT.CTRL) && (event.detail == SWT.TRAVERSE_RETURN)) {
                    doEditDosboxVersion();
                } else if (event.detail == SWT.TRAVERSE_RETURN) {
                    doRunDosbox();
                }
            }
        });
        dbversion_table.addMouseListener(new MouseAdapter() {
            public void mouseDoubleClick(final MouseEvent event) {
                doRunDosbox();
            }
        });

        dbversion_table.setLinesVisible(true);
        dbversion_table.setHeaderVisible(true);

        addDBColumn("Title", 0);
        addDBColumn("Version", 1);
        addDBColumn("Path", 2);
        addDBColumn("Default", 3);
        addDBColumn("ID", 4);

        final Menu menu_4 = new Menu(dbversion_table);
        dbversion_table.setMenu(menu_4);

        final MenuItem runDBMenuItem = new MenuItem(menu_4, SWT.NONE);
        runDBMenuItem.setText("Run");
        runDBMenuItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRunDosbox();
            }
        });

        new MenuItem(menu_4, SWT.SEPARATOR);

        final MenuItem addDBMenuItem = new MenuItem(menu_4, SWT.NONE);
        addDBMenuItem.setText("Add");
        addDBMenuItem.addSelectionListener(addDosboxAdapter);

        final MenuItem editDBMenuItem = new MenuItem(menu_4, SWT.NONE);
        editDBMenuItem.setText("Edit");
        editDBMenuItem.addSelectionListener(editDosboxAdapter);

        final MenuItem removeDBMenuItem = new MenuItem(menu_4, SWT.NONE);
        removeDBMenuItem.setText(REMOVE);
        removeDBMenuItem.addSelectionListener(removeDbAdapter);

        new MenuItem(menu_4, SWT.SEPARATOR);

        final MenuItem toggleDefaultItem = new MenuItem(menu_4, SWT.NONE);
        toggleDefaultItem.setText("Toggle default");
        toggleDefaultItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doToggleDefaultVersion();
            }
        });

        new MenuItem(menu_4, SWT.SEPARATOR);

        final MenuItem openFolderItem = new MenuItem(menu_4, SWT.NONE);
        openFolderItem.setText("Open folder");
        openFolderItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                int index = dbversion_table.getSelectionIndex();
                if (index != -1) {
                    PlatformUtils.openDirForViewing(dbversionsList.get(index).getCanonicalConfFile().getParentFile());
                }
            }
        });

        final TabItem templatesTabItem = new TabItem(tabFolder, SWT.NONE);
        templatesTabItem.setText("Templates");

        final Composite composite_1_1 = new Composite(tabFolder, SWT.NONE);
        composite_1_1.setLayout(new BorderLayout(0, 0));
        templatesTabItem.setControl(composite_1_1);

        final ToolBar toolBar_1_1 = new ToolBar(composite_1_1, SWT.NONE);
        toolBar_1_1.setLayoutData(BorderLayout.NORTH);

        SelectionAdapter addTemplAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doAddTemplate();
            }
        };
        SelectionAdapter editTemplAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doEditTemplate();
            }
        };
        SelectionAdapter remTemplAdapter = new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doRemoveTemplate();
            }
        };

        final ToolItem newItemToolItem_6 = new ToolItem(toolBar_1_1, SWT.PUSH);
        newItemToolItem_6.setText("Add template");
        newItemToolItem_6.addSelectionListener(addTemplAdapter);

        final ToolItem newItemToolItem_7 = new ToolItem(toolBar_1_1, SWT.PUSH);
        newItemToolItem_7.setText("Edit template");
        newItemToolItem_7.addSelectionListener(editTemplAdapter);

        final ToolItem newItemToolItem_8 = new ToolItem(toolBar_1_1, SWT.PUSH);
        newItemToolItem_8.setText("Remove template");
        newItemToolItem_8.addSelectionListener(remTemplAdapter);

        template_table = new Table(composite_1_1, SWT.FULL_SELECTION | SWT.BORDER);
        template_table.addKeyListener(new KeyAdapter() {
            public void keyPressed(final KeyEvent event) {
                if (event.keyCode == SWT.DEL) {
                    doRemoveTemplate();
                } else if (event.keyCode == SWT.INSERT) {
                    doAddTemplate();
                } else if (event.stateMask == SWT.CTRL && (Character.toLowerCase(event.keyCode) == 'm')) {
                    doToggleDefaultTemplate();
                }
            }
        });
        template_table.addTraverseListener(new TraverseListener() {
            public void keyTraversed(final TraverseEvent event) {
                if ((event.stateMask == SWT.CTRL) && (event.detail == SWT.TRAVERSE_RETURN)) {
                    doEditTemplate();
                }
            }
        });

        template_table.setLinesVisible(true);
        template_table.setHeaderVisible(true);

        addTemplateColumn("Title", 0);
        addTemplateColumn("Default", 1);

        final Menu menu_3 = new Menu(template_table);
        template_table.setMenu(menu_3);

        final MenuItem addTemplItem = new MenuItem(menu_3, SWT.NONE);
        addTemplItem.setText("Add");
        addTemplItem.addSelectionListener(addTemplAdapter);

        final MenuItem editTemplItem = new MenuItem(menu_3, SWT.NONE);
        editTemplItem.setText("Edit");
        editTemplItem.addSelectionListener(editTemplAdapter);

        final MenuItem removeTemplItem = new MenuItem(menu_3, SWT.NONE);
        removeTemplItem.setText(REMOVE);
        removeTemplItem.addSelectionListener(remTemplAdapter);

        new MenuItem(menu_3, SWT.SEPARATOR);

        final MenuItem toggleTemplItem = new MenuItem(menu_3, SWT.NONE);
        toggleTemplItem.setText("Toggle default");
        toggleTemplItem.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doToggleDefaultTemplate();
            }
        });

        // init values
        for (Profile prof: profilesList) {
            addProfileToTable(prof);
        }
        for (DosboxVersion dbversion: dbversionsList) {
            addDosboxVersionToTable(dbversion);
        }
        for (Template template: templatesList) {
            addTemplateToTable(template);
        }

        profile_table.setSortColumn(profile_table.getColumn(findColumnById(settings.getIntValues(Settings.GUI, Settings.SORTCOLUMN)[0])));
        profile_table.setSortDirection(settings.getBooleanValues(Settings.GUI, Settings.SORTASCENDING)[0]? SWT.UP: SWT.DOWN);
        profile_table.setColumnOrder(settings.getIntValues(Settings.GUI, Settings.COLUMNORDER));
        profile_table.setSelection(settings.getIntValue(Settings.GUI, "selectedprofile"));
        profile_table.setFocus();
        displayProfileInformation();
    }

    private void doReorderProfiles(final int columnId, final SelectionEvent event) {
        TableColumn sortColumn = profile_table.getSortColumn();
        TableColumn currentColumn = (TableColumn) event.widget;
        int dir = profile_table.getSortDirection();
        if (sortColumn.equals(currentColumn)) {
            dir = dir == SWT.UP ? SWT.DOWN : SWT.UP;
        } else {
            profile_table.setSortColumn(currentColumn);
            dir = SWT.UP;
        }
        int currentProfileId = -1;
        if (profile_table.getSelectionIndex() != -1) {
            currentProfileId = profilesList.get(profile_table.getSelectionIndex()).getId();
        }
        try {
            orderingVector.addOrdering(columnIds[columnId], dir == SWT.UP);
            profilesList = dbase.readProfilesList(orderingVector.toClause());
        } catch (SQLException ex) {
            GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
        }
        for (int i = 0; i < profilesList.size(); i++) {
            setTableItem(profile_table.getItem(i), profilesList.get(i));
        }
        profile_table.setSortDirection(dir);
        if (currentProfileId != -1) {
            profile_table.setSelection(Profile.findById(profilesList, currentProfileId));
        }
        settings.setIntValues(Settings.GUI, Settings.SORTCOLUMN, orderingVector.getColumns());
        settings.setBooleanValues(Settings.GUI, Settings.SORTASCENDING, orderingVector.getAscendings());
    }

    private void displayNotes(final String notes) {
        notesField.setText(notes);
    }

    private void displayLinks(final String[] p_link) {
        for (int i = 0; i < link.length; i++) {
            if (p_link[i] == null || "".equals(p_link[i])) {
                link[i].setText(" ");
                link[i].setToolTipText(null);
            } else {
                String url = p_link[i];
                String tag = p_link[i];
                if (url.indexOf("://") == -1) {
                    url = "file://" + FileUtils.canonicalToData(url).getPath();
                    tag = FileUtils.makeRelativeToDosroot(FileUtils.canonicalToData(tag)).getPath();
                }
                StringBuffer text = new StringBuffer("<a href=\"").append(url).append("\">").append(tag).append("</a>");
                link[i].setText(text.toString());
                link[i].setToolTipText(url);
            }
        }
    }

    private void displayScreenshots(final Profile prof) {
        ToolItem[] toolItems = thumbsToolBar.getItems();
        for (int i = 0; i < toolItems.length; i++) {
            toolItems[i].dispose();
        }
        if (prof != null) {
            File path = prof.getCanonicalCaptures();
            File[] files = path.listFiles();
            if (files != null) {
                for (File file: files) {
                    if (file.getName().toLowerCase().endsWith(".png")) {
                        final ToolItem toolItem = new ToolItem(thumbsToolBar, SWT.PUSH);
                        toolItem.setToolTipText(file.getPath());
                        toolItem.setText(file.getName());
                        toolItem.setImage(SWTImageManager.getResizedImage(display, thumbHeight, file.getPath()));
                        toolItem.addSelectionListener(new SelectionAdapter() {
                            public void widgetSelected(final SelectionEvent event) {
                                final Thumb thumbDialog = new Thumb(shell);
                                thumbDialog.setThumb(toolItem.getToolTipText());
                                thumbDialog.open();
                            }
                        });
                    }
                }
            }
        }
        thumbsToolBar.setVisible(thumbsToolBar.getItemCount() != 0);
        thumbsToolBar.pack();
    }

    private void doAddProfile() {
        doAddProfile(null);
    }

    private void doAddProfile(final String file) {
        if (checkDefaultDBVersion() == null) {
            return;
        }
        final EditProfileDialog addProfileDialog = new EditProfileDialog(shell);
        if (file != null) {
            addProfileDialog.sendToProfile(file);
        }
        updateWithAddedProfile((Profile)addProfileDialog.open());
    }

    public void addProfile(final String file) {
        display.syncExec(new Runnable() {
            public void run() {
                doAddProfile(file);
            }
        });
    }

    private void doAddDosboxVersion() {
        final EditDosboxVersionDialog addVersionDialog = new EditDosboxVersionDialog(shell);
        DosboxVersion dbversion = (DosboxVersion)addVersionDialog.open();
        if (dbversion != null) {
            updateDosboxVersionList(dbversion);
        }
    }

    private void doAddTemplate() {
        if (checkDefaultDBVersion() == null) {
            return;
        }
        final EditTemplateDialog addTemplateDialog = new EditTemplateDialog(shell);
        Template template = (Template)addTemplateDialog.open();
        if (template != null) {
            updateTemplateList(template);
        }
    }

    private void doToggleFavoriteProfile() {
        int index = profile_table.getSelectionIndex();
        if (index != -1) {
            Profile profile = profilesList.get(index);
            try {
                dbase.toggleFavorite(profile.getId());
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
            profile.toggleDefault();
            profilesList.set(index, profile);
            setTableItem(profile_table.getItem(index), profile);
        }
    }

    private void doToggleDefaultVersion() {
        int index = dbversion_table.getSelectionIndex();
        if (index != -1) {
            DosboxVersion ver = dbversionsList.get(index);
            ver.toggleDefault();
            try {
                dbase.addOrEditDosboxVersion(ver.getTitle(), ver.getPath(), ver.isMultiConfig(), ver.isDefault(),
                        ver.getParameters(), ver.getVersion(), ver.getId());
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
            updateDosboxVersionList(ver);
        }
    }

    private void doToggleDefaultTemplate() {
        int index = template_table.getSelectionIndex();
        if (index != -1) {
            Template temp = templatesList.get(index);
            temp.toggleDefault();
            try {
                dbase.addOrEditTemplate(temp.getTitle(), temp.getdbversionId(), temp.isDefault(), temp.getId());
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
            updateTemplateList(temp);
        }
    }

    private void doEditProfile() {
        doEditProfile(false);
    }

    private void doEditProfile(final boolean focusTitle) {
        int index = profile_table.getSelectionIndex();
        if (index != -1) {
            final EditProfileDialog editProfileDialog = new EditProfileDialog(shell);
            editProfileDialog.setProfile(profilesList.get(index));
            if (focusTitle) {
                editProfileDialog.focusTitle();
            }
            Profile profile = (Profile)editProfileDialog.open();
            if (profile != null) {
                boolean quickUpdate = true;
                if (settings.getBooleanValue(Settings.GUI, "autosortonupdate")) {
                    try {
                        profilesList = dbase.readProfilesList(orderingVector.toClause());
                        if (index != Profile.findById(profilesList, profile.getId())) {
                            quickUpdate = false;
                        }
                    } catch (SQLException e) {
                        GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
                    }
                }
                if (quickUpdate) {
                    profilesList.set(index, profile);
                    setTableItem(profile_table.getItem(index), profile);
                } else {
                    updateProfilesList(profile);
                }
                displayProfileInformation();
            }
        }
    }

    private void doDuplicateProfile() {
        int index = profile_table.getSelectionIndex();
        if (index != -1) {
            Profile orgProf = profilesList.get(index);
            updateWithAddedProfile(EditProfileDialog.duplicateProfile(orgProf, dbase, shell));
        }
    }

    private void updateWithAddedProfile(final Profile profile) {
        if (profile != null) {
            if (settings.getBooleanValue(Settings.GUI, "autosortonupdate")) {
                updateProfilesList(profile);
            } else {
                profilesList.add(profile);
                addProfileToTable(profile);
                profile_table.setSelection(profile_table.getItemCount() - 1);
                profile_table.setFocus();
            }
            displayProfileInformation();
        }
    }

    private void doEditDosboxVersion() {
        int index = dbversion_table.getSelectionIndex();
        if (index != -1) {
            final EditDosboxVersionDialog editVersionDialog = new EditDosboxVersionDialog(shell);
            editVersionDialog.setDosboxVersion(dbversionsList.get(index));
            DosboxVersion dbversion = (DosboxVersion)editVersionDialog.open();
            if (dbversion != null) {
                updateDosboxVersionList(dbversion);
            }
        }
    }

    private void updateProfilesList(final Profile profile) {
        try {
            profilesList = dbase.readProfilesList(orderingVector.toClause());
        } catch (SQLException e) {
            GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
        }
        profile_table.setRedraw(false);
        profile_table.removeAll();
        for (Profile prof: profilesList) {
            addProfileToTable(prof);
        }
        if (profile != null) {
            profile_table.setSelection(Profile.findById(profilesList, profile.getId()));
        }
        profile_table.setRedraw(true);
        profile_table.setFocus();
    }

    private void updateDosboxVersionList(final DosboxVersion dbversion) {
        try {
            dbversionsList = dbase.readDosboxVersionsList();
        } catch (SQLException e) {
            GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
        }
        dbversion_table.removeAll();
        for (DosboxVersion version: dbversionsList) {
            addDosboxVersionToTable(version);
        }
        dbversion_table.setSelection(DosboxVersion.findById(dbversionsList, dbversion.getId()));
        dbversion_table.setFocus();

        updateProfileDbversionsSubmenu();
        updateSetupDbversionsSubmenu();
    }

    private void updateTemplateList(final Template template) {
        try {
            templatesList = dbase.readTemplatesList();
        } catch (SQLException e) {
            GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
        }
        template_table.removeAll();
        for (Template temp: templatesList) {
            addTemplateToTable(temp);
        }
        template_table.setSelection(Template.findById(templatesList, template.getId()));
        template_table.setFocus();
    }

    private void doEditTemplate() {
        int index = template_table.getSelectionIndex();
        if (index != -1) {
            final EditTemplateDialog editTemplDialog = new EditTemplateDialog(shell);
            editTemplDialog.setTemplate(templatesList.get(index));
            Template template = (Template)editTemplDialog.open();
            if (template != null) {
                updateTemplateList(template);
            }
        }
    }

    private void doRemoveProfile() {
        int index = profile_table.getSelectionIndex();
        if ((index != -1) && GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected profile?")) {
            Profile prof = profilesList.get(index);
            try {
                dbase.removeProfile(prof.getId());
                profile_table.remove(index);
                profilesList.remove(index);
                displayProfileInformation();
                SWTImageManager.dispose();
                if (GeneralPurposeDialogs.confirmMessage(shell,
                        "Do you also want to remove the associated conf file " + prof.getConfPathAndFile() + " ?")) {
                    FileUtils.removeFile(prof.getCanonicalConfFile());
                }
                if (GeneralPurposeDialogs.confirmMessage(shell,
                        "Do you also want to remove the associated captures directory " + prof.getCaptures() +
                " including its files?")) {
                    FileUtils.removeFilesInDirAndDir(prof.getCanonicalCaptures());
                }
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
        }
    }

    private void doRemoveDosboxVersion() {
        int index = dbversion_table.getSelectionIndex();
        if ((index != -1) && GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected DOSBox version?")) {
            try {
                dbase.removeDosboxVersion((dbversionsList.get(index)).getId());
                dbversion_table.remove(index);
                dbversionsList.remove(index);
                updateProfileDbversionsSubmenu();
                updateSetupDbversionsSubmenu();
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
        }
    }

    private void doRemoveTemplate() {
        int index = template_table.getSelectionIndex();
        if ((index != -1) && GeneralPurposeDialogs.confirmMessage(shell, "Do you want to remove the selected template?")) {
            int templateId = (templatesList.get(index)).getId();
            try {
                dbase.removeTemplate(templateId);
                template_table.remove(index);
                templatesList.remove(index);
                File conffile = FileUtils.constructCanonicalTemplateFileLocation(templateId);
                if (GeneralPurposeDialogs.confirmMessage(shell,
                        "Do you also want to remove the associated conf file " + conffile + " ?")) {
                    FileUtils.removeFile(conffile);
                }
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
        }
    }

    private void addProfileToTable(final Profile prof) {
        final TableItem newItemTableItem = new TableItem(profile_table, SWT.BORDER); 
        setTableItem(newItemTableItem, prof);
    }

    private void addDosboxVersionToTable(final DosboxVersion dbversion) {
        final TableItem newItemTableItem = new TableItem(dbversion_table, SWT.BORDER); 
        setTableItem(newItemTableItem, dbversion);
    }

    private void addTemplateToTable(final Template template) {
        final TableItem newItemTableItem = new TableItem(template_table, SWT.BORDER); 
        setTableItem(newItemTableItem, template);
    }

    private void setTableItem(final TableItem newItemTableItem, final Profile prof) {
        for (int i = 0; i < columnIds.length; i++) {
            String value;
            switch (columnIds[i]) {
                case 0: value = prof.getTitle(); break;
                case 1: value = prof.hasSetupString(); break;
                case 2: value = prof.getDeveloperName(); break;
                case 3: value = prof.getPublisherName(); break;
                case 4: value = prof.getGenre(); break;
                case 5: value = prof.getYear(); break;
                case 6: value = prof.getStatus(); break;
                case 7: value = prof.isDefaultString(); break;
                case 8: value = String.valueOf(prof.getId()); break;
                case 9: value = String.valueOf(prof.getDbversionId()); break;
                case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
                    value = prof.getCustomString(columnIds[i] - RO_COLUMN_NAMES); break;
                case 18:value = prof.getCustomInt(0) + " %"; break;
                case 19:value = String.valueOf(prof.getCustomInt(1)); break;
                default:value = "";
            }
            newItemTableItem.setText(i, value);
        }
    }

    private void setTableItem(final TableItem newItemTableItem, final DosboxVersion dbversion) {
        newItemTableItem.setText(0, dbversion.getTitle());
        newItemTableItem.setText(1, dbversion.getVersion());
        newItemTableItem.setText(2, dbversion.getPath());
        newItemTableItem.setText(3, dbversion.isDefaultString());
        newItemTableItem.setText(4, String.valueOf(dbversion.getId()));
    }

    private void setTableItem(final TableItem newItemTableItem, final Template template) {
        newItemTableItem.setText(0, template.getTitle());
        newItemTableItem.setText(1, template.isDefaultString());
    }

    private void doRunProfile(final boolean setup) {
        int index = profile_table.getSelectionIndex();
        if (index != -1) {
            Profile prof = profilesList.get(index);
            if (!setup || prof.hasSetup()) {
                try {
                    FileUtils.doRunProfile(prof, dbversionsList, setup);
                } catch (IOException e) {
                    GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
                }
            }
        }
    }

    private void doRunDosbox() {
        int index = dbversion_table.getSelectionIndex();
        if (index != -1) {
            try {
                FileUtils.doRunDosbox(dbversionsList.get(index));
            } catch (IOException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
        }
    }

    private DosboxVersion checkDefaultDBVersion() {
        int defaultVersionIdx = DosboxVersion.findDefault(dbversionsList);
        if (defaultVersionIdx == -1) {
            GeneralPurposeDialogs.infoMessage(shell, "You must first define a default DOSBox version!");
            return null;
        }
        return dbversionsList.get(defaultVersionIdx);
    }

    private void doDFendImport() {
        DosboxVersion defaultDbversion = checkDefaultDBVersion();
        if (defaultDbversion == null) {
            return;
        }

        if (settings.getIntValue("profiledefaults", "confpath") == 1) {
            GeneralPurposeDialogs.infoMessage(shell, "Please realize that you have currently configured " +
                    "DBGL to store profile configuration files in the game directory. " +
                    "Therefore, if you continue and perform the import, all configuration files are " +
                    "saved to the various game directories configured in D-Fend.\n\n" +
                    "If you'd rather have those DOSBox config files stored in a central place, please open " +
                    "the Settings dialog first and change the default configuration-file location to '" +
                    SettingsDialog.confLocations[0] + "'!");
        }

        DFendImportDialog importDialog = new DFendImportDialog(shell);
        importDialog.setDefaultDosboxVersion(defaultDbversion);
        if (importDialog.open() != null) {
            try {
                profilesList = dbase.readProfilesList(orderingVector.toClause());
                profile_table.removeAll();
                for (Profile prof: profilesList) {
                    addProfileToTable(prof);
                }
            } catch (SQLException e) {
                GeneralPurposeDialogs.warningMessage(shell, e.getMessage());
            }
        }
    }

    private void doMigrate() {
        GeneralPurposeDialogs.infoMessage(shell, "Please realize that you are about to alter all your profile data.\n\n" +
                "DBGL will change all absolute profile location information inside the .conf files to relative paths. " +
                "Therefore, please make sure that you have made a backup of your DBGL directory!\n\n" +
        "You will have to manually move your games, afterwards.");
        String from = (String) new MigrateDialog(shell).open();
        if (from != null) {
            GeneralPurposeDialogs.infoMessage(shell, "Now close DBGL, move your game-related files and directories " +
                    "inside " + from + "\n" +
                    "to the new location " + FileUtils.getDosRoot() + "\n" + 
            "and restart DBGL to complete the migration.");
        }
    }

    private void displayProfileInformation() {
        int index = profile_table.getSelectionIndex();
        if (index == -1) {
            displayScreenshots(null);
            displayNotes("");
            displayLinks(new String[] {null, null, null, null});
        } else {
            settings.setIntValue(Settings.GUI, "selectedprofile", index);
            Profile prof = profilesList.get(index);
            displayScreenshots(prof);
            displayNotes(prof.getNotes());
            displayLinks(prof.getLinks());
        }
    }

    private void doRemoveThumb() {
        if (GeneralPurposeDialogs.confirmMessage(shell, "Would you like to remove the selected " +
                "screenshot \"" + currentThumbFile + "\"?")) {
            FileUtils.removeFile(currentThumbFile);
            displayProfileInformation();
        }
        currentThumbFile = null;
    }

    private void doCreateShortcut() {
        int index = profile_table.getSelectionIndex();
        if (index != -1) {
            PlatformUtils.createShortcut(profilesList.get(index), dbversionsList);
        }
    }

    private void addProfileColumn(final String title, final int colIndex) {
        final String width = "column" + (columnIds[colIndex] + 1) + Settings.WIDTH;
        final TableColumn column = new TableColumn(profile_table, SWT.NONE);
        column.setWidth(settings.getIntValue(Settings.GUI, width));
        column.setMoveable(true);
        column.setText(title);
        if ((columnIds[colIndex] == 8) || (columnIds[colIndex] == 9) ||
                (columnIds[colIndex] == 18) || (columnIds[colIndex] == 19)) { // numeric values
            column.setAlignment(SWT.RIGHT);
        }
        column.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                doReorderProfiles(colIndex, event);
            }
        });
        column.addControlListener(new ControlAdapter() {
            public void controlResized(final ControlEvent event) {
                settings.setIntValue(Settings.GUI, width, column.getWidth());
            }
            public void controlMoved(final ControlEvent event) {
                settings.setIntValues(Settings.GUI, Settings.COLUMNORDER, profile_table.getColumnOrder());
            }
        });
    }

    private void addDBColumn(final String title, final int colIndex) {
        final String width = "column2_" + (colIndex + 1) + Settings.WIDTH;
        final TableColumn column = new TableColumn(dbversion_table, SWT.NONE);
        column.setWidth(settings.getIntValue(Settings.GUI, width));
        column.setText(title);
        column.addControlListener(new ControlAdapter() {
            public void controlResized(final ControlEvent event) {
                settings.setIntValue(Settings.GUI, width, column.getWidth());
            }
        });
    }

    private void addTemplateColumn(final String title, final int colIndex) {
        final String width = "column3_" + (colIndex + 1) + Settings.WIDTH;
        final TableColumn column = new TableColumn(template_table, SWT.NONE);
        column.setWidth(settings.getIntValue(Settings.GUI, width));
        column.setText(title);
        column.addControlListener(new ControlAdapter() {
            public void controlResized(final ControlEvent event) {
                settings.setIntValue(Settings.GUI, width, column.getWidth());
            }
        });
    }

    private void doOpenSettingsDialog() {
        SettingsDialog sDialog = new SettingsDialog(shell);
        if (((Boolean)sDialog.open())) {
            // first, store current table properties
            int index = profile_table.getSelectionIndex();
            int nrColumns = profile_table.getColumnCount();
            String order = settings.getValue(Settings.GUI, Settings.COLUMNORDER);

            // then clean the table, reset column order and dispose all columns
            profile_table.removeAll();
            StringBuffer cols = new StringBuffer();
            for (int i = 0; i < nrColumns; i++) {
                cols.append(i).append(' ');
            }
            profile_table.setColumnOrder(StringUtils.stringToIntArray(cols.toString()));
            for (int i = nrColumns - 1; i >= 0; i--) {
                profile_table.getColumn(i).dispose();
            }

            // next, recreate column ids and add selected columns to table
            initColumnIds();
            for (int i = 0; i < columnIds.length; i++) {
                addProfileColumn(columnNames[columnIds[i]], i);
            }

            // redefine sort column in table
            profile_table.setSortColumn(profile_table.getColumn(findColumnById(settings.getIntValues(Settings.GUI, Settings.SORTCOLUMN)[0])));
            profile_table.setSortDirection(settings.getBooleanValues(Settings.GUI, Settings.SORTASCENDING)[0]? SWT.UP: SWT.DOWN);

            // restore saved columnorder since disposing and adding columns invalidates that
            settings.setValue(Settings.GUI, Settings.COLUMNORDER, order);

            // also reset orderingvector
            orderingVector = new OrderingVector(settings.getIntValues(Settings.GUI, Settings.SORTCOLUMN),
                    settings.getBooleanValues(Settings.GUI, Settings.SORTASCENDING));

            // finally, refill table with data
            Profile prof = (index == -1)? null: profilesList.get(index);
            updateProfilesList(prof);
        }
    }

    private void doImportConfigfile() {
        FileDialog dialog = new FileDialog (shell, SWT.OPEN);
        dialog.setFilterNames (new String [] {
                "DOSBox configuration files", "Executables, Booter Disk Images", FileUtils.ALL_FILTER});
        dialog.setFilterExtensions (new String [] {FileUtils.CNF_FILTER, 
                FileUtils.EXE_FILTER + ";" + FileUtils.BTR_FILTER, FileUtils.ALL_FILTER});
        String result = dialog.open();
        if (result != null) {
            doAddProfile(result);
        }
    }

    private void initColumnIds() {
        columnNames = new String[RO_COLUMN_NAMES + EDIT_COLUMN_NAMES];
        columnNames[0] = "Title";
        columnNames[1] = "Setup";
        columnNames[2] = "Developer";
        columnNames[3] = "Publisher";
        columnNames[4] = "Genre";
        columnNames[5] = "Year";
        columnNames[6] = "Status";
        columnNames[7] = "Favorite";
        columnNames[8] = "ID";
        columnNames[9] = "DB ID";
        for (int i = 0; i < EDIT_COLUMN_NAMES; i++) {
            columnNames[i + RO_COLUMN_NAMES] = settings.getValue(Settings.GUI, "custom" + (i + 1));
        }

        int amount = 0;
        for (int i = 0; i < columnNames.length; i++) {
            if (settings.getBooleanValue(Settings.GUI, "column" + (i + 1) + "visible")) {
                amount++;
            }
        }

        int cNr = 0;
        columnIds = new int[amount];
        for (int i = 0; i < columnNames.length; i++) {
            if (settings.getBooleanValue(Settings.GUI, "column" + (i + 1) + "visible")) {
                columnIds[cNr++] = i;
            }
        }
    }

    private int findColumnById(final int id) {
        for (int i = 0; i < columnIds.length; i++) {
            if (columnIds[i] == id) {
                return i;
            }
        }
        return -1;
    }

    private void updateProfileDbversionsSubmenu() {
        MenuItem parent = runWithSubMenu.getParentItem();
        runWithSubMenu.dispose();
        runWithSubMenu = new Menu(parent);
        parent.setMenu(runWithSubMenu);
        for (int i = 0; i < dbversionsList.size(); i++) {
            final MenuItem runWithMenuItem = new MenuItem(runWithSubMenu, SWT.NONE);
            runWithMenuItem.setText(dbversionsList.get(i).getTitle());
            runWithMenuItem.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(final SelectionEvent event) {
                    int index = profile_table.getSelectionIndex();
                    if (index != -1) {
                        Profile prof = profilesList.get(index);
                        DosboxVersion dbversion = dbversionsList.get(runWithSubMenu.indexOf(runWithMenuItem));
                        try {
                            FileUtils.doRunProfile(prof, dbversion, false);
                        } catch (IOException exc) {
                            GeneralPurposeDialogs.warningMessage(shell, exc.getMessage());
                        }
                    }
                }
            });
        }
    }

    private void updateSetupDbversionsSubmenu() {
        MenuItem parent = setupWithSubMenu.getParentItem();
        setupWithSubMenu.dispose();
        setupWithSubMenu = new Menu(parent);
        parent.setMenu(setupWithSubMenu);
        for (int i = 0; i < dbversionsList.size(); i++) {
            final MenuItem setupWithMenuItem = new MenuItem(setupWithSubMenu, SWT.NONE);
            setupWithMenuItem.setText(dbversionsList.get(i).getTitle());
            setupWithMenuItem.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(final SelectionEvent event) {
                    int index = profile_table.getSelectionIndex();
                    if (index != -1) {
                        Profile prof = profilesList.get(index);
                        DosboxVersion dbversion = dbversionsList.get(setupWithSubMenu.indexOf(setupWithMenuItem));
                        try {
                            FileUtils.doRunProfile(prof, dbversion, true);
                        } catch (IOException exc) {
                            GeneralPurposeDialogs.warningMessage(shell, exc.getMessage());
                        }
                    }
                }
            });
        }
    }
}
