package com.model.conf;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import com.exception.InvalidMountstringException;
import com.model.Mount;
import com.util.FileUtils;
import com.util.PlatformUtils;
import com.util.StringUtils;


public class Configuration {

    public static final String JOYSTICK = "joystick";
    public static final String BASE = "base";
    public static final String GUSBASE = "gusbase";
    public static final String GUSRATE = "gusrate";
    public static final String SBBASE = "sbbase";
    public static final String SBTYPE = "sbtype";
    public static final String JOYSTICKTYPE = "joysticktype";
    public static final String MACHINE = "machine";
    public static final String CPUTYPE = "cputype";
    public static final String DOSBOX = "dosbox";
    public static final String CPU = "cpu";
    public static final String SBLASTER = "sblaster";
    public static final String GUS = "gus";
    
    private static final String CYCLES = "cycles";
    
    private Map<String, Section> sections;
    private Autoexec autoexec;
    private StringBuffer parsingProblems;


    public Configuration() {
        parsingProblems = new StringBuffer();
        sections = new LinkedHashMap<String, Section>();
        autoexec = new Autoexec(new String[0]);
    }
    
    public Configuration(final Configuration conf, final boolean keepLoadfix) {
        this();
        this.autoexec = new Autoexec(conf.autoexec);
        if (!keepLoadfix) {
            this.autoexec.loadfix = false;
            this.autoexec.loadfixValue = 0;
        }
    }

    public Configuration(final Configuration conf, final boolean mountingsOnly, final Configuration baseConf, final boolean switchDbversion) {
        this();
        for (String key: conf.sections.keySet()) {
            sections.put(key, new Section(conf.sections.get(key)));
        }
        if (mountingsOnly) {
            this.autoexec = new Autoexec(conf.autoexec.mountingpoints, baseConf == null? null: baseConf.autoexec, switchDbversion);
        } else {
            this.autoexec = new Autoexec(conf.autoexec);
        }
    }

    public Configuration(final File file, final String setup, final String setupParams) throws IOException {
        this(file, true);
        autoexec.main = setup;
        autoexec.params = setupParams;
    }

    public Configuration(final File file, final boolean toLowerCase) throws IOException {
        this();
        parseConfigurationFile(file, toLowerCase);
    }

    public Configuration(final String content) throws IOException {
        this();
        parseConfigurationFile(new StringReader(content), FileUtils.PROFILES_XML, false);
    }

    public Set<String> getSectionKeys() {
        return sections.keySet();
    }

    public void saveToFile(final File file, final boolean ordered, final Configuration baseConf) throws IOException {
        try {
            BufferedWriter configData = new BufferedWriter(new FileWriter(file));
            configData.write(toString(ordered, baseConf));
            configData.flush();
            configData.close();
        } catch (IOException e) {
            e.printStackTrace();
            throw new IOException("Something went wrong while trying to save " + file);
        }
    }

    public final void parseConfigurationFile(final File file, final boolean toLowerCase) throws IOException {
        try {
            parseConfigurationFile(new FileReader(file), file.getPath(), toLowerCase);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            throw new IOException("Could not read " + e.getMessage());
        }
    }

    private void parseConfigurationFile(final Reader reader, final String file, final boolean toLowerCase) throws IOException {
        try {
            BufferedReader configData = new BufferedReader(reader);
            String orgTextLine;
            String currSectionTitle = null;
            Section currSection = null;
            List<String> autoexecLines = new ArrayList<String>();

            while ((orgTextLine = configData.readLine()) != null) {
                orgTextLine = orgTextLine.trim();
                String textLine = toLowerCase? orgTextLine.toLowerCase(): orgTextLine;
                if ((textLine.length() > 0) && textLine.charAt(0) != '#') {
                    if (textLine.charAt(0) == '[') { // a new section starts here
                        int start = textLine.indexOf(('['));
                        int end = textLine.lastIndexOf(']');
                        if (end == -1) {
                            parsingProblems.append("Parsing error in " + file + ", line: " + textLine + "\n");
                        } else {
                            currSectionTitle = textLine.substring(start + 1, end);
                            if (sections.containsKey(currSectionTitle)) {
                                currSection = sections.get(currSectionTitle);
                            } else {
                                if (!"autoexec".equals(currSectionTitle)) {
                                    currSection = new Section();
                                    sections.put(currSectionTitle, currSection);
                                }
                            }
                        }
                    } else { // an item starts here
                        if (currSectionTitle == null) { // value before section
                            parsingProblems.append("Parsing error (no section) in " + file + ", line: " + textLine + "\n");
                        } else { 
                            if ("autoexec".equals(currSectionTitle)) { // autoexec config item
                                autoexecLines.add(orgTextLine);
                            } else { // normal config item
                                int end = textLine.indexOf('=');
                                if (end == -1) {
                                    parsingProblems.append("Parsing error in " + file + ", line: " + textLine + "\n");
                                } else {
                                    String name = textLine.substring(0, end).trim();
                                    String value = textLine.substring(end + 1).trim();
                                    if ("gui".equals(currSectionTitle) || "database".equals(currSectionTitle)
                                            || "directory".equals(currSectionTitle)) {
                                        value = orgTextLine.substring(end + 1).trim();
                                    }
                                    currSection.items.put(name, value);
                                }
                            }
                        }
                    }
                }
            }
            configData.close();
            autoexec.parseLines(autoexecLines.toArray(new String[autoexecLines.size()]), false);
        } catch (IOException e) {
            e.printStackTrace();
            StringBuffer msg = new StringBuffer("Something went wrong while trying to read ").append(file);
            if (parsingProblems.length() > 0) {
                msg.append('\n').append(getParsingProblems());
            }
            throw new IOException(msg.toString());
        }
    }

    public boolean hasValue(final String sectionTitle, final String sectionItem) {
        return sections.containsKey(sectionTitle) && sections.get(sectionTitle).items.containsKey(sectionItem);
    }

    public String getValue(final String sectionTitle, final String sectionItem) {
        String result = null;
        if (sections.containsKey(sectionTitle)) {
            result = sections.get(sectionTitle).items.get(sectionItem);
        }
        if (result == null) {
            result = ""; // in case the item was not found
        }
        return result;
    }

    public int getIntValue(final String sectionTitle, final String sectionItem) {
        try {
            return Integer.parseInt(getValue(sectionTitle, sectionItem));
        } catch (NumberFormatException e) {
            return -1; // value is not a number
        }
    }

    public boolean getBooleanValue(final String sectionTitle, final String sectionItem) {
        return Boolean.valueOf(getValue(sectionTitle, sectionItem));
    }

    public void setValue(final String sectionTitle, final String sectionItem, final String value) {
        Section sec = createSection(sectionTitle);
        sec.items.put(sectionItem, value);
    }

    public void setIntValue(final String sectionTitle, final String sectionItem, final int value) {
        setValue(sectionTitle, sectionItem, String.valueOf(value));
    }

    public void setBooleanValue(final String sectionTitle, final String sectionItem, final boolean value) {
        setValue(sectionTitle, sectionItem, String.valueOf(value));
    }

    public void updateValue(final String sectionTitle, final String sectionItem, final String value) {
        if (sections.containsKey(sectionTitle) && sections.get(sectionTitle).items.containsKey(sectionItem)) {
            sections.get(sectionTitle).items.put(sectionItem, value);
        }
    }
    
    public String[] getItems(final String sectionTitle) {
        return new TreeMap<String, String>(sections.get(sectionTitle).items).keySet().toArray(new String[0]);
    }

    public String[] getValues(final String sectionTitle, final String sectionItem) {
        String val = getValue(sectionTitle, sectionItem);
        if (val.length() <= 0) {
            return new String[0];
        }
        String[] res = val.split(" ");
        for (int i = 0; i < res.length; i++) {
            res[i] = res[i].replaceAll("<space>", " ");
        }
        return res;
    }

    public String getMultilineValues(final String sectionTitle, final String sectionItem, final String delimiter) {
        return StringUtils.stringArrayToString(getValues(sectionTitle, sectionItem), delimiter);
    }

    public void setMultilineValues(final String sectionTitle, final String sectionItem, final String values, final String delimiter) {
        setValue(sectionTitle, sectionItem, values.replaceAll(" ", "<space>").replace(delimiter, " ").trim());
    }

    public int[] getIntValues(final String sectionTitle, final String sectionItem) {
        return StringUtils.stringToIntArray(getValue(sectionTitle, sectionItem));
    }

    public void setIntValues(final String sectionTitle, final String sectionItem, final int[] values) {
        setValue(sectionTitle, sectionItem, StringUtils.intArrayToString(values));
    }

    public boolean[] getBooleanValues(final String sectionTitle, final String sectionItem) {
        return StringUtils.stringToBooleanArray(getValue(sectionTitle, sectionItem));
    }

    public void setBooleanValues(final String sectionTitle, final String sectionItem, final boolean[] values) {
        setValue(sectionTitle, sectionItem, StringUtils.booleanArrayToString(values));
    }

    public void removeValue(final String sectionTitle, final String sectionItem) {
        if (sections.containsKey(sectionTitle)) {
            Section sec = sections.get(sectionTitle);
            sec.items.remove(sectionItem);
            if (sec.items.isEmpty()) {
                sections.remove(sectionTitle);
            }
        }
    }

    public void removeSection(final String title) {
        sections.remove(title);
    }

    public void updateAndExtendWithValuesFrom(final Configuration conf) {
        // transfer 'old' configuration data from conf
        // to equivalent new settings
        
        // 0.63 -> 0.65
        conf.transferSetting(SBLASTER, "type", SBLASTER, SBTYPE, this);
        conf.transferSetting(SBLASTER, BASE, SBLASTER, SBBASE, this);
        conf.transferSetting(GUS, "rate", GUS, GUSRATE, this);
        conf.transferSetting(GUS, BASE, GUS, GUSBASE, this);
        
        // 0.65 -> 0.70
        conf.transferSetting("bios", JOYSTICKTYPE, JOYSTICK, JOYSTICKTYPE, this);
        
        // machine from old to new style
        if (this.hasValue(CPU, CPUTYPE) && conf.getValue(DOSBOX, MACHINE).equalsIgnoreCase("vga")) {
            conf.removeValue(DOSBOX, MACHINE);
        }
        // machine from new to old style
        if (conf.hasValue(CPU, CPUTYPE) && !this.hasValue(CPU, CPUTYPE)) {
            String mach = conf.getValue(DOSBOX, MACHINE);
            if (!(mach.equalsIgnoreCase("cga") || mach.equalsIgnoreCase("hercules") ||
                  mach.equalsIgnoreCase("pcjr") || mach.equalsIgnoreCase("tandy"))) {
                conf.removeValue(DOSBOX, MACHINE);
            }
        }
        
        for (String sectionTitle: conf.sections.keySet()) {
            for (String sectionItem: conf.getItems(sectionTitle)) {
                setValue(sectionTitle, sectionItem, conf.getValue(sectionTitle, sectionItem));
            }
        }
    }
    
    private void transferSetting(final String oldSection, final String oldItem,
            final String newSection, final String newItem, final Configuration newConf) {
        if (hasValue(oldSection, oldItem) && !newConf.hasValue(oldSection, oldItem) && newConf.hasValue(newSection, newItem)) {
            setValue(newSection, newItem, getValue(oldSection, oldItem));
            removeValue(oldSection, oldItem);
        } else if (hasValue(newSection, newItem) && !newConf.hasValue(newSection, newItem) && newConf.hasValue(oldSection, oldItem)) {
            setValue(oldSection, oldItem, getValue(newSection, newItem));
            removeValue(newSection, newItem);
        }
    }

    public void strip(final Configuration conf) {
        Configuration newConf = new Configuration();
        for (String key: sections.keySet()) {
            if (conf.sections.containsKey(key)) {
                for (String item: sections.get(key).items.keySet()) {
                    if (conf.sections.get(key).items.containsKey(item)) {
                        newConf.setValue(key, item, sections.get(key).items.get(item));
                    }
                }
            }
        }
        this.sections = newConf.sections;
    }
    
    public void setFixedCycles() {
        if (hasValue(CPU, CPUTYPE) && hasValue(CPU, CYCLES)) { // DOSBox v0.73 or later
            String cycles = getValue(CPU, CYCLES);
            if (! (cycles.contains("auto") || cycles.contains("max")) ) {
                setValue(CPU, CYCLES, "fixed " + cycles);
            }
        }
    }
    
    public void unsetFixedCycles() {
        if (hasValue(CPU, CYCLES)) {
            String cycles = this.getValue(CPU, CYCLES);
            if (cycles.startsWith("fixed ")) {
                setValue(CPU, CYCLES, cycles.substring(6));
            }
        }
    }
    
    public String toString(final boolean ordered, final Configuration baseConf) {
        StringBuffer result = new StringBuffer();
        for (String key: sections.keySet()) {
            result.append("[" + key + "]" + PlatformUtils.EOLN);
            result.append((sections.get(key)).toString(ordered)).append(PlatformUtils.EOLN);
        }
        String auto;
        if (baseConf == null) {
            auto = autoexec.getString(null);
        } else {
            auto = autoexec.getString(baseConf.autoexec.mountingpoints);
        }
        if (auto.length() > 0) {
            result.append("[autoexec]").append(PlatformUtils.EOLN).append(auto).append(PlatformUtils.EOLN);
        }
        return result.toString();
    }
    
    protected Section createSection(final String sectionTitle) {
        if (sections.containsKey(sectionTitle)) {
            return sections.get(sectionTitle);
        }
        Section newSection = new Section();
        sections.put(sectionTitle, newSection);
        return newSection;
    }
    
    public void substract(final Configuration conf) {    
        for (String key: conf.sections.keySet()) {
            if (sections.containsKey(key)) {
                Section mysec = sections.get(key);
                mysec.substract(conf.sections.get(key));
                if (mysec.items.isEmpty()) {
                    sections.remove(key);
                }
            }
        }
        //autoexec.substract(conf.autoexec);
    }

    public boolean hasParsingProblems() {
        return parsingProblems.length() > 0;
    }

    public final String getParsingProblems() {
        String result = parsingProblems.toString();
        parsingProblems = new StringBuffer();
        return result;
    }

    public File getMainDir() {
        if (isBooter()) {
            return new File(autoexec.img1).getParentFile();
        } else {
            return new File(autoexec.main).getParentFile();
        }
    }

    public void makeRelative(final File basePath) {
        for (Mount mount: autoexec.mountingpoints) {
            mount.makeRelative(basePath);
        }
        if (isBooter()) {
            autoexec.img1 = FileUtils.makeRelativeTo(new File(autoexec.img1), basePath).getPath();
            autoexec.img2 = FileUtils.makeRelativeTo(new File(autoexec.img2), basePath).getPath();
        } else {
            autoexec.main = FileUtils.makeRelativeTo(new File(autoexec.main), basePath).getPath();
        }
    }

    public String[] addMount(final String mount) {
        autoexec.addMount(mount);
        return getMountingpoints();
    }
    
    public void addMounts(final Configuration conf) {
        for (Mount m: conf.autoexec.mountingpoints) {
            autoexec.mountingpoints.add(m);
        }
    }

    public String[] removeMount(final int index) {
        autoexec.mountingpoints.remove(index);
        return getMountingpoints();
    }

    public String[] toggleMount(final int index) {
        autoexec.mountingpoints.get(index).toggleMount();
        return getMountingpoints();
    }
    
    public void unMount(final int index) {
        Mount mnt = autoexec.mountingpoints.get(index);
        if (!mnt.isUnmounted()) {
            mnt.toggleMount();
        }
    }

    public String[] replaceMount(final int index, final String mount) {
        try {
            autoexec.mountingpoints.set(index, new Mount(mount));
        } catch (InvalidMountstringException e) {
            // nothing we can do
        }
        return getMountingpoints();
    }
    
    public int getNrOfMounts() {
        return autoexec.mountingpoints.size();
    }
    
    public String getRequiredMount(final boolean booter, final String main) {
        if (autoexec.convertToDosboxPath(main)[0].length() > 0) {
            return null;
        }
        try {
            return new Mount(booter, main, getMountingpoints()).getPathAsString();
        } catch (Exception e) {
        	// this is not entirely correct; returning null assumes no mounts required
        	// but this should never happen anyway
            return null;
        }
    }
    
    public String[] addRequiredMount(final boolean booter, final String main) {
        try {
            return addMount(new Mount(booter, main, getMountingpoints()).toString());
        } catch (Exception e) {
            // could not add, just return previous values
            return getMountingpoints();
        }
    }
    
    public boolean isExit() {
        return autoexec.exit;
    }
    
    public boolean isLoadfix() {
        return autoexec.loadfix;
    }

    public String getLoadfixValue() {
        return autoexec.loadfixValue > 0 ? String.valueOf(autoexec.loadfixValue): "";
    }
    
    public String getKeyb() {
        return autoexec.keyb;
    }
    
    public String getMixer() {
        return autoexec.mixer;
    }
    
    public String[] getMountingpoints() {
        String[] result = new String[autoexec.mountingpoints.size()];
        int mountIndex = 0;
        for (Mount mount: autoexec.mountingpoints) {
            result[mountIndex++] = mount.toString(true);
        }
        return result;
    }
    
    public String getImg1() {
        if ("file".equals(autoexec.img1)) {
            return "";
        }
        return autoexec.img1;
    }

    public String getImg2() {
        return autoexec.img2;
    }

    public String getMain() {
        return autoexec.main;
    }
    
    public boolean isBooter() {
        return autoexec.img1.length() > 0;
    }

    public String getMainParameters() {
        return autoexec.params;
    }
    
    public boolean isIncomplete() {
        return (autoexec.img1.length() == 0 && autoexec.main.length() == 0);
    }
    
    public void setAutoexecSettings(final boolean exit) {
        autoexec.exit = exit;
    }
    
    public void setAutoexecSettings(final boolean exit, final boolean booter) {
        autoexec.exit = exit;
        autoexec.img1 = booter? "file": "";  // for templates
    }
    
    public void setAutoexecSettings(final boolean exit, final String mixer, final String keyb, final boolean booter) {
        setAutoexecSettings(exit, booter);
        autoexec.mixer = mixer;
        autoexec.keyb = keyb;
    }
    
    public void setAutoexecSettings(final boolean loadfix, final String loadfixValue, final String main, final String params,
            final String img1, final String img2) {
        autoexec.loadfix = loadfix;
        try {
            autoexec.loadfixValue = Integer.parseInt(loadfixValue);
        } catch (NumberFormatException e) {
            // no change
        }
        autoexec.main = main;
        autoexec.params = params;
        autoexec.img1 = img1;
        autoexec.img2 = img2;
    }
}
