package com.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import SevenZip.MyRandomAccessFile;
import SevenZip.Archive.IInArchive;
import SevenZip.Archive.SevenZipEntry;
import SevenZip.Archive.SevenZip.Handler;
import com.gui.StreamGobbler;
import com.model.DosboxVersion;
import com.model.Profile;
import com.model.Settings;
import com.model.conf.Configuration;


public final class FileUtils {

    private static final File DATA_DIR_FILE;
    private static final File DOSROOT_DIR_FILE;
    private static final File DOSBOX_DIR_FILE;
    private static final String TEMPLATES_DIR = "templates" + File.separatorChar;
    private static final String SETUP_CONF = "setup.conf";

    public static final String PROFILES_XML = "profiles.xml";
    public static final String DOSROOT_DIR = "dosroot" + File.separatorChar;
    public static final String CAPTURES_DIR = "captures" + File.separatorChar;
    public static final String PROFILES_DIR = "profiles" + File.separatorChar;
    public static final String CONF_EXT = ".conf";
    public static final String[] EXECUTABLES = { ".exe", ".com", ".bat" };
    public static final String[] ARCHIVES = { ".zip", ".7z" };
    public static final String[] BOOTERIMAGES = { ".cp2", ".dcf", ".img", ".jrc", ".td0" };

    public static final String DOSBOX_CONF = "dosbox.conf";
    public static final String CNF_FILTER = "*.conf;*.CONF";
    public static final String EXE_FILTER = "*.com;*.COM;*.exe;*.EXE;*.bat;*.BAT";
    public static final String ARC_FILTER = "*.zip;*.ZIP;*.7z;*.7Z";
    public static final String BTR_FILTER = "*.cp2;*.CP2;*.dcf;*.DCF;*.img;*.IMG;*.jrc;*.JRC;*.td0;*.TD0";
    public static final String CDI_FILTER = "*.iso;*.ISO;*.cue;*.CUE";
    public static final String ALL_FILTER = "*";


    static {
        final Settings settings = Settings.getInstance();
        final String DATADIR = getCanonicalPath(settings.getValue("directory", "data"));
        DATA_DIR_FILE = new File(DATADIR);
        DOSROOT_DIR_FILE = new File(DATADIR, DOSROOT_DIR);
        final String DOSBOXDIR = getCanonicalPath(settings.getValue("directory", "dosbox"));
        DOSBOX_DIR_FILE = new File(DOSBOXDIR);

        createDosrootIfNecessary();
    }

    public static String getCanonicalPath(final String path) {
        String canPath = path;
        if ((PlatformUtils.IS_LINUX || PlatformUtils.IS_OSX) &&
                (canPath.startsWith("~/") || (canPath.length() == 1 && canPath.charAt(0) == '~'))) {
            //Linux and OSX have ~/ for homedirectory. Allow single ~ as well
            canPath = canPath.replaceAll("^~", System.getProperty("user.home"));
        }
        try {
            return new File(canPath).getCanonicalPath();
        } catch (IOException e) {
            return new File(canPath).getAbsolutePath();
        }
    }

    private static boolean isExistingFile(final File file) {
        return file.isFile() && file.exists();
    }

    private static void copy(final File source, final File dest) {
        FileChannel inStream = null, outStream = null;
        try {          
            inStream = new FileInputStream(source).getChannel();
            outStream = new FileOutputStream(dest).getChannel();
            inStream.transferTo(0, inStream.size(), outStream);
        } catch (IOException e) {
            System.err.println("Couldn't copy file " + source + " to directory " + dest + "!!\n");
        } finally {
            if (inStream != null) { 
                try {
                    inStream.close();
                } catch (IOException ex) {
                    System.err.println("Couldn't close inputStream for " + source);
                }
            }
            if (outStream != null) {
                try {
                    outStream.close();
                } catch (IOException ex) {
                    System.err.println("Couldn't close outputStream for " + source);
                }
            }
        }
    }

    private static void executeCommand(final List<String> execCommands, final File cwd,
            final boolean waitFor) throws IOException {
        StringBuffer cmd = new StringBuffer();
        try {
            if (Settings.getInstance().getBooleanValue("dosbox", "hideconsole")) {
                execCommands.add("-noconsole");
            }
            String[] rtCmds = new String[execCommands.size()];
            for (int i = 0; i < execCommands.size(); i++) {
                rtCmds[i] = execCommands.get(i);
                cmd.append(rtCmds[i]).append(' ');
            }
            System.out.println(cmd);
            File dir = (cwd == null)? DOSROOT_DIR_FILE: cwd;
            Process proc = Runtime.getRuntime().exec(rtCmds, null, dir);
            StreamGobbler errorGobbler = new StreamGobbler(proc.getErrorStream(), "DOSBox stderr");            
            StreamGobbler outputGobbler = new StreamGobbler(proc.getInputStream(), "DOSBox stdout");
            outputGobbler.start();
            errorGobbler.start();
            if (waitFor) {
                try {	proc.waitFor();} catch (InterruptedException e) {}
            }
        } catch (IOException e) {
            e.printStackTrace();
            throw new IOException("Dosbox could not be started with the following command:\n" + cmd);
        }
    }

    private static void doRunDosbox(final DosboxVersion dbversion, final List<String> parameters,
            final File cwd, final boolean waitFor) throws IOException {
        List<String> commandItems = new ArrayList<String>();
        commandItems.add(dbversion.getCanonicalExecutable().getPath());
        commandItems.addAll(parameters);
        if (dbversion.getParameters().length() > 0) {
            for (String p: dbversion.getParameters().split(" ")) {
                commandItems.add(p);
            }
        }
        executeCommand(commandItems, cwd, waitFor);
    }
  
    private static File canonicalTo(final File base, final String path) {
        File file = new File(path);
        if (file.isAbsolute()) {
            return file;
        } else {
            try {
                return new File(base, file.getPath()).getCanonicalFile();
            } catch (IOException e) {
                return new File(base, file.getPath()).getAbsoluteFile();
            }
        }
    }

    public static boolean areRelated(final File parent, final File child) {
        File remainder = child.getParentFile();
        while (remainder != null) {
            if (parent.equals(remainder)) {
                return true;
            }
            remainder = remainder.getParentFile();
        }
        return false;
    }
  
    public static String getDosRoot() {
        return DOSROOT_DIR_FILE.getPath();
    }

    public static File makeRelativeTo(final File file, final File basePath) {
        if (!file.isAbsolute()) {
            return file;
        }
        if (file.equals(basePath)) {
            return new File(".");
        }
        File remainder = new File(file.getName());
        File parent = file.getParentFile();
        while (parent != null) {
            if (parent.equals(basePath)) {
                return remainder;
            }
            remainder = new File(parent.getName(), remainder.getPath());
            parent = parent.getParentFile();
        }
        return file;
    }
  
    public static File makeRelativeToData(final File file) {
        return makeRelativeTo(file, DATA_DIR_FILE);
    }

    public static File makeRelativeToDosroot(final File file) {
        return makeRelativeTo(file, DOSROOT_DIR_FILE);
    }

    public static File makeRelativeToDosbox(final File file) {
        return makeRelativeTo(file, DOSBOX_DIR_FILE);
    }

    public static File canonicalToData(final String path) {
        return canonicalTo(DATA_DIR_FILE, path);
    }

    public static File canonicalToDosbox(final String path) {
        return canonicalTo(DOSBOX_DIR_FILE, path);
    }

    public static File canonicalToDosroot(final String path) {
        return canonicalTo(DOSROOT_DIR_FILE, path);
    }

    public static String sanitizeToDosroot(final String path) {
        return makeRelativeToDosroot(canonicalToDosroot(path)).getPath();
    }
  
    public static File constructCanonicalDBConfLocation(final String path) {
        return canonicalToDosbox(new File(path, DOSBOX_CONF).getPath());
    }

    public static File constructCanonicalDBExeLocation(final String path) {
        return canonicalToDosbox(new File(path, PlatformUtils.DB_EXECUTABLE).getPath());
    }

    public static String constructCapturesDir(final int profileId) {
        return CAPTURES_DIR + profileId;
    }

    public static String constructRelativeCapturesDir(final int profileId) {
        return ".." + File.separatorChar + constructCapturesDir(profileId);
    }

    public static File constructCanonicalTemplateFileLocation(final int templateId) {
        return canonicalToData(TEMPLATES_DIR + templateId + CONF_EXT);
    }

    public static String constructUniqueConfigFileString(final int profileId, final String profileTitle, final Configuration gameConf) {
        Settings set = Settings.getInstance();
        File candidate = null;
        int candidateNr = 1;
        boolean unique = false;
        while (!unique) {
            File candidatePath = null;
            if (set.getIntValue("profiledefaults", "confpath") == 1) {
                File gameRoot = gameConf.getMainDir();
                if (gameRoot != null) {
                    if (gameRoot.isAbsolute()) {
                        candidatePath = gameRoot;
                    } else {
                        candidatePath = new File(DOSROOT_DIR, gameRoot.getPath());
                    }
                }
            } else {
                candidatePath = new File(PROFILES_DIR);
            }

            String candidateFile = null;
            if (set.getIntValue("profiledefaults", "conffile") == 1) {
                String candidateTitle = profileTitle;
                if (candidateNr > 1) {
                    candidateTitle += "(" + candidateNr + ")";
                }
                candidateFile = candidateTitle.replaceAll("[^a-zA-Z_0-9()]", "") + CONF_EXT;
            } else {
                if (candidateNr > 1) {
                    return null; // id files should be unique, else give up
                }
                candidateFile = profileId + CONF_EXT;
            }
            candidateNr++;
            candidate = new File(candidatePath, candidateFile);
            unique = !isExistingFile(candidate);
        }
        return candidate.getPath();
    }

    public static void doRunDosbox(final DosboxVersion dbversion) throws IOException {
        List<String> parameters = new ArrayList<String>();
        parameters.add("-conf");
        parameters.add(dbversion.getCanonicalConfFile().getPath());
        doRunDosbox(dbversion, parameters, null, false);
    }

    public static void doCreateDosboxConf(final DosboxVersion dbversion) throws IOException {
        List<String> parameters = new ArrayList<String>();
        parameters.add("-c");
        parameters.add("config -writeconf " + DOSBOX_CONF);
        parameters.add("-c");
        parameters.add("exit");
        doRunDosbox(dbversion, parameters, dbversion.getCanonicalExecutable().getParentFile(), true);
    }
  
    public static void doRunProfile(final Profile prof, final List<DosboxVersion> dbversions,
            final boolean setup) throws IOException {
        DosboxVersion dbv = dbversions.get(DosboxVersion.findById(dbversions, prof.getDbversionId()));
        doRunProfile(prof, dbv, setup);
    }

    public static void doRunProfile(final Profile prof, final DosboxVersion dbversion,
            final boolean setup) throws IOException {
        List<String> confs = new ArrayList<String>();
        confs.add(dbversion.getCanonicalExecutable().getPath());
        if (dbversion.isMultiConfig()) {
            // selected default dosbox config file
            confs.add("-conf");
            confs.add(dbversion.getCanonicalConfFile().getPath()); 
        }
        // add profile-specific settings
        File file = prof.getCanonicalConfFile();
        if (setup && prof.hasSetup()) {
            Configuration conf = new Configuration(file, prof.getSetup(), prof.getSetupParameters());
            file = canonicalToData(PROFILES_DIR + SETUP_CONF);
            conf.saveToFile(file, false, null);
        }
        confs.add("-conf");
        confs.add(file.getPath());
        if (dbversion.getParameters().length() > 0) {
            for (String p: dbversion.getParameters().split(" ")) {
                confs.add(p);
            }
        }
        executeCommand(confs, null, false);
    }
  
    public static void createDir(final File dir) {
        if (!dir.exists() && !dir.mkdirs()) {
            System.err.println("Couldn't create " + dir + " directory!!\n");
        }
    }

    public static void copyFiles(final File srcDir, final File dstDir) {
        File[] srcFiles = srcDir.listFiles();
        if (srcFiles != null) {
            for (File src: srcFiles) {
                if (src.isFile()) {
                    File dst = new File(dstDir, src.getName());
                    copy(src, dst);
                    dst.setLastModified(src.lastModified());
                }
            }
        }
    }

    public static void removeFile(final File file) {
        if (! (file.isFile() && file.delete()) ) {
            System.err.println("Couldn't delete file " + file.getPath());
        }      
    }

    public static void removeFilesInDirAndDir(final File dir) {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file: files) {
                if (file.isDirectory()) {
                    System.err.println("Directory found inside " + dir.getPath() + ", aborting removal");
                    return;
                }
            }
            for (File file: files) {
                if (!file.delete()) {
                    System.err.println("Couldn't delete file " + file);
                }
            }
        }
        if (! (dir.isDirectory() && dir.delete()) ) {
            System.err.println("Couldn't delete directory " + dir.getPath());
        }
    }
  
    public static boolean isReadableFile(final File file) {
        return file.isFile() && file.canRead();
    }

    public static void createDosrootIfNecessary() {
        if (!DOSROOT_DIR_FILE.isDirectory()) {
            System.out.print("Creating new directory '" + DOSROOT_DIR_FILE.getPath() + "' ... ");
            if (DOSROOT_DIR_FILE.mkdirs()) {
                System.out.println("Ok");
            } else {
                System.out.println("Failure");
            }
        }
    }

    public static boolean isExecutable(final String filename) {
        for (String ext: EXECUTABLES) {
            if (filename.toLowerCase().endsWith(ext)) {
                return true;
            }
        }
        return false;
    }
  
    public static boolean isArchive(final String filename) {
        for (String ext: ARCHIVES) {
            if (filename.toLowerCase().endsWith(ext)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isPhysFS(final String mountPath) {
        for (String ext: ARCHIVES) {
            if (mountPath.toLowerCase().endsWith(ext + ':' + File.separatorChar)) {
                return true;
            }
        }
        return false;
    }
  
    public static boolean containsPhysFS(final String mountPath) {
        for (String ext: ARCHIVES) {
            if (mountPath.toLowerCase().contains(ext + ':' + File.separatorChar)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isBooterImage(final String filename) {
        for (String ext: BOOTERIMAGES) {
            if (filename.toLowerCase().endsWith(ext)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isConfFile(final String filename) {
        return filename.toLowerCase().endsWith(CONF_EXT);
    }
  
    public static String[] getExecutablesInZip(final String archive) throws IOException {
        List<String> result = new ArrayList<String>();
        File arcFile = new File(archive);

        if (archive.toLowerCase().endsWith(ARCHIVES[0])) { // zip
            ZipFile zfile = new ZipFile(arcFile);
            for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
                ZipEntry entry = entries.nextElement();
                String name = entry.getName();
                if (!entry.isDirectory() && isExecutable(name)) {
                    result.add(PlatformUtils.archiveToNativePath(name));
                }
            }
        } else if (archive.toLowerCase().endsWith(ARCHIVES[1])) { // 7-zip
            MyRandomAccessFile istream = new MyRandomAccessFile(archive, "r");
            IInArchive zArchive = new Handler();
            if (zArchive.Open(istream) != 0) {
                throw new IOException("Failure opening 7-zip archive");
            }
            for (int i = 0; i < zArchive.size(); i++) {
                SevenZipEntry entry = zArchive.getEntry(i);
                String name = entry.getName();
                if (!entry.isDirectory() && isExecutable(name)) {
                    result.add(PlatformUtils.archiveToNativePath(name));
                }
            }
        }

        Collections.sort(result, new Comparator<String>() {
            public int compare(final String string1, final String string2) {
                int count1 = StringUtils.countCharacters(string1, '\\');
                int count2 = StringUtils.countCharacters(string2, '\\');
                if (count1 == count2) {
                    return string1.compareTo(string2);
                } else {
                    return count1 - count2;
                }
            }
        });

        return result.toArray(new String[result.size()]);
    }
}
