package com.gui;

import java.io.File;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import com.model.Settings;
import com.util.FileUtils;
import com.util.StringUtils;


public final class BrowseButton {

    public static final String DFEND_PATH = "C:\\Program Files\\D-Fend\\";
    public static final String DFEND_PROFILES = "Profiles.dat";

    public enum BrowseType { DIR, FILE };
    public enum CanonicalType { DOSROOT, DFEND, CDIMAGE, ZIP, DOSBOX, DOC, BOOTER, EXE, NONE };

    private final Button button;
    private Settings settings;


    private class Sa extends SelectionAdapter {

        private final Text field;
        private final Text alt;
        private final BrowseType browse;
        private final CanonicalType canon;
        private final Combo combo;
        private final Shell shell;
        

        public Sa(final Shell shell, final Text control, final Text altControl, final BrowseType browse,
                final CanonicalType canon, final Combo combo) {
            this.shell = shell;
            this.field = control;
            this.alt = altControl;
            this.browse = browse;
            this.canon = canon;
            this.combo = combo;
        }

        private String filterPath() {
            String path = field.getText();
            CanonicalType fileType = canon;
            if (path.equals("") && (alt != null)) {
                path = alt.getText();
                fileType = CanonicalType.DOSROOT;
            }
            switch (fileType) {
                case DOC:
                    return FileUtils.canonicalToData(path).getPath();
                case EXE:
                case ZIP:
                case BOOTER:
                case DOSROOT:
                    return FileUtils.canonicalToDosroot(path).getPath();
                case DOSBOX:
                    return FileUtils.canonicalToDosbox(path).getPath();
                case DFEND:
                    return DFEND_PATH;
                case CDIMAGE:
                    String[] fPaths = StringUtils.textAreaToStringArray(field.getText(), field.getLineDelimiter());
                    if (fPaths.length > 0) {
                        return fPaths[0];
                    }
                default:
                    return "";
            }
        }

        private String getResult(final String result, final File filterPath, final String[] filenames) {
            switch (canon) {
                case DOC:
                    return FileUtils.makeRelativeToData(new File(result)).getPath();
                case EXE:
                    if (FileUtils.isArchive(result) || FileUtils.isIsoFile(result)) {
                        BrowseArchiveDialog dialog = new BrowseArchiveDialog(shell, SWT.OPEN);
                        dialog.setFileToBrowse(result);
                        return (String) dialog.open();
                    }
                case ZIP:
                case BOOTER:
                case DOSROOT:
                    return FileUtils.makeRelativeToDosroot(new File(result)).getPath();
                case DOSBOX:
                    return FileUtils.makeRelativeToDosbox(new File(result)).getPath();
                case CDIMAGE:
                    File path = FileUtils.makeRelativeToDosroot(filterPath);
                    StringBuffer images = new StringBuffer();
                    for (String file: filenames) {
                        images.append(new File(path, file)).append(field.getLineDelimiter());
                    }
                    return images.toString();
                default:
                    return result;
            }
        }

        public void widgetSelected(final SelectionEvent event) {
            String result = null;
            File filterPath = null;
            String[] filenames = null;

            if (browse == BrowseType.DIR) {
                DirectoryDialog dialog = new DirectoryDialog(shell);
                dialog.setFilterPath(filterPath());
                result = dialog.open();
            } else if (browse == BrowseType.FILE && (canon == CanonicalType.EXE) && 
                    (FileUtils.isArchive(filterPath()) || FileUtils.isIsoFile(filterPath()))) {
                result = filterPath();
            } else if (browse == BrowseType.FILE) {
                int style = (canon == CanonicalType.CDIMAGE) ? SWT.OPEN | SWT.MULTI: SWT.OPEN;
                FileDialog dialog = new FileDialog(shell, style);

                String[] filterNames = null;
                String[] filterExts = null;
                switch (canon) {
                    case DOC:
                        filterNames = new String[] { FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.ALL_FILTER };
                        break;
                    case EXE:
                        String defFilterTitle = settings.msg("filetype.exe");
                        String defaultFilter = FileUtils.EXE_FILTER;
                        if (combo.isEnabled()) { // check to see if this dbversion support physfs (Gulikoza/Ykhwong)
                            defFilterTitle += ", " + settings.msg("filetype.archive");
                            defaultFilter += ";" + FileUtils.ARC_FILTER;
                        }
                        filterNames = new String[] { defFilterTitle, settings.msg("filetype.cdimage"), settings.msg("filetype.archive") };
                        filterExts = new String[] { defaultFilter, FileUtils.CDI_FILTER, FileUtils.ARC_FILTER };
                        break;
                    case ZIP:
                        filterNames = new String[] { settings.msg("filetype.archive"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.ARC_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case BOOTER:
                        filterNames = new String[] { settings.msg("filetype.booterimage"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.BTR_FILTER, FileUtils.ALL_FILTER };
                        break;
                    case DFEND:
                        filterNames = new String[] { settings.msg("filetype.dfendprofile") };
                        filterExts = new String[] { DFEND_PROFILES };
                        break;
                    case CDIMAGE:
                        filterNames = new String[] { settings.msg("filetype.cdimage"), FileUtils.ALL_FILTER };
                        filterExts = new String[] { FileUtils.CDI_FILTER, FileUtils.ALL_FILTER };
                        break;
                    default:
                }
                if (filterNames != null) {
                    dialog.setFilterNames(filterNames);
                }
                if (filterExts != null) {
                    dialog.setFilterExtensions(filterExts);
                }

                if (canon == CanonicalType.DFEND) {
                    dialog.setFileName(DFEND_PROFILES);
                }

                dialog.setFilterPath(filterPath());

                result = dialog.open();
                if (canon == CanonicalType.CDIMAGE) {
                    filterPath = new File(dialog.getFilterPath());
                    filenames = dialog.getFileNames();
                }
            }
            if (result != null) {
                result = getResult(result, filterPath, filenames);
                if (result != null) {
                    field.setText(result);
                }
            }
        }
    }


    public BrowseButton(final Composite composite, final int style) {
        settings = Settings.getInstance();
        button = new Button(composite, style);
        button.setText(settings.msg("button.browse"));
    }

    public void connect(final Shell shell, final Text control, final Text altControl, final BrowseType browse, final CanonicalType canon, final Combo combo) {
        button.addSelectionListener(new Sa(shell, control, altControl, browse, canon, combo));
    }

    public void setLayoutData(final Object arg0) {
        button.setLayoutData(arg0);
    }

    public void setEnabled(final boolean enabled) {
        button.setEnabled(enabled);
    }
}
