package com.loopy.impl;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import com.loopy.iface.FileEntry;
import com.loopy.iface.VolumeDescriptorSet;

/**
 * A block file system is segmented into multiple fixed-size blocks. It consists of a set of volume
 * descriptors followed by an index, which points to the file locations.
 */
public abstract class AbstractBlockFileSystem extends AbstractFileSystem {
    
    private final int blockSize;
    private final int reservedBlocks;
    private VolumeDescriptorSet volumeDescriptorSet;

    
    protected AbstractBlockFileSystem(final File file, final int blockSize, final int reservedBlocks) throws IOException {
        super(file);
        if (blockSize <= 0) {
            throw new IllegalArgumentException("'blockSize' must be > 0");
        }
        if (reservedBlocks < 0) {
            throw new IllegalArgumentException("'reservedBlocks' must be >= 0");
        }
        this.blockSize = blockSize;
        this.reservedBlocks = reservedBlocks;
        loadVolumeDescriptors();
    }

    public Enumeration<FileEntry> getEntries() {
        ensureOpen();
        return enumerate(volumeDescriptorSet.getRootEntry());
    }

    protected void loadVolumeDescriptors() throws IOException {
        final byte[] buffer = new byte[this.blockSize];
        volumeDescriptorSet = createVolumeDescriptorSet();
        // skip the reserved blocks, then read volume descriptor blocks sequentially and add them to the VolumeDescriptorSet
        for (int block = this.reservedBlocks; readBlock(block, buffer) && !volumeDescriptorSet.deserialize(buffer); block++);
    }

    /**
     * Read the data for the specified block into the specified buffer.
     *
     * @param block
     * @param buffer
     * @return if the block was actually read
     * @throws IOException if the number of bytes read into the buffer was less than the expected
     * number (i.e. the block size)
     */
    protected boolean readBlock(final long block, final byte[] buffer) throws IOException {
        final int bytesRead = readData(block * blockSize, buffer, 0, blockSize);
        if (bytesRead <= 0) {
            return false;
        }
        if (blockSize != bytesRead) {
            throw new IOException("Could not deserialize a complete block");
        }
        return true;
    }

    /**
     * Read file data, starting at the specified position.
     *
     * @param startPos
     * @param buffer
     * @param offset
     * @param len
     * @return the number of bytes read into the buffer
     * @throws IOException
     */
    protected synchronized int readData(final long startPos, final byte[] buffer, final int offset, final int len) throws IOException {
        seek(startPos);
        return read(buffer, offset, len);
    }

    protected VolumeDescriptorSet getVolumeDescriptorSet() {
        return volumeDescriptorSet;
    }

    /**
     * Returns an enumeration of the file entries starting at <code>root</code>.
     *
     * @param root
     * @return
     */
    protected abstract Enumeration<FileEntry> enumerate(FileEntry root);

    /**
     * Creates the VolumeDescriptorSet that deserializes volume descriptors for this file system.
     *
     * @return
     */
    protected abstract VolumeDescriptorSet createVolumeDescriptorSet();
}