package com.loopy.impl;

import java.io.RandomAccessFile;
import java.io.File;
import java.io.IOException;
import java.io.FileNotFoundException;
import com.loopy.exception.LoopyException;
import com.loopy.iface.FileSystem;


public abstract class AbstractFileSystem implements FileSystem {
    /**
     * The file containing the file system image.
     */
    private File file;
    /**
     * Channel to the open file.
     */
    private RandomAccessFile channel;

    
    protected AbstractFileSystem(File file) throws LoopyException {
        this.file = file;
        try {
            if (!file.exists()) {
                throw new FileNotFoundException("File does not exist: " + this.file);
            }
            channel = new RandomAccessFile(this.file, "r");
        } catch (IOException ex) {
            throw new LoopyException("Error opening the file", ex);
        }
    }

    public synchronized void close() throws LoopyException {
        if (isClosed()) {
            return;
        }
        try {
            channel.close();
        } catch (IOException ex) {
            throw new LoopyException("Error closing file system", ex);
        } finally {
            channel = null;
        }
    }

    public boolean isClosed() {
        return channel == null;
    }

    protected void ensureOpen() throws IllegalStateException {
        if (isClosed()) {
            throw new IllegalStateException("File has been closed");
        }
    }

    protected void seek(long pos) throws IOException {
        ensureOpen();
        channel.seek(pos);
    }

    protected int read(byte[] buffer, int bufferOffset, int len) throws IOException {
        ensureOpen();
        return channel.read(buffer, bufferOffset, len);
    }
}
