package com.loopy.iso9660;

import com.loopy.iface.FileEntry;
import com.loopy.util.Util;

/**
 * Represents a file in an ISO9660 file system.
 */
public final class ISO9660FileEntry implements FileEntry {
    
    public static final char ID_SEPARATOR = ';';

    private ISO9660FileSystem fileSystem;
    private String parentPath;
    private final int entryLength;
    private final long startSector;
    private final int dataLength;
    private final int flags;
    private final String identifier;


    public ISO9660FileEntry(final ISO9660FileSystem fileSystem, final byte[] block, final int pos) {
        this(fileSystem, null, block, pos);
    }

    /**
     * Initialize this instance.
     * @param fileSystem the parent file system
     * @param parentPath the path of the parent directory
     * @param block the bytes of the sector containing this file entry
     * @param startPos the starting position of this file entry
     */
    public ISO9660FileEntry(final ISO9660FileSystem fileSystem, final String parentPath, final byte[] block, final int startPos) {
        this.fileSystem = fileSystem;
        this.parentPath = parentPath;
        final int offset = startPos - 1;
        this.entryLength = Util.getUInt8(block, offset + 1);
        this.startSector = Util.getUInt32LE(block, offset + 3);
        this.dataLength = (int) Util.getUInt32LE(block, offset + 11);
        this.flags = Util.getUInt8(block, offset + 26);
        this.identifier = getFileIdentifier(block, offset, isDirectory());
    }

    private String getFileIdentifier(final byte[] block, final int offset, final boolean isDir) {
        final int fidLength = Util.getUInt8(block, offset + 33);
        if (isDir) {
            final int buff34 = Util.getUInt8(block, offset + 34);
            if ((fidLength == 1) && (buff34 == 0x00)) {
                return ".";
            } else if ((fidLength == 1) && (buff34 == 0x01)) {
                return "..";
            }
        }
        final String id = Util.getDChars(block, offset + 34, fidLength, fileSystem.getEncoding());
        final int sepIdx = id.indexOf(ID_SEPARATOR);
        if (sepIdx >= 0) {
            return id.substring(0, sepIdx);
        } else {
            return id;
        }
    }

    public String getName() {
        return identifier;
    }

    public String getPath() {
        if (".".equals(getName())) {
            return "";
        }
        StringBuffer buf = new StringBuffer();
        if (parentPath != null) {
            buf.append(parentPath);
        }
        buf.append(getName());
        if (isDirectory()) {
            buf.append("/");
        }
        return buf.toString();
    }

    public boolean isDirectory() {
        return (flags & 0x03) != 0;
    }

    int getSize() {
        return dataLength;
    }

    /**
     * Returns the block number where this entry starts.
     *
     * @return
     */
    long getStartBlock() {
        return startSector;
    }

    /**
     * Returns the size this entry takes up in the file table.
     *
     * @return
     */
    int getEntryLength() {
        return entryLength;
    }
}
