package com.model.conf;

import java.io.File;
import java.io.IOException;
import org.eclipse.swt.widgets.Shell;
import com.gui.GeneralPurposeDialogs;
import com.model.DosboxVersion;
import com.model.Profile;
import com.model.Settings;
import com.model.Template;
import com.util.FileUtils;


public class CompositeConfiguration {

    private Configuration baseConf;
    private Configuration compositeConf;
    private DosboxVersion dbversion;

    
    public CompositeConfiguration(final DosboxVersion dbversion, final boolean force, final Shell shell) {
        processConfiguration(dbversion.getCanonicalConfFile(), true, force, shell);
        this.dbversion = dbversion; 
    }
    
    public CompositeConfiguration(final DosboxVersion dbversion, final Profile profile, final boolean force, final Shell shell) {
        this(dbversion, profile.getCanonicalConfFile(), force, shell);
    }

    public CompositeConfiguration(final DosboxVersion dbversion, final Template template, final boolean force, final Shell shell) {
        this(dbversion, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), force, shell);
    }
    
    public CompositeConfiguration(final DosboxVersion dbversion, final File file, final boolean force, final Shell shell) {
        this(dbversion, force, shell);
        if (file != null) {
            processConfiguration(file, false, force, shell);
        }
    }
    
    public CompositeConfiguration(final CompositeConfiguration comp, final DosboxVersion dbversion, final Template template, final boolean force, final Shell shell) {
        this(dbversion, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), force, shell);
        if (baseConf != null && compositeConf != null) {
            Configuration otherAutoSettings = new Configuration(comp.compositeConf, false);
            this.compositeConf.setAutoexecSettings(
                    otherAutoSettings.isLoadfix(), otherAutoSettings.getLoadfixValue(),
                    otherAutoSettings.getMain(), otherAutoSettings.getMainParameters(),
                    otherAutoSettings.getImg1(), otherAutoSettings.getImg2());
        }
    }
    
    public CompositeConfiguration(final CompositeConfiguration comp, final DosboxVersion dbversion,
            final boolean keepValues, final boolean isTemplate, final Shell shell) {
        this.baseConf = comp.baseConf;
        this.compositeConf = comp.compositeConf;
        this.dbversion = comp.dbversion;
        
        Configuration uniqueMountPoints = null;
        Configuration otherAutoSettings = null;
        if (baseConf != null && compositeConf != null) {
            uniqueMountPoints = new Configuration(compositeConf, true, baseConf, this.dbversion != dbversion);
            otherAutoSettings = new Configuration(compositeConf, keepValues);
        }
        
        if (keepValues) {
            if (!update(dbversion, shell, isTemplate)) {
                this.compositeConf = null;
            }
        } else {
            processConfiguration(dbversion.getCanonicalConfFile(), true, false, shell);
            this.dbversion = dbversion;
        }
        
        if (compositeConf != null && otherAutoSettings != null) {
            if (keepValues) {
                this.compositeConf.setAutoexecSettings(otherAutoSettings.isExit(), otherAutoSettings.getMixer(),
                    otherAutoSettings.getKeyb(), otherAutoSettings.getIpxnet(), otherAutoSettings.isBooter());
            } else {
                this.compositeConf.setAutoexecSettings(otherAutoSettings.isExit(), otherAutoSettings.isBooter());
            }
            this.compositeConf.setAutoexecSettings(
                    otherAutoSettings.isLoadfix(), otherAutoSettings.getLoadfixValue(),
                    otherAutoSettings.getMain(), otherAutoSettings.getMainParameters(),
                    otherAutoSettings.getImg1(), otherAutoSettings.getImg2());
        }
        
        if (compositeConf != null && uniqueMountPoints != null && 
                ((this.dbversion != comp.dbversion) || !keepValues) ) {
            compositeConf.addMounts(uniqueMountPoints);
        }
    }
    
    private boolean update(final DosboxVersion dbversion, final Shell shell, final boolean isTemplate) {
        Configuration org = new Configuration(compositeConf, false, null, false);
        processConfiguration(dbversion.getCanonicalConfFile(), true, false, shell);
        if (baseConf == null) {
            if (GeneralPurposeDialogs.confirmMessage(shell, Settings.getInstance().msg( 
                    isTemplate? "general.confirm.missingdosboxconftemplate": "general.confirm.missingdosboxconfprofile"))) {
                processConfiguration(dbversion.getCanonicalConfFile(), true, true, shell);
            } else {
                return false;
            }
        }
        compositeConf.updateAndExtendWithValuesFrom(org);
        this.dbversion = dbversion;
        return true;
    }
    
    public void setCompositeConf(final Configuration configuration) {
        this.compositeConf = configuration;
    }
    
    private void save(final boolean substract, final File target, final Shell shell) {
        Configuration toSave = new Configuration(compositeConf, false, null, false);
        toSave.strip(baseConf);
        toSave.setFixedCycles();
        if (substract) {
            toSave.substract(baseConf);
        }
        try {
            toSave.saveToFile(target, false, baseConf);
        } catch (IOException ex) {
            GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
        }
    }
    
    public void save(final Template template, final Shell shell) {
        save(true, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), shell);
    }
    
    public void save(final Profile profile, final Shell shell) {
        compositeConf.setValue("dosbox", "captures", FileUtils.constructRelativeCapturesDir(profile.getId()));
        save((dbversion != null && dbversion.isMultiConfig()), profile.getCanonicalConfFile(), shell);
    }
    
    public String getFullConf() {
        return compositeConf.toString(false, baseConf);
    }
    
    public String getIncrConf() {
        Configuration toSave = new Configuration(compositeConf, false, null, false);
        toSave.strip(baseConf);
        toSave.setFixedCycles();
        if (dbversion != null && dbversion.isMultiConfig()) {
            toSave.substract(baseConf);
        }
        return toSave.toString(false, baseConf);
    }
    
    public File getExportGameDir(int index) {
        return new File("./" + index + "/");
    }

    public Configuration getBaseConf() {
        return baseConf;
    }

    public Configuration getCompositeConf() {
        return compositeConf;
    }
    
    public int getDbversionId() {
        return dbversion.getId();
    }
    
    private void init() {
        baseConf = null;
        compositeConf = null;
        dbversion = null;
    }
    
    private void processConfiguration(final File file, final boolean base, final boolean force, final Shell shell) {
        if (base) {
            try {
                init();
                baseConf = new Configuration(file, true);
                if (baseConf.hasParsingProblems()) {
                    GeneralPurposeDialogs.warningMessage(shell, baseConf.getParsingProblems());
                }
            } catch (IOException ex) {
                GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
                if (force) {
                    baseConf = new Configuration();
                }
            } finally {
                if (baseConf != null) {
                    compositeConf = new Configuration(baseConf, true, null, false);
                    compositeConf.unsetFixedCycles();
                }
            }
        } else {
            try {
                compositeConf.setAutoexecSettings(false);
                compositeConf.parseConfigurationFile(file, true);
                compositeConf.unsetFixedCycles();
            } catch (IOException ex) {
                GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
            }
            if (compositeConf.hasParsingProblems()) {
                GeneralPurposeDialogs.warningMessage(shell, compositeConf.getParsingProblems());
            }
        }
    }
    
    public String[] addMount(final String mount) {
        return compositeConf.addMount(mount);
    }
    
    public String[] removeMount(final int index) {
        if (index < baseConf.getNrOfMounts()) {
            return compositeConf.toggleMount(index);
        } else {
            return compositeConf.removeMount(index);
        }
    }
    
    public String[] editMount(final int index, final String mount) {
        if (index < baseConf.getNrOfMounts()) {
            compositeConf.unMount(index);
            return compositeConf.addMount(mount);
        } else {
            return compositeConf.replaceMount(index, mount);
        }
    }
    
    public String getRequiredMount(final boolean booter, final String exe) {
        return compositeConf.getRequiredMount(booter, exe);
    }
    
    public String[] addRequiredMount(final boolean booter, final String exe) {
        return compositeConf.addRequiredMount(booter, exe);
    }
}
