/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.dbgl.gui.MainWindow;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.ExpProfile;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;
import org.dbgl.model.conf.CompositeConfiguration;


public class XmlUtils {
    
    private final static String XML_FORMAT_VERSION = "1.0";
    

    public static Element addElement(Document dom, Element el, String name, String value) {
        Element newElement = dom.createElement(name);
        newElement.appendChild(dom.createTextNode(value));
        el.appendChild(newElement);
        return el;
    }

    public static Element addCDataElement(Document dom, Element el, String name, String value) {
        Element newElement = dom.createElement(name);
        newElement.appendChild(dom.createCDATASection(value));
        el.appendChild(newElement);
        return el;
    }

    public static void saveDomSource(DOMSource source, File target, File xslt) throws TransformerException, IOException {
        TransformerFactory transFact = TransformerFactory.newInstance();
        transFact.setAttribute("indent-number", 2);
        Transformer trans;
        if (xslt == null) {
            trans = transFact.newTransformer();
        } else {
            trans = transFact.newTransformer(new StreamSource(xslt));
        }
        trans.setOutputProperty(OutputKeys.INDENT, "yes");
        FileOutputStream fos = new FileOutputStream(target);
        trans.transform(source, new StreamResult(new OutputStreamWriter(fos, "utf-8")));
        fos.close();
    }
    
    public static Document getProfilesXML(List<Profile> profs, List<DosboxVersion> dbversions) throws ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document xmldoc = builder.newDocument();
        Element root = xmldoc.createElement("document");
        root.appendChild(getExportElement(xmldoc, Settings.getInstance().msg("exportlist.title"), "", false, false));

        for (Profile profile: profs) {
            ExpProfile expProf = new ExpProfile(profile.getId(), null, null, null, profile);
            Element profElement = expProf.getXml(xmldoc, dbversions);
            root.appendChild(profElement);
        }

        xmldoc.appendChild(root);
        return xmldoc;
    }
    
    public static Document getFullProfilesXML(List<Profile> profs, List<DosboxVersion> dbversions,
            String title, String notes, boolean captures, boolean gameData) throws ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document xmldoc = builder.newDocument();
        Element root = xmldoc.createElement("document");
        root.appendChild(getExportElement(xmldoc, title, notes, captures, gameData));

        int profileNumber = 0;
        for (Profile profile: profs) {
            int dbversionIndex = DosboxVersion.findById(dbversions, profile.getDbversionId());
            DosboxVersion dbversion = dbversions.get(dbversionIndex);
            CompositeConfiguration compConf = new CompositeConfiguration(dbversion, profile, true, null, System.out);
            ExpProfile expProf = new ExpProfile(profileNumber, compConf.getFullConf(),
                    compConf.getIncrConf(), compConf.getExportGameDir(profileNumber), profile);
            Element profElement = expProf.getXml(xmldoc, dbversions);
            root.appendChild(profElement);
            profileNumber++;
        }

        xmldoc.appendChild(root);
        return xmldoc;
    }

    private static Element getExportElement(Document xmldoc, String title, String notes, boolean captures, boolean gameData) {
        Element export = xmldoc.createElement("export");
        XmlUtils.addElement(xmldoc, export, "format-version", XML_FORMAT_VERSION);
        XmlUtils.addElement(xmldoc, export, "title", title);
        XmlUtils.addCDataElement(xmldoc, export, "notes", notes);
        XmlUtils.addElement(xmldoc, export, "generator-title", MainWindow.PROGRAM_NAME_FULL);
        XmlUtils.addElement(xmldoc, export, "generator-version", MainWindow.PROGRAM_VERSION);
        Settings settings = Settings.getInstance();
        for (int i = 0; i < MainWindow.EDIT_COLUMN_NAMES; i++) {
            String s = settings.getValue("gui", "custom" + (i + 1));
            XmlUtils.addElement(xmldoc, export, "custom" + (i + 1), s);
        }
        XmlUtils.addElement(xmldoc, export, "captures-available", String.valueOf(captures));
        XmlUtils.addElement(xmldoc, export, "gamedata-available", String.valueOf(gameData));
        return export;
    }
}
