package org.dbgl.gui;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.dbgl.gui.BrowseButton.BrowseType;
import org.dbgl.gui.BrowseButton.CanonicalType;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.Profile;
import org.dbgl.util.ExportThread;
import org.dbgl.util.FileUtils;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import swing2swt.layout.BorderLayout;


public class ExportDialog extends Dialog {

    private Text logText;
    private Text filename;
    private Label profileLabel;
    private Button cancelButton;
    protected Shell shell;

    private ExportThread exportThread;
    private Profile[] profs;
    private List<DosboxVersion> dbversionsList;


    public ExportDialog(Shell parent, int style) {
        super(parent, style);
    }

    public ExportDialog(Shell parent, List<DosboxVersion> dbList, Profile[] p) {
        this(parent, SWT.NONE);
        dbversionsList = dbList;
        profs = p;
    }

    public Object open() {
        createContents();
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (exportThread != null && !exportThread.isAlive() && !cancelButton.getText().equals("Close")) {
                GeneralPurposeDialogs.infoMessage(shell, "Export completed successfully.");
                profileLabel.setText("Please review the log below for possible problems");
                profileLabel.pack();
                cancelButton.setText("Close");
            }
            if (!display.readAndDispatch()) display.sleep();
        }
        return exportThread;
    }

    protected void createContents() {
        shell = new Shell(getParent(), SWT.TITLE | SWT.CLOSE | SWT.BORDER | SWT.RESIZE | SWT.APPLICATION_MODAL);
        shell.setLayout(new BorderLayout(0, 0));
        shell.setSize(500, 375);
        String titleString = "Export " + profs.length + " profile" + (profs.length > 1 ? "s": "");
        shell.setText(titleString);

        final Group settingsGroup = new Group(shell, SWT.NONE);
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        settingsGroup.setLayout(gridLayout);
        settingsGroup.setText("Options");
        settingsGroup.setLayoutData(BorderLayout.NORTH);

        final Label titleLabel = new Label(settingsGroup, SWT.NONE);
        titleLabel.setText("Title");

        final Text title = new Text(settingsGroup, SWT.BORDER);
        title.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));

        final Label notesLabel = new Label(settingsGroup, SWT.NONE);
        notesLabel.setText("Notes");

        final Text notes = new Text(settingsGroup, SWT.WRAP | SWT.V_SCROLL | SWT.MULTI | SWT.H_SCROLL | SWT.BORDER);
        final GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 2, 2);
        gridData.heightHint = 40;
        notes.setLayoutData(gridData);
        new Label(settingsGroup, SWT.NONE);

        final Label fileLabel = new Label(settingsGroup, SWT.NONE);
        fileLabel.setText("File to export to");

        filename = new Text(settingsGroup, SWT.BORDER);
        filename.setText("GAMEPACK_EXPORT");
        filename.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        final BrowseButton browseButton = new BrowseButton(settingsGroup, SWT.NONE);
        browseButton.connect(shell, filename, null, BrowseType.FILE, CanonicalType.ZIP, false, null);

        final Button exportCapturesButton = new Button(settingsGroup, SWT.CHECK);
        exportCapturesButton.setText("Also Export Captures");
        new Label(settingsGroup, SWT.NONE);
        new Label(settingsGroup, SWT.NONE);

        final Button exportGameDataButton = new Button(settingsGroup, SWT.CHECK);
        exportGameDataButton.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 2, 1));
        exportGameDataButton.setText("Also Export Game Data");
        new Label(settingsGroup, SWT.NONE);

        final Button startButton = new Button(settingsGroup, SWT.NONE);
        shell.setDefaultButton(startButton);
        startButton.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
        startButton.setText("Start Export");

        cancelButton = new Button(settingsGroup, SWT.NONE);
        cancelButton.setLayoutData(new GridData(80, SWT.DEFAULT));
        cancelButton.setText("Cancel");
        cancelButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                shell.close();
            }
        });
        new Label(settingsGroup, SWT.NONE);

        final Group progressGroup = new Group(shell, SWT.NONE);
        progressGroup.setLayout(new GridLayout());
        progressGroup.setText("Progress");

        final ProgressBar progressBar = new ProgressBar(progressGroup, SWT.NONE);
        progressBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));

        profileLabel = new Label(progressGroup, SWT.NONE);

        logText = new Text(progressGroup, SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER | SWT.H_SCROLL);
        logText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        startButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent e) {
                if (!isValid()) return;
                try {
                    exportThread = new ExportThread(profs, dbversionsList, title.getText(),
                            notes.getText(), exportCapturesButton.getSelection(),
                            exportGameDataButton.getSelection(), FileUtils.canonicalToData(filename.getText()),
                            logText, progressBar, profileLabel);
                    browseButton.setEnabled(false);
                    startButton.setEnabled(false);
                    exportThread.start();
                } catch (IOException ex) {
                    GeneralPurposeDialogs.warningMessage(shell, ex.getMessage());
                    exportThread = null;
                }
            }
        });
    }

    private boolean isValid() {
        GeneralPurposeDialogs.initErrorDialog();
        String f = filename.getText();
        if (f.equals(""))
            GeneralPurposeDialogs.addError("You must enter a filename for export.", filename);
        else if (FileUtils.isExistingFile(FileUtils.canonicalToData(f)))
            GeneralPurposeDialogs.addError(FileUtils.canonicalToData(f)
                    + " already exists. You must enter another filename for export.", filename);
        else {
            File dir = FileUtils.canonicalToData(f).getParentFile();
            if (dir == null || !dir.exists()) 
                GeneralPurposeDialogs.addError("This directory does not exist. "
                        + "Create it first, or enter another filename for export.", filename);
        }
        return !GeneralPurposeDialogs.displayErrorDialog(shell);
    }
}
