/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model.conf;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringReader;
import org.eclipse.swt.widgets.Shell;
import org.dbgl.gui.GeneralPurposeDialogs;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;
import org.dbgl.model.Template;
import org.dbgl.util.FileUtils;


public class CompositeConfiguration {

    private Configuration baseConf;
    private Configuration compositeConf;
    private DosboxVersion dbversion;

    
    public CompositeConfiguration(final DosboxVersion dbversion, final boolean force, final Shell shell, final PrintStream ps) {
        processConfiguration(dbversion.getCanonicalConfFile(), null, true, force, shell, ps);
        this.dbversion = dbversion; 
    }
    
    public CompositeConfiguration(final DosboxVersion dbversion, final Profile profile, final boolean force, final Shell shell, final PrintStream ps) {
        this(dbversion, profile.getCanonicalConfFile(), null, force, shell, ps);
    }

    public CompositeConfiguration(final DosboxVersion dbversion, final Template template, final boolean force, final Shell shell, final PrintStream ps) {
        this(dbversion, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), null, force, shell, ps);
    }
    
    public CompositeConfiguration(final DosboxVersion dbversion, final File file, final String contents, final boolean force, final Shell shell, final PrintStream ps) {
        this(dbversion, force, shell, ps);
        if (file != null) {
            processConfiguration(file, contents, false, force, shell, ps);
        }
    }
    
    public CompositeConfiguration(final CompositeConfiguration comp, final DosboxVersion dbversion, final Template template, final boolean force, final Shell shell, final PrintStream ps) {
        this(dbversion, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), null, force, shell, ps);
        if (baseConf != null && compositeConf != null) {
            Configuration otherAutoSettings = new Configuration(comp.compositeConf, false);
            this.compositeConf.setAutoexecSettings(
                    otherAutoSettings.isLoadfix(), otherAutoSettings.getLoadfixValue(),
                    otherAutoSettings.getMain(), otherAutoSettings.getMainParameters(),
                    otherAutoSettings.getImg1(), otherAutoSettings.getImg2());
        }
    }
    
    public CompositeConfiguration(final CompositeConfiguration comp, final DosboxVersion dbversion,
            final boolean keepValues, final boolean isTemplate, final Shell shell, final PrintStream ps) {
        this.baseConf = comp.baseConf;
        this.compositeConf = comp.compositeConf;
        this.dbversion = comp.dbversion;
        
        Configuration uniqueMountPoints = null;
        Configuration otherAutoSettings = null;
        if (baseConf != null && compositeConf != null) {
            uniqueMountPoints = new Configuration(compositeConf, true, baseConf, this.dbversion != dbversion);
            otherAutoSettings = new Configuration(compositeConf, keepValues);
        }
        
        if (keepValues) {
            if (!update(dbversion, shell, ps, isTemplate)) {
                this.compositeConf = null;
            }
        } else {
            processConfiguration(dbversion.getCanonicalConfFile(), null, true, false, shell, ps);
            this.dbversion = dbversion;
        }
        
        if (compositeConf != null && otherAutoSettings != null) {
            if (keepValues) {
                this.compositeConf.setAutoexecSettings(otherAutoSettings.isExit(), otherAutoSettings.getMixer(),
                    otherAutoSettings.getKeyb(), otherAutoSettings.getIpxnet(), otherAutoSettings.isBooter());
            } else {
                this.compositeConf.setAutoexecSettings(otherAutoSettings.isExit(), otherAutoSettings.isBooter());
            }
            this.compositeConf.setAutoexecSettings(
                    otherAutoSettings.isLoadfix(), otherAutoSettings.getLoadfixValue(),
                    otherAutoSettings.getMain(), otherAutoSettings.getMainParameters(),
                    otherAutoSettings.getImg1(), otherAutoSettings.getImg2());
        }
        
        if (compositeConf != null && uniqueMountPoints != null && 
                ((this.dbversion != comp.dbversion) || !keepValues) ) {
            compositeConf.addMounts(uniqueMountPoints);
        }
    }
    
    private boolean update(final DosboxVersion dbversion, final Shell shell, final PrintStream ps, final boolean isTemplate) {
        Configuration org = new Configuration(compositeConf, false, null, false);
        processConfiguration(dbversion.getCanonicalConfFile(), null, true, false, shell, ps);
        if (baseConf == null) {
            if (shell != null && GeneralPurposeDialogs.confirmMessage(shell, Settings.getInstance().msg( 
                    isTemplate? "general.confirm.missingdosboxconftemplate": "general.confirm.missingdosboxconfprofile"))) {
                processConfiguration(dbversion.getCanonicalConfFile(), null, true, true, shell, ps);
            } else {
                GeneralPurposeDialogs.warningMessage(ps, Settings.getInstance().msg("general.error.readconf", new Object[] {dbversion.getCanonicalConfFile()}));
                return false;
            }
        }
        compositeConf.updateAndExtendWithValuesFrom(org);
        this.dbversion = dbversion;
        return true;
    }
    
    public void setCompositeConf(final Configuration configuration) {
        this.compositeConf = configuration;
    }
    
    private void save(final boolean substract, final File target, final Shell shell, final PrintStream ps) {
        Configuration toSave = new Configuration(compositeConf, false, null, false);
        toSave.strip(baseConf);
        toSave.setFixedCycles();
        if (substract) {
            toSave.substract(baseConf);
        }
        try {
            toSave.saveToFile(target, false, baseConf);
        } catch (IOException ex) {
            GeneralPurposeDialogs.warningMessage(shell, ps, ex.getMessage());
        }
    }
    
    public void save(final Template template, final Shell shell, final PrintStream ps) {
        save(true, FileUtils.constructCanonicalTemplateFileLocation(template.getId()), shell, ps);
    }
    
    public void save(final Profile profile, final Shell shell, final PrintStream ps) {
        compositeConf.setValue("dosbox", "captures", FileUtils.constructRelativeCapturesDir(profile.getId()));
        save((dbversion != null && dbversion.isMultiConfig()), profile.getCanonicalConfFile(), shell, ps);
    }
    
    public void save(final File file, final Shell shell, final PrintStream ps) {
        save((dbversion != null && dbversion.isMultiConfig()), file, shell, ps);
    }

    public String getFullConf() {
        return compositeConf.toString(false, baseConf);
    }
    
    public String getIncrConf() {
        Configuration toSave = new Configuration(compositeConf, false, null, false);
        toSave.strip(baseConf);
        toSave.setFixedCycles();
        if (dbversion != null && dbversion.isMultiConfig()) {
            toSave.substract(baseConf);
        }
        return toSave.toString(false, baseConf);
    }
    
    public File getExportGameDir(int index) {
        return new File("./" + index + "/");
    }

    public Configuration getBaseConf() {
        return baseConf;
    }

    public Configuration getCompositeConf() {
        return compositeConf;
    }
    
    public int getDbversionId() {
        return dbversion.getId();
    }
    
    private void init() {
        baseConf = null;
        compositeConf = null;
        dbversion = null;
    }
    
    private void processConfiguration(final File file, final String contents, final boolean base, final boolean force, final Shell shell, final PrintStream ps) {
        if (base) {
            try {
                init();
                if (file == null) {
                	baseConf = new Configuration(new StringReader(contents));
                } else {
                	baseConf = new Configuration(file, true);
                }
                if (baseConf.hasParsingProblems()) {
                    GeneralPurposeDialogs.warningMessage(shell, ps, baseConf.getParsingProblems());
                }
            } catch (IOException ex) {
                GeneralPurposeDialogs.warningMessage(shell, ps, ex.getMessage());
                if (force) {
                    baseConf = new Configuration();
                }
            } finally {
                if (baseConf != null) {
                    compositeConf = new Configuration(baseConf, true, null, false);
                    compositeConf.unsetFixedCycles();
                }
            }
        } else {
            try {
                compositeConf.setAutoexecSettings(false);
                if (file == null) {
                	compositeConf.parseConfigurationFile(new StringReader(contents), "unknown", true);
                } else {
                	compositeConf.parseConfigurationFile(file, true);
                }
                compositeConf.unsetFixedCycles();
            } catch (IOException ex) {
                GeneralPurposeDialogs.warningMessage(shell, ps, ex.getMessage());
            }
            if (compositeConf.hasParsingProblems()) {
                GeneralPurposeDialogs.warningMessage(shell, ps, compositeConf.getParsingProblems());
            }
        }
    }
    
    public String[] addMount(final String mount) {
        return compositeConf.addMount(mount);
    }
    
    public String[] removeMount(final int index) {
        if (index < baseConf.getNrOfMounts()) {
            return compositeConf.toggleMount(index);
        } else {
            return compositeConf.removeMount(index);
        }
    }
    
    public String[] editMount(final int index, final String mount) {
        if (index < baseConf.getNrOfMounts()) {
            compositeConf.unMount(index);
            return compositeConf.addMount(mount);
        } else {
            return compositeConf.replaceMount(index, mount);
        }
    }
    
    public String getRequiredMount(final boolean booter, final String exe) {
        return compositeConf.getRequiredMount(booter, exe);
    }
    
    public String[] addRequiredMount(final boolean booter, final String exe) {
        return compositeConf.addRequiredMount(booter, exe);
    }
}
