/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model.conf;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.dbgl.exception.InvalidMountstringException;
import org.dbgl.model.Mount;
import org.dbgl.model.Settings;
import org.dbgl.util.FileUtils;
import org.dbgl.util.PlatformUtils;
import org.dbgl.util.StringUtils;


public class Configuration {

    private Map<String, Section> sections;
    private Autoexec autoexec;
    private StringBuffer parsingProblems;
    private Settings settings;


    public Configuration() {
        parsingProblems = new StringBuffer();
        sections = new LinkedHashMap<String, Section>();
        autoexec = new Autoexec(new String[0]);
        settings = Settings.getInstance();
    }
    
    public Configuration(final Configuration conf, final boolean keepLoadfix) {
        this();
        this.autoexec = new Autoexec(conf.autoexec);
        if (!keepLoadfix) {
            this.autoexec.loadfix = false;
            this.autoexec.loadfixValue = 0;
        }
    }

    public Configuration(final Configuration conf, final boolean mountingsOnly, final Configuration baseConf, final boolean switchDbversion) {
        this();
        for (String key: conf.sections.keySet()) {
            sections.put(key, new Section(conf.sections.get(key)));
        }
        if (mountingsOnly) {
            this.autoexec = new Autoexec(conf.autoexec.mountingpoints, baseConf == null? null: baseConf.autoexec, switchDbversion);
        } else {
            this.autoexec = new Autoexec(conf.autoexec);
        }
    }

    public Configuration(final File file, final boolean toLowerCase) throws IOException {
        this();
        parseConfigurationFile(file, toLowerCase);
    }

    public Configuration(final Reader reader) throws IOException {
        this();
        parseConfigurationFile(reader, FileUtils.PROFILES_XML, false);
    }

    public Set<String> getSectionKeys() {
        return sections.keySet();
    }

    public void saveToFile(final File file, final boolean ordered, final Configuration baseConf) throws IOException {
        try {
            BufferedWriter configData = new BufferedWriter(new FileWriter(file));
            configData.write(toString(ordered, baseConf));
            configData.flush();
            configData.close();
        } catch (IOException e) {
            e.printStackTrace();
            throw new IOException(settings.msg("general.error.savefile", new Object[] {file}));
        }
    }

    public final void parseConfigurationFile(final File file, final boolean toLowerCase) throws IOException {
        try {
            parseConfigurationFile(new FileReader(file), file.getPath(), toLowerCase);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            throw new IOException(settings.msg("general.error.openfile", new Object[] {file}));
        }
    }

    public void parseConfigurationFile(final Reader reader, final String file, final boolean toLowerCase) throws IOException {
        try {
            BufferedReader configData = new BufferedReader(reader);
            String orgTextLine;
            String currSectionTitle = null;
            Section currSection = null;
            boolean lastItemHadMissingSection = false;
            List<String> autoexecLines = new ArrayList<String>();

            while ((orgTextLine = configData.readLine()) != null) {
                orgTextLine = orgTextLine.trim();
                String textLine = toLowerCase? orgTextLine.toLowerCase(): orgTextLine;
                if ((textLine.length() > 0) && textLine.charAt(0) != '#') {
                    if (textLine.charAt(0) == '[') { // a new section starts here
                        int start = textLine.indexOf(('['));
                        int end = textLine.lastIndexOf(']');
                        if (end == -1) {
                            parsingProblems.append(settings.msg("general.error.parseconf", new Object[] {file, textLine})).append('\n');
                        } else {
                            currSectionTitle = textLine.substring(start + 1, end);
                            if (sections.containsKey(currSectionTitle)) {
                                currSection = sections.get(currSectionTitle);
                            } else {
                                if (!"autoexec".equals(currSectionTitle)) {
                                    currSection = new Section();
                                    sections.put(currSectionTitle, currSection);
                                }
                            }
                        }
                    } else { // an item starts here
                        if (currSectionTitle == null) { // value before section
                            if (!lastItemHadMissingSection) {
                                parsingProblems.append(settings.msg("general.error.sectionmissing", new Object[] {file, textLine})).append('\n');
                            }
                            lastItemHadMissingSection = true;
                        } else { 
                            if ("autoexec".equals(currSectionTitle)) { // autoexec config item
                                autoexecLines.add(orgTextLine);
                            } else { // normal config item
                                int end = textLine.indexOf('=');
                                if (end == -1) {
                                    parsingProblems.append(settings.msg("general.error.parseconf", new Object[] {file, textLine})).append('\n');
                                } else {
                                    String name = textLine.substring(0, end).trim();
                                    String value = textLine.substring(end + 1).trim();
                                    if ("gui".equals(currSectionTitle) || "database".equals(currSectionTitle)
                                            || "directory".equals(currSectionTitle)) {
                                        value = orgTextLine.substring(end + 1).trim();
                                    }
                                    currSection.items.put(name, value);
                                }
                            }
                            lastItemHadMissingSection = false;
                        }
                    }
                }
            }
            configData.close();
            autoexec.parseLines(autoexecLines.toArray(new String[autoexecLines.size()]), false);
        } catch (IOException e) {
            e.printStackTrace();
            StringBuffer msg = new StringBuffer(settings.msg("general.error.readconf", new Object[] {file}));
            if (parsingProblems.length() > 0) {
                msg.append('\n').append(getParsingProblems());
            }
            throw new IOException(msg.toString());
        }
    }

    public boolean hasValue(final String sectionTitle, final String sectionItem) {
        return sections.containsKey(sectionTitle) && sections.get(sectionTitle).items.containsKey(sectionItem);
    }

    public String getValue(final String sectionTitle, final String sectionItem) {
        String result = null;
        if (sections.containsKey(sectionTitle)) {
            result = sections.get(sectionTitle).items.get(sectionItem);
        }
        if (result == null) {
            result = ""; // in case the item was not found
        }
        return result;
    }

    public int getIntValue(final String sectionTitle, final String sectionItem) {
        try {
            return Integer.parseInt(getValue(sectionTitle, sectionItem));
        } catch (NumberFormatException e) {
            return -1; // value is not a number
        }
    }

    public boolean getBooleanValue(final String sectionTitle, final String sectionItem) {
        return Boolean.valueOf(getValue(sectionTitle, sectionItem));
    }

    public void setValue(final String sectionTitle, final String sectionItem, final String value) {
        Section sec = createSection(sectionTitle);
        sec.items.put(sectionItem, value);
    }

    public void setIntValue(final String sectionTitle, final String sectionItem, final int value) {
        setValue(sectionTitle, sectionItem, String.valueOf(value));
    }

    public void setBooleanValue(final String sectionTitle, final String sectionItem, final boolean value) {
        setValue(sectionTitle, sectionItem, String.valueOf(value));
    }

    public void updateValue(final String sectionTitle, final String sectionItem, final String value) {
        if (sections.containsKey(sectionTitle) && sections.get(sectionTitle).items.containsKey(sectionItem)) {
            sections.get(sectionTitle).items.put(sectionItem, value);
        }
    }
    
    public void updateValue(final String section, final String oldItem, final String newItem, final String value, final Configuration base) {
        if (base.hasValue(section, newItem)) {
            updateValue(section, newItem, value);
        } else {
            updateValue(section, oldItem, value);
        }
    }
    
    public String[] getItems(final String sectionTitle) {
        return new TreeMap<String, String>(sections.get(sectionTitle).items).keySet().toArray(new String[0]);
    }

    public String[] getValues(final String sectionTitle, final String sectionItem) {
        String val = getValue(sectionTitle, sectionItem);
        if (val.length() <= 0) {
            return new String[0];
        }
        String[] res = val.split(" ");
        for (int i = 0; i < res.length; i++) {
            res[i] = res[i].replaceAll("<space>", " ");
        }
        return res;
    }

    public String getMultilineValues(final String sectionTitle, final String sectionItem, final String delimiter) {
        return StringUtils.stringArrayToString(getValues(sectionTitle, sectionItem), delimiter);
    }

    public void setMultilineValues(final String sectionTitle, final String sectionItem, final String values, final String delimiter) {
        setValue(sectionTitle, sectionItem, values.replaceAll(" ", "<space>").replace(delimiter, " ").trim());
    }

    public int[] getIntValues(final String sectionTitle, final String sectionItem) {
        return StringUtils.stringToIntArray(getValue(sectionTitle, sectionItem));
    }

    public void setIntValues(final String sectionTitle, final String sectionItem, final int[] values) {
        setValue(sectionTitle, sectionItem, StringUtils.intArrayToString(values));
    }

    public boolean[] getBooleanValues(final String sectionTitle, final String sectionItem) {
        return StringUtils.stringToBooleanArray(getValue(sectionTitle, sectionItem));
    }

    public void setBooleanValues(final String sectionTitle, final String sectionItem, final boolean[] values) {
        setValue(sectionTitle, sectionItem, StringUtils.booleanArrayToString(values));
    }

    public void removeValue(final String sectionTitle, final String sectionItem) {
        if (sections.containsKey(sectionTitle)) {
            Section sec = sections.get(sectionTitle);
            sec.items.remove(sectionItem);
            if (sec.items.isEmpty()) {
                sections.remove(sectionTitle);
            }
        }
    }

    public void removeSection(final String title) {
        sections.remove(title);
    }

    public void updateAndExtendWithValuesFrom(final Configuration conf) {
        // transfer 'old' configuration data from conf
        // to equivalent new settings
        
        update(conf);
        
        for (String sectionTitle: conf.sections.keySet()) {
            for (String sectionItem: conf.getItems(sectionTitle)) {
                setValue(sectionTitle, sectionItem, conf.getValue(sectionTitle, sectionItem));
            }
        }
    }
    
    public void update(final Configuration conf) {
        // 0.63 -> 0.65
        conf.transferSetting("sblaster", "type", "sblaster", "sbtype", this);
        conf.transferSetting("sblaster", "base", "sblaster", "sbbase", this);
        conf.transferSetting("gus", "rate", "gus", "gusrate", this);
        conf.transferSetting("gus", "base", "gus", "gusbase", this);
        
        if (!this.hasValue("midi", "intelligent") && conf.hasValue("midi", "intelligent")) {
            boolean mpu = conf.getBooleanValue("midi", "mpu401");
            boolean intelli = conf.getBooleanValue("midi", "intelligent");
            conf.setValue("midi", "mpu401", mpu? (intelli? "intelligent": "uart"): "none");
            conf.removeValue("midi", "intelligent");
        } else if (this.hasValue("midi", "intelligent") && !conf.hasValue("midi", "intelligent")) {
            String mpu = conf.getValue("midi", "mpu401");
            conf.setBooleanValue("midi", "mpu401", !mpu.equalsIgnoreCase("none"));
            conf.setBooleanValue("midi", "intelligent", mpu.equalsIgnoreCase("intelligent"));
        }
        
        // 0.65 -> 0.70
        conf.transferSetting("bios", "joysticktype", "joystick", "joysticktype", this);
        
        // 0.72 -> 0.73
        if (this.hasValue("cpu", "cputype") && !conf.hasValue("cpu", "cputype")) {
            // if machine was set to vga, reset it
            if (conf.getValue("dosbox", "machine").equalsIgnoreCase("vga")) {
                conf.removeValue("dosbox", "machine");
            }
            // if keyboard layout was set to none, reset it
            if (conf.getValue("dos", "keyboardlayout").equalsIgnoreCase("none")) {
                conf.removeValue("dos", "keyboardlayout");
            }
        } else if (conf.hasValue("cpu", "cputype") && !this.hasValue("cpu", "cputype")) {
            // 0.73 -> 0.72
            String mach = conf.getValue("dosbox", "machine");
            // if machine was NOT set to cga/hercules/pcjr/tandy, reset it
            if (!(mach.equalsIgnoreCase("cga") || mach.equalsIgnoreCase("hercules") ||
                  mach.equalsIgnoreCase("pcjr") || mach.equalsIgnoreCase("tandy"))) {
                conf.removeValue("dosbox", "machine");
            }
            // if keyboard layout was set to auto, reset it
            if (conf.getValue("dos", "keyboardlayout").equalsIgnoreCase("auto")) {
                conf.removeValue("dos", "keyboardlayout");
            }
            conf.removeValue("cpu", "cputype");
        }
        
        conf.transferSetting("midi", "device", "midi", "mididevice", this);
        conf.transferSetting("midi", "config", "midi", "midiconfig", this);
        conf.transferSetting("sblaster", "mixer", "sblaster", "sbmixer", this);
        if (conf.hasValue("gus", "irq1") && !this.hasValue("gus", "irq1") && this.hasValue("gus", "gusirq")) {
            // gus from old to new style
            conf.setValue("gus", "gusirq", conf.getValue("gus", "irq1"));
            conf.removeValue("gus", "irq1");
            conf.removeValue("gus", "irq2");
            conf.setValue("gus", "gusdma", conf.getValue("gus", "dma1"));
            conf.removeValue("gus", "dma1");
            conf.removeValue("gus", "dma2");
        } else if (conf.hasValue("gus", "gusirq") && !this.hasValue("gus", "gusirq") && this.hasValue("gus", "irq1")) {
            // gus from new to old style
            conf.setValue("gus", "irq1", conf.getValue("gus", "gusirq"));
            conf.setValue("gus", "irq2", conf.getValue("gus", "gusirq"));
            conf.removeValue("gus", "gusirq");
            conf.setValue("gus", "dma1", conf.getValue("gus", "gusdma"));
            conf.setValue("gus", "dma2", conf.getValue("gus", "gusdma"));
            conf.removeValue("gus", "gusdma");
        }
    }
    
    private void transferSetting(final String oldSection, final String oldItem,
            final String newSection, final String newItem, final Configuration newConf) {
        if (hasValue(oldSection, oldItem) && !newConf.hasValue(oldSection, oldItem) && newConf.hasValue(newSection, newItem)) {
            setValue(newSection, newItem, getValue(oldSection, oldItem));
            removeValue(oldSection, oldItem);
        } else if (hasValue(newSection, newItem) && !newConf.hasValue(newSection, newItem) && newConf.hasValue(oldSection, oldItem)) {
            setValue(oldSection, oldItem, getValue(newSection, newItem));
            removeValue(newSection, newItem);
        }
    }

    public void strip(final Configuration conf) {
        Configuration newConf = new Configuration();
        for (String key: sections.keySet()) {
            if (conf.sections.containsKey(key)) {
                for (String item: sections.get(key).items.keySet()) {
                    if (conf.sections.get(key).items.containsKey(item)) {
                        newConf.setValue(key, item, sections.get(key).items.get(item));
                    }
                }
            }
        }
        this.sections = newConf.sections;
    }
    
    public void setFixedCycles() {
        if (hasValue("cpu", "cputype") && hasValue("cpu", "cycles")) { // CVS from 05/18/2008
            if (!hasValue("sblaster", "oplemu")) { // to (roughly) 02/15/2009
                String cycles = getValue("cpu", "cycles");
                if (! (cycles.contains("auto") || cycles.contains("max")) ) {
                    setValue("cpu", "cycles", "fixed " + cycles);
                }
            }
        }
    }
    
    public void unsetFixedCycles() {
        if (hasValue("cpu", "cycles")) {
            String cycles = this.getValue("cpu", "cycles");
            if (cycles.startsWith("fixed ")) {
                setValue("cpu", "cycles", cycles.substring(6));
            }
        }
    }
    
    public String toString(final boolean ordered, final Configuration baseConf) {
        StringBuffer result = new StringBuffer();
        for (String key: sections.keySet()) {
            result.append("[" + key + "]" + PlatformUtils.EOLN);
            result.append((sections.get(key)).toString(ordered)).append(PlatformUtils.EOLN);
        }
        String auto;
        if (baseConf == null) {
            auto = autoexec.getString(null);
        } else {
            auto = autoexec.getString(baseConf.autoexec.mountingpoints);
        }
        if (auto.length() > 0) {
            result.append("[autoexec]").append(PlatformUtils.EOLN).append(auto).append(PlatformUtils.EOLN);
        }
        return result.toString();
    }
    
    protected Section createSection(final String sectionTitle) {
        if (sections.containsKey(sectionTitle)) {
            return sections.get(sectionTitle);
        }
        Section newSection = new Section();
        sections.put(sectionTitle, newSection);
        return newSection;
    }
    
    public void substract(final Configuration conf) {    
        for (String key: conf.sections.keySet()) {
            if (sections.containsKey(key)) {
                Section mysec = sections.get(key);
                mysec.substract(conf.sections.get(key));
                if (mysec.items.isEmpty()) {
                    sections.remove(key);
                }
            }
        }
    }

    public boolean hasParsingProblems() {
        return parsingProblems.length() > 0;
    }

    public final String getParsingProblems() {
        String result = parsingProblems.toString();
        parsingProblems = new StringBuffer();
        return result;
    }

    public File getCanonicalMainDir() {
        if (isBooter()) {
            return FileUtils.getCanMainFile(new File(autoexec.img1)).getParentFile();
        } else {
            return FileUtils.getCanMainFile(new File(autoexec.main)).getParentFile();
        }
    }

    public void makeRelative(final File basePath) {
        for (Mount mount: autoexec.mountingpoints) {
            mount.makeRelative(basePath);
        }
        if (isBooter()) {
            autoexec.img1 = FileUtils.makeRelativeTo(new File(autoexec.img1), basePath).getPath();
            autoexec.img2 = FileUtils.makeRelativeTo(new File(autoexec.img2), basePath).getPath();
        } else {
            autoexec.main = FileUtils.makeRelativeTo(new File(autoexec.main), basePath).getPath();
        }
    }

    public String[] addMount(final String mount) {
        autoexec.addMount(mount);
        return getMountingpoints();
    }
    
    public void addMounts(final Configuration conf) {
        for (Mount m: conf.autoexec.mountingpoints) {
            autoexec.mountingpoints.add(m);
        }
    }

    public String[] removeMount(final int index) {
        autoexec.mountingpoints.remove(index);
        return getMountingpoints();
    }

    public String[] toggleMount(final int index) {
        autoexec.mountingpoints.get(index).toggleMount();
        return getMountingpoints();
    }
    
    public void unMount(final int index) {
        Mount mnt = autoexec.mountingpoints.get(index);
        if (!mnt.isUnmounted()) {
            mnt.toggleMount();
        }
    }

    public String[] replaceMount(final int index, final String mount) {
        try {
            autoexec.mountingpoints.set(index, new Mount(mount));
        } catch (InvalidMountstringException e) {
            // nothing we can do
        }
        return getMountingpoints();
    }
    
    public int getNrOfMounts() {
        return autoexec.mountingpoints.size();
    }
    
    public String getRequiredMount(final boolean booter, final String main) {
        if (autoexec.convertToDosboxPath(main)[0].length() > 0) {
            return null;
        }
        try {
            return new Mount(booter, main, getMountingpoints()).getPathAsString();
        } catch (Exception e) {
        	// this is not entirely correct; returning null assumes no mounts required
        	// but this should never happen anyway
            return null;
        }
    }
    
    public String[] addRequiredMount(final boolean booter, final String main) {
        try {
            if (FileUtils.containsIso(main) != -1) {
                Mount tmp = new Mount(booter, ".", getMountingpoints());
                if (tmp.getDriveletter() == 'C')
                    addMount(tmp.toString());
                return addMount(new Mount(booter, main, getMountingpoints()).toString());
            } else {
                return addMount(new Mount(booter, main, getMountingpoints()).toString());
            }
        } catch (Exception e) {
            // could not add, just return previous values
            return getMountingpoints();
        }
    }
    
    public boolean isExit() {
        return autoexec.exit;
    }
    
    public boolean isLoadfix() {
        return autoexec.loadfix;
    }

    public String getLoadfixValue() {
        return autoexec.loadfixValue > 0 ? String.valueOf(autoexec.loadfixValue): "";
    }
    
    public String getKeyb() {
        return autoexec.keyb;
    }
    
    public String getMixer() {
        return autoexec.mixer;
    }
    
    public String getIpxnet() {
        return autoexec.ipxnet;
    }
    
    public String[] getMountingpoints() {
        String[] result = new String[autoexec.mountingpoints.size()];
        int mountIndex = 0;
        for (Mount mount: autoexec.mountingpoints) {
            result[mountIndex++] = mount.toString(true);
        }
        return result;
    }
    
    public String getImg1() {
        if ("file".equals(autoexec.img1)) {
            return "";
        }
        return autoexec.img1;
    }

    public String getImg2() {
        return autoexec.img2;
    }

    public String getMain() {
        return autoexec.main;
    }
    
    public boolean isBooter() {
        return autoexec.img1.length() > 0;
    }

    public String getMainParameters() {
        return autoexec.params;
    }
    
    public boolean isIncomplete() {
        return (autoexec.img1.length() == 0 && autoexec.main.length() == 0);
    }
    
    public void setAutoexecSettings(final boolean exit) {
        autoexec.exit = exit;
    }
    
    public void setAutoexecSettings(final boolean exit, final boolean booter) {
        autoexec.exit = exit;
        autoexec.img1 = booter? "file": "";  // for templates
    }
    
    public void setAutoexecSettings(final boolean exit, final String mixer, final String keyb, final String ipxnet, final boolean booter) {
        setAutoexecSettings(exit, booter);
        autoexec.mixer = mixer;
        autoexec.keyb = keyb;
        autoexec.ipxnet = ipxnet;
    }
    
    public void setAutoexecSettings(final String main, final String params) {
        autoexec.main = main;
        autoexec.params = params;
    }

    public void setAutoexecSettings(final boolean loadfix, final String loadfixValue, final String main, final String params,
            final String img1, final String img2) {
    	setAutoexecSettings(main, params);
        autoexec.loadfix = loadfix;
        try {
            autoexec.loadfixValue = Integer.parseInt(loadfixValue);
        } catch (NumberFormatException e) {
            // no change
        }
        autoexec.img1 = img1;
        autoexec.img2 = img2;
    }
}
