package org.dbgl.util;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import org.dbgl.db.Database;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.ExpGame;
import org.dbgl.model.KeyValuePair;
import org.dbgl.model.conf.CompositeConfiguration;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;


public final class ImportThread extends Thread {

    private String title;
    private int profileNumber;
    private StringBuffer messageLog;
    private Display display;
    private Text log;
    private ProgressBar progressBar;
    private Label profileLabel;

    private boolean importProfileSettings;
    private boolean importCaptures;
    private boolean importGameData;
    private boolean importFullSettings;
    private File archive;
    private List<ExpGame> profiles;
    private List<DosboxVersion> dbversionsList;


    public ImportThread(boolean ip, boolean ic, boolean ig, boolean f, File a, List<ExpGame> profs,
            List<DosboxVersion> dbList, Text log, ProgressBar progressBar, Label profileLabel) throws IOException {
        this.log = log;
        this.progressBar = progressBar;
        this.profileLabel = profileLabel;
        this.display = log.getShell().getDisplay();
        this.importProfileSettings = ip;
        this.importCaptures = ic;
        this.importGameData = ig;
        this.importFullSettings = f;
        this.archive = a;
        this.profiles = profs;
        this.dbversionsList = dbList;
        this.progressBar.setMaximum(profiles.size());
    }

    public void run() {
        Database db = Database.getInstance();

        for (profileNumber = 0; profileNumber < profiles.size(); profileNumber++) {
            try {
                ExpGame p = profiles.get(profileNumber);

                title = p.getTitle();
                messageLog = new StringBuffer();

                String newCapturesString = null;
                if (importProfileSettings) {
                    int devId = KeyValuePair.findIdByValue(db.readDevelopersList(), p.getDeveloperName());
                    int publId = KeyValuePair.findIdByValue(db.readPublishersList(), p.getPublisherName());
                    int genId = KeyValuePair.findIdByValue(db.readGenresList(), p.getGenre());
                    int yrId = KeyValuePair.findIdByValue(db.readYearsList(), p.getYear());
                    int statId = KeyValuePair.findIdByValue(db.readStatusList(), p.getStatus());
                    int[] custIndex = new int[] { -1, -1, -1, -1 };
                    for (int i = 0; i < 4; i++)
                        custIndex[i] = KeyValuePair.findIdByValue(db.readCustomList(i), p.getCustomString(i));
                    int profileId = db.addOrEditProfile(title, p.getDeveloperName(), p.getPublisherName(),
                            p.getGenre(), p.getYear(), p.getStatus(), p.getSetup(), p.getSetupParameters(),
                            p.getNotes(), p.isDefault(), devId, publId, genId, yrId, statId, p.getDbversionId(),
                            p.getLinks(), p.getCustomStrings(), p.getCustomInts(), custIndex, p.getLinkTitles(), -1);

                    //Configuration gameConf = new Configuration(importFullSettings? p.getFullConfig(): p.getIncrConfig());
                    DosboxVersion assocDBVersion = dbversionsList.get(DosboxVersion.findById(dbversionsList, p.getDbversionId()));
                    
                    CompositeConfiguration gameCConf = new CompositeConfiguration(assocDBVersion, null, importFullSettings? p.getFullConfig(): p.getIncrConfig(), false, null, System.out);
                    /*if (assocDBVersion.isMultiConfig()) {
                        Configuration dosboxConf = new Configuration(assocDBVersion.getCanonicalConfFile());
                        gameConf.substract(dosboxConf);
                    }*/
                    
                    String newConfString = FileUtils.constructUniqueConfigFileString(profileId, p.getTitle(), gameCConf.getCompositeConf());
                    newCapturesString = FileUtils.constructCapturesDir(profileId);
                    //gameConf.saveToFile(FileUtils.canonicalToData(newConfString));
                    gameCConf.save(FileUtils.canonicalToData(newConfString), null, System.out);
                    db.updateProfileConf(newConfString, newCapturesString, profileId);
                    //messageLog.append(gameConf.getParsingProblems());
                    
                    if (importCaptures) {
                        File cap = FileUtils.canonicalToData(newCapturesString);
                        if (!cap.exists()) {
                            FileUtils.createDir(cap);
                            try {
                                FileUtils.extractZip(archive, new File(p.getCaptures()), cap);
                            } catch (IOException e) {
                                messageLog.append(title + ": captures could not be imported: " + e.getMessage() + "\n");
                            }
                        } else {
                            messageLog.append("Directory " + cap + " already exists, " +
                                    "skipping capture extraction for " + title + "\n");
                        }
                    }
                }
                if (importGameData) {
                    File gamedir = FileUtils.canonicalToDosroot(p.getGameDir().getPath());
                    if (!gamedir.exists()) {
                        FileUtils.createDir(gamedir);
                        try {
                            FileUtils.extractZip(archive, p.getGameDir(), gamedir);
                        } catch (IOException e) {
                            messageLog.append(title + ": game data could not be imported: " + e.getMessage() + "\n");
                        }
                    } else {
                        messageLog.append("Directory " + gamedir + " already exists, " +
                                "skipping game data extraction for " + title + "\n");
                    }
                }
            } catch (SQLException e) {
                messageLog.append(e.getMessage() + '\n');
            }/* catch (IOException e) {
                messageLog.append(e.getMessage() + '\n');
            }*/

            if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) break;
            display.asyncExec(new Runnable() {
                public void run() {
                    profileLabel.setText("Imported " + title + "...");
                    profileLabel.pack();
                    log.append(messageLog.toString());
                    progressBar.setSelection(profileNumber + 1);
                }
            });
        }
    }
}
