/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.sql.SQLException;
import java.util.Collections;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.dbgl.db.Database;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.Profile;
import org.dbgl.model.Settings;
import org.dbgl.model.conf.CompositeConfiguration;
import org.dbgl.util.MobyGamesThread;
import swing2swt.layout.BorderLayout;


public class EditMultiProfileDialog extends EditProfileDialog {

    private Label profileLabel;
    private ProgressBar progressBar;
    private Text logText;
    private Button consultMobyGames;
    
    private static final int AMOUNT_OF_THREADS = 4;
    private Thread[] mbt = new Thread[AMOUNT_OF_THREADS];
    
    
    public EditMultiProfileDialog(Shell parent) {
        super(parent);
    }
    
    public void setProfiles(final java.util.List<Profile> profs) {
        this.result = profs;
    }
    
    @Override
    public Object open() {
        settings = Settings.getInstance();
        dbase = Database.getInstance();
        if (init()) { 
            createContents();
            shell.open();
            shell.layout();
            Display display = getParent().getDisplay();

            while (!shell.isDisposed()) {
            	
            	boolean alldone = true;
            	if (mbt != null) {
            		for (int i = 0; i < AMOUNT_OF_THREADS; i++) {
            			alldone &= (mbt[i] != null) && (!mbt[i].isAlive());
            		}
            	}
            	if (alldone) {
            		if ((logText.getCharCount() > 0) && (!cancelButton.getText().equals(settings.msg("button.close")))) {
                        GeneralPurposeDialogs.infoMessage(shell, settings.msg("dialog.multiprofile.reviewlog"));
                        profileLabel.setText(settings.msg("dialog.multiprofile.reviewlog"));
                        profileLabel.pack();
                        cancelButton.setText(settings.msg("button.close"));
                        progressBar.setSelection(progressBar.getMaximum());
                        mbt = new Thread[AMOUNT_OF_THREADS];
                    } else {
                        shell.close();
                    }
            	}
            	
                if (!display.readAndDispatch()) {
                    display.sleep();
                }
            }
        }
        return result;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected void createContents() {
        shell = new Shell(getParent(), SWT.TITLE | SWT.CLOSE | SWT.BORDER | SWT.RESIZE | SWT.APPLICATION_MODAL);
        shell.setLayout(new BorderLayout(0, 0));
        shell.addControlListener(new SizeControlAdapter(shell, "multiprofiledialog"));
        
        final java.util.List<Profile> profs = (java.util.List<Profile>)result;
        shell.setText(settings.msg("dialog.multiprofile.title.edit", new Object[] {profs.size()}));
        
        final TabFolder tabFolder = new TabFolder(shell, SWT.NONE);
        createInfoTab(tabFolder);
        createCustomTab(tabFolder);
        createGeneralTab(tabFolder);
        createDisplayTab(tabFolder);
        createMachineTab(tabFolder);
        createAudioTab(tabFolder);
        createIOTab(tabFolder);
        createMountingTab(tabFolder);
        
        // init values
        for (DosboxVersion dbv : dbversionsList) {
            dbversion.add(dbv.getTitle());
        }

        createOkCancelButtons();
        okButton.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(final SelectionEvent event) {
                
                okButton.setEnabled(false);
                
                int tasks = 0;
                if (dbversion.getSelectionIndex() != -1) {
                	tasks += profs.size();
                }
                if (consultMobyGames.getSelection()) {
                	tasks += profs.size();
                }
                progressBar.setMaximum(tasks);
                
                if (dbversion.getSelectionIndex() != -1) {
	                int profileNumber = 0;
	                for (Profile prof: profs) {
	                    final StringBuffer messageLog = new StringBuffer();
	                    
	                    ByteArrayOutputStream bos = new ByteArrayOutputStream();
	                    PrintStream ps = new PrintStream(bos);
	                    
	                    profileLabel.setText(Settings.getInstance().msg("dialog.multiprofile.updating", new Object[] {prof.getTitle()}));
	                    profileLabel.pack();
	                    
	                    try {
	                        int dbversionIndex = DosboxVersion.findById(dbversionsList, prof.getDbversionId());
	                        DosboxVersion orgDbversion = dbversionsList.get(dbversionIndex);
	                        DosboxVersion newDbversion = dbversionsList.get(dbversion.getSelectionIndex());
	                        if (orgDbversion.getId() != newDbversion.getId()) {
	                            CompositeConfiguration compConf = new CompositeConfiguration(orgDbversion, prof, true, null, ps);
	                            if (compConf.getCompositeConf().isIncomplete()) {
	                                if (bos.size() > 0) {
	                                    messageLog.append(prof.getTitle()).append(": ").append(bos.toString());
	                                }
	                                messageLog.append(prof.getTitle()).append(": ").append(settings.msg("dialog.multiprofile.error.profileincomplete"));
	                            } else {
	                                compConf = new CompositeConfiguration(compConf, newDbversion, true, false, null, ps);
	                                if (compConf.getCompositeConf() == null) {
	                                    if (bos.size() > 0) {
	                                        messageLog.append(prof.getTitle()).append(": ").append(bos.toString());
	                                    }
	                                    messageLog.append(prof.getTitle()).append(": ").append(settings.msg("dialog.template.notice.noloadnochanges"));
	                                } else {
	                                    compConf.save(prof, null, ps);
	                                    dbase.updateProfileDosboxversion(newDbversion, prof.getId());
	                                    if (bos.size() > 0) {
	                                        messageLog.append(prof.getTitle()).append(": ").append(bos.toString());
	                                    }
	                                }
	                            }
	                        }
	                    } catch (SQLException se) {
	                        messageLog.append(se.getMessage());
	                    }
	                    
	                    if (messageLog.length() > 0) {
	                        logText.append(messageLog.append('\n').toString());
	                    }
	                    progressBar.setSelection(++profileNumber);
	                    
	                }
                }
                
                if (consultMobyGames.getSelection()) {
	                for (int i = 0; i < AMOUNT_OF_THREADS; i++) {
	                	try {
	                    	mbt[i] = new MobyGamesThread(Collections.synchronizedList(profs), logText, progressBar, profileLabel);
	                    	mbt[i].start();
	                	} catch (IOException e) {}
	                }
                } else {
                	mbt = null;
                }
            }
        });
    }
    
    @Override
    protected void createInfoTab(TabFolder tabFolder) {
    }
    
    @Override
    protected void createCustomTab(TabFolder tabFolder) {
    }
    
    @Override
    protected void createGeneralTab(TabFolder tabFolder) {
        final TabItem generalTabItem = new TabItem(tabFolder, SWT.NONE);
        generalTabItem.setText(settings.msg("dialog.template.tab.general"));

        final Composite composite_1 = new Composite(tabFolder, SWT.NONE);
        final GridLayout gridLayout_1 = new GridLayout();
        composite_1.setLayout(gridLayout_1);
        generalTabItem.setControl(composite_1);

        associationGroup = new Group(composite_1, SWT.NONE);
        associationGroup.setText(settings.msg("dialog.template.association"));
        associationGroup.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        final GridLayout gridLayout_4 = new GridLayout();
        gridLayout_4.numColumns = 4;
        associationGroup.setLayout(gridLayout_4);
        
        final Label dbversionLabel = new Label(associationGroup, SWT.NONE);
        dbversionLabel.setText(settings.msg("dialog.template.dosboxversion"));
        dbversion = new Combo(associationGroup, SWT.READ_ONLY);
        dbversion.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        dbversion.setVisibleItemCount(20);
        dbversion.select(0);
        
        Group mobygamesGroup = new Group(composite_1, SWT.NONE);
        mobygamesGroup.setText(settings.msg("dialog.template.tab.info"));
        mobygamesGroup.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        final GridLayout gridLayout_5 = new GridLayout();
        gridLayout_5.numColumns = 4;
        mobygamesGroup.setLayout(gridLayout_5);
        
        final Label waitOnErrorLabel = new Label(mobygamesGroup, SWT.NONE);
        waitOnErrorLabel.setText(settings.msg("dialog.profile.consultmobygames"));
        consultMobyGames = new Button(mobygamesGroup, SWT.CHECK);
        
        final Group progressGroup = new Group(composite_1, SWT.NONE);
        progressGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        progressGroup.setText(settings.msg("dialog.dfendimport.progress"));
        progressGroup.setLayout(new GridLayout());

        progressBar = new ProgressBar(progressGroup, SWT.NONE);
        progressBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        
        profileLabel = new Label(progressGroup, SWT.NONE);

        logText = new Text(progressGroup, SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.BORDER | SWT.H_SCROLL);
        logText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
    }
    
    @Override
    protected void createDisplayTab(TabFolder tabFolder) {
    }
    
    @Override
    protected void createMachineTab(TabFolder tabFolder) {
    }

    @Override
    protected void createAudioTab(TabFolder tabFolder) {
    }
    
    @Override
    protected void createIOTab(TabFolder tabFolder) {
    }
    
    @Override
    protected void createMountingTab(TabFolder tabFolder) {
    }
}
