/*
 *  Copyright (C) 2006-2009  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.zip.ZipOutputStream;
import org.dbgl.model.ExpProfile;
import org.dbgl.model.Settings;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.Document;


public final class ExportThread extends Thread {

    private List<ExpProfile> profs;
    private Document xmlDoc;
    private boolean exportCaptures;
    private boolean exportGameData;
    private File target;
    private Text log;
    private ProgressBar progressBar;
    private Label status;
    private Display display;


    public ExportThread(List<ExpProfile> p, Document doc, boolean captures, boolean gamedata,
            File zipfile, Text log, ProgressBar progressBar, Label status) throws IOException {
    	this.profs = p;
        this.xmlDoc = doc;
        this.exportCaptures = captures;
        this.exportGameData = gamedata;
        this.target = zipfile;
        this.log = log;
        this.progressBar = progressBar;
        this.progressBar.setMaximum(profs.size());
        this.status = status;
        this.display = log.getShell().getDisplay();
    }

    public void run() {
    	final Settings settings = Settings.getInstance();
    	final StringBuffer messageLog = new StringBuffer();
        try {
            final ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(target));
            XmlUtils.domToZipOutputStream(xmlDoc, new File(FileUtils.PROFILES_XML), zos);
            
            for (final ExpProfile prof: profs) {
                if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) break;
                display.asyncExec(new Runnable() {
                    public void run() {
                        status.setText(settings.msg("dialog.export.exporting", new Object[] { prof.getTitle() }));
                        status.pack();
                    }
                });
                
                File relativeGameDir = prof.getGameDir();
                File relativeGameDirInZip = new File(FileUtils.DOSROOT_DIR, new File(String.valueOf(prof.getId()), relativeGameDir.getPath()).getPath());
                
                if (exportCaptures) {
                    File captures = new File(FileUtils.constructCapturesDir(prof.getId()));
                    try {
                        FileUtils.zipDir(prof.getCanonicalCaptures(), zos, captures);
                    } catch (IOException e) {
                        messageLog.append(settings.msg("dialog.export.error.exportcaptures", new Object[] { prof.getTitle(), e.getMessage() })).append('\n');
                    }
                }

                if (exportGameData) {
                    try {
                    	boolean uniqueGameFolder = true;
                    	for (ExpProfile prof2: profs) {
                    		if (prof != prof2 && prof.getGameDir().equals(prof2.getGameDir()) && (prof.getId() > prof2.getId())) {
                    			uniqueGameFolder = false;
                    		}
                    	}
                    	if (uniqueGameFolder) {
                    		FileUtils.zipDir(FileUtils.canonicalToDosroot(relativeGameDir.getPath()), zos, relativeGameDirInZip);
                    	}
                    } catch (IOException e) {
                        messageLog.append(settings.msg("dialog.export.error.exportgamedata", new Object[] { prof.getTitle(), e.getMessage() })).append('\n');
                    }
                }

                if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) break;
                display.asyncExec(new Runnable() {
                    public void run() {
                        log.setText(messageLog.toString());
                        progressBar.setSelection(progressBar.getSelection() + 1);
                    }
                });
            }

            zos.close();

        } catch (Exception e) {
            messageLog.append(e.getMessage() + '\n');
            display.asyncExec(new Runnable() {
                public void run() {
                    log.setText(messageLog.toString());
                }
            });
        }
    }
}
