/*
 *  Copyright (C) 2006-2010  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.sql.SQLException;
import java.util.Set;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Text;
import org.dbgl.db.Database;
import org.dbgl.model.DosboxVersion;
import org.dbgl.model.KeyValuePair;
import org.dbgl.model.conf.Conf;
import org.dbgl.model.conf.Settings;


public final class DFendImportThread extends Thread {

    public static final String[] CUST_STRINGS = new String[] { "", "", "", "", "", "", "", "" };
    public static final int[] CUST_INTS = new int[] { 0, 0 };
    public static final int[] CUST_IDS = new int[] { 0, 0, 0, 0 };

    private static final String[] LINKS = new String[] { "", "", "", "" };
    
    private final DosboxVersion defaultDBVersion;
    private final Text log;
    private final ProgressBar progressBar;
    private final Label profileLabel;
    private final boolean performCleanup;
    private final Display display;
    private String title;
    private int profileNumber;
    private File dfendProfilesFile;


    public DFendImportThread(final File dfendProfilesFile, final boolean performCleanup,
            final DosboxVersion defaultDBVersion, final Text log, final ProgressBar progressBar,
            final Label profileLabel) throws IOException {
        this.performCleanup = performCleanup;
        this.defaultDBVersion = defaultDBVersion;
        this.log = log;
        this.progressBar = progressBar;
        this.profileLabel = profileLabel;
        this.display = log.getShell().getDisplay();
        this.dfendProfilesFile = dfendProfilesFile;
        this.profileNumber = 0;
    }

    public void run() {
        Database dbase = Database.getInstance();
        final StringBuffer displayedLog = new StringBuffer();
        final StringBuffer messageLog = new StringBuffer();
        
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        PrintStream ps = new PrintStream(bos);
        
        try {
        	Conf profiles = new Conf(dfendProfilesFile, ps);
        	final Set<String> sectionNames = profiles.getSettings().getAllSectionNames();
        	display.asyncExec(new Runnable() { public void run() { progressBar.setMaximum(sectionNames.size()); } });
            
            for (String titleLoop: sectionNames) {
                title = titleLoop;
                String profFile = profiles.getSettings().getValue(title, "prof");
                String confFile = profiles.getSettings().getValue(title, "conf");
                boolean favorite = profiles.getSettings().getValue(title, "fav").equals("1");
                try {
                	Conf dfendExtra = new Conf(new File(profFile), ps);
                	String setup = dfendExtra.getSettings().getValue("Extra", "Setup");
                    String setupParams = dfendExtra.getSettings().getValue("Extra", "SetupParameters");
                    String notes = dfendExtra.getSettings().getValue("ExtraInfo", "Notes");
                    String dev = dfendExtra.getSettings().getValue("ExtraInfo", "Developer");
                    String pub = dfendExtra.getSettings().getValue("ExtraInfo", "Publisher");
                    String gen = dfendExtra.getSettings().getValue("ExtraInfo", "Genre");
                    String year = dfendExtra.getSettings().getValue("ExtraInfo", "Year");
                    String status = Settings.getInstance().msg("dialog.dfendimport.defaultprofilestatus");
                    String cap = dfendExtra.getSettings().getValue("dosbox", "captures");
                    int devId = KeyValuePair.findIdByValue(dbase.readDevelopersList(), dev);
                    int publId = KeyValuePair.findIdByValue(dbase.readPublishersList(), pub);
                    int genId = KeyValuePair.findIdByValue(dbase.readGenresList(), gen);
                    int yrId = KeyValuePair.findIdByValue(dbase.readYearsList(), year);
                    int statId = KeyValuePair.findIdByValue(dbase.readStatusList(), status);
                    int[] custIDs = CUST_IDS;
                    for (int i = 0; i < 4; i++) {
                        custIDs[i] = KeyValuePair.findIdByValue(dbase.readCustomList(i), "");
                	}

                    try {
                        int profileId = dbase.addOrEditProfile(title, dev, pub, gen, year, status, setup, setupParams,
                                notes, favorite, devId, publId, genId, yrId, statId, defaultDBVersion.getId(),
                                LINKS, CUST_STRINGS, CUST_INTS, custIDs, LINKS, -1);

                        Conf dfendProfile = new Conf(new File(confFile), title, profileId, defaultDBVersion, ps);

                        String dstCap = FileUtils.constructCapturesDir(profileId);
                        String dstCapRelative = FileUtils.constructRelativeCapturesDir(profileId);
                        File dstCapAbsolute = FileUtils.canonicalToData(dstCap);
                        FileUtils.createDir(dstCapAbsolute);
                        FileUtils.copyFiles(new File(cap), dstCapAbsolute);
                        dfendProfile.getSettings().setValue("dosbox", "captures", dstCapRelative);

                        if (performCleanup) {
                            dfendProfile.getSettings().removeSection("directserial");
                            dfendProfile.getSettings().removeSection("modem");
                            dfendProfile.getSettings().removeSection("ipx");
                            dfendProfile.getSettings().removeSection("sdl");
                        }
                        
                        // The profile is associated to the Default DOSBox version
                        // However,the imported profile is probably still associated to an older DB version
                        // Therefore, update the settings to defaultDBVersion
                        dfendProfile.alterToDosboxVersionGeneration(dfendProfile);
                        
                        dfendProfile.save();

                        dbase.updateProfileConf(FileUtils.makeRelativeToData(dfendProfile.getConfFile()).getPath(), dstCap, profileId);

                        if (dfendProfile.getAutoexec().isIncomplete()) {
					    	messageLog.append(title).append(": ").append(Settings.getInstance().msg("dialog.multiprofile.error.profileincomplete")).append('\n');
					    }
                        if (bos.size() > 0) {
                        	messageLog.append(bos.toString());
                        	bos.reset();
                        }
                    } catch (SQLException se) {
                        messageLog.append(se.getMessage()).append('\n');
                    }
                } catch (IOException e) {
                    messageLog.append(e.getMessage()).append('\n');
                }

                if (display.isDisposed() || log.isDisposed() || progressBar.isDisposed()) {
                    break;
                }
                display.asyncExec(new Runnable() {
                    public void run() {
                        profileLabel.setText(Settings.getInstance().msg("dialog.dfendimport.importing", new Object[] {title}));
                        profileLabel.pack();
                        if (messageLog.length() > displayedLog.length()) {
                            String newOutput = messageLog.substring(displayedLog.length());
                            log.append(newOutput);
                            displayedLog.append(newOutput);
                        }
                        progressBar.setSelection(profileNumber++);
                    }
                });
            }
        } catch (Exception se) {
            messageLog.append(se.getMessage());
            display.asyncExec(new Runnable() {
                public void run() {
                    if (messageLog.length() > displayedLog.length()) {
                        String newOutput = messageLog.substring(displayedLog.length());
                        log.append(newOutput);
                        displayedLog.append(newOutput);
                    }
                }
            });
        }
        
        display.asyncExec(new Runnable() { public void run() { progressBar.setSelection(progressBar.getMaximum()); } });
    }
}
