/*
 *  Copyright (C) 2006-2013  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.test;

import static org.junit.Assert.*;
import java.io.File;
import org.junit.Test;
import org.dbgl.util.FileUtils;


public class FilesTest {

    @Test
    public void testGetCanonicalPath() {
        assertEquals("C:\\data\\bla.exe", FileUtils.canonical("c:\\data\\bla.exe"));
        assertEquals("C:\\Projects\\DBGL\\data\\bla.exe", FileUtils.canonical("data\\bla.exe"));
        assertEquals("C:\\data2\\bla.exe", FileUtils.canonical("c:\\data\\..\\data2\\bla.exe"));
        assertEquals("C:\\data\\bla.exe", FileUtils.canonical("c:\\data\\..\\data\\bla.exe"));
    }

    @Test
    public void testIsExistingFile() {
        assertTrue(FileUtils.isExistingFile(new File("com\\test\\FilesTest.java")));
        assertFalse(FileUtils.isExistingFile(new File("com\\test")));
    }

    @Test
    public void testAreRelated() {
        File f1 = FileUtils.canonicalToDosroot(".");
        File f2 = FileUtils.canonicalToDosroot("ACTION\\KEEN4\\KEEN4.EXE");
        assertTrue(FileUtils.areRelated(f1, f2));
        assertFalse(FileUtils.areRelated(f2, f1));
    }

    @Test
    public void testGetDosRoot() {
        assertEquals("D:\\Projects\\Personal\\DBGL\\dosroot", FileUtils.getDosRoot());
    }

    @Test
    public void testMakeRelativeTo() {
        File f1 = FileUtils.canonicalToDosroot(".");
        File f2 = FileUtils.canonicalToDosroot("ACTION\\KEEN4\\KEEN4.EXE");
        assertEquals("ACTION\\KEEN4\\KEEN4.EXE", FileUtils.makeRelativeTo(f2, f1).getPath());
    }

    @Test
    public void testIsAbsolute() {
    	File f1 = new File("c:\\bla.txt");

    	File f2 = new File("\\bla.txt");
    	File f3 = new File("c:bla.txt");
    	File f4 = new File("bla.txt");

    	assertEquals(f1.isAbsolute(), true);
    	assertEquals(f2.isAbsolute(), false);
    	assertEquals(f3.isAbsolute(), false);
    	assertEquals(f4.isAbsolute(), false);
    }
}
