/*
 *  Copyright (C) 2006-2013  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.test;

import static org.junit.Assert.assertEquals;
import java.io.IOException;
import java.util.List;
import org.dbgl.model.SearchEngineImageInformation;
import org.dbgl.model.WebProfile;
import org.dbgl.util.searchengine.MobyGamesSearchEngine;
import org.junit.Test;


public class MobyGamesTest {

	@Test
    public void testGetEntries() {
		try {
			List<WebProfile> entries1 = MobyGamesSearchEngine.getInstance().getEntries("doom", new String[] {"dos", "pc booter"});
			assertEquals(33, entries1.size());
			
			List<WebProfile> entries2 = MobyGamesSearchEngine.getInstance().getEntries("mario", new String[] {"dos", "pc booter"});
			assertEquals(10, entries2.size());
			
			List<WebProfile> entries3 = MobyGamesSearchEngine.getInstance().getEntries("noresultsplease", new String[] {"dos", "pc booter"});
			assertEquals(0, entries3.size());
		} catch (IOException e) {
			e.printStackTrace();
		}
    }
	
	@Test
    public void testGetEntryDetailsDoom() {
		try {
			WebProfile doom = new WebProfile();
			doom.setTitle("DOOM");
			doom.setPlatform("DOS");
			doom.setYear("1993");
			doom.setUrl("http://www.mobygames.com/game/dos/doom");
			doom = MobyGamesSearchEngine.getInstance().getEntryDetailedInformation(doom);
			assertEquals("id Software, Inc.", doom.getDeveloperName());
			assertEquals("id Software, Inc.", doom.getPublisherName());
			assertEquals("Action", doom.getGenre());
			assertEquals("1993", doom.getYear());
			assertEquals(80, doom.getRank());
			assertEquals("/game/dos/doom/cover-art/gameCoverId,3907/", doom.getCoreGameCoverUrl());
			
			SearchEngineImageInformation[] images = MobyGamesSearchEngine.getInstance().getEntryImages(doom, Integer.MAX_VALUE, Integer.MAX_VALUE, true);
			assertEquals(60, images.length);
			
			images = MobyGamesSearchEngine.getInstance().getEntryImages(doom, Integer.MAX_VALUE, Integer.MAX_VALUE, false);
			assertEquals(40, images.length);
			
			images = MobyGamesSearchEngine.getInstance().getEntryImages(doom, 2, 2, true);
			assertEquals(4, images.length);
			
		} catch (IOException e) {
			e.printStackTrace();
		}
    }
	
	@Test
    public void testGetEntryDetailsKingdomOfKroz() {
		try {
			WebProfile kingdomofkroz = new WebProfile();
			kingdomofkroz.setTitle("Kingdom of Kroz");
			kingdomofkroz.setPlatform("DOS");
			kingdomofkroz.setYear("1988");
			kingdomofkroz.setUrl("http://www.mobygames.com/game/dos/kingdom-of-kroz");
			kingdomofkroz = MobyGamesSearchEngine.getInstance().getEntryDetailedInformation(kingdomofkroz);
			assertEquals("Scott Miller", kingdomofkroz.getDeveloperName());
			assertEquals("Softdisk Publishing", kingdomofkroz.getPublisherName());
			assertEquals("Action", kingdomofkroz.getGenre());
			assertEquals("1988", kingdomofkroz.getYear());
			assertEquals(60, kingdomofkroz.getRank());
			assertEquals(null, kingdomofkroz.getCoreGameCoverUrl());
			
			SearchEngineImageInformation[] images = MobyGamesSearchEngine.getInstance().getEntryImages(kingdomofkroz, Integer.MAX_VALUE, Integer.MAX_VALUE, true);
			assertEquals(3, images.length);
			
			images = MobyGamesSearchEngine.getInstance().getEntryImages(kingdomofkroz, Integer.MAX_VALUE, Integer.MAX_VALUE, false);
			assertEquals(3, images.length);
			
		} catch (IOException e) {
			e.printStackTrace();
		}
    }
	
	@Test
    public void testGetEntryDetailsXargon() {
		try {
			WebProfile xargon = new WebProfile();
			xargon.setTitle("Xargon: The Mystery of the Blue Builders - Beyond Reality");
			xargon.setPlatform("DOS");
			xargon.setYear("1994");
			xargon.setUrl("http://www.mobygames.com/game/dos/xargon-the-mystery-of-the-blue-builders-beyond-reality");
			xargon = MobyGamesSearchEngine.getInstance().getEntryDetailedInformation(xargon);
			assertEquals("Epic MegaGames, Inc.", xargon.getDeveloperName());
			assertEquals("Monkey Business, Inc.", xargon.getPublisherName());
			assertEquals("Action", xargon.getGenre());
			assertEquals("1994", xargon.getYear());
			assertEquals(0, xargon.getRank());
			assertEquals("/game/dos/xargon-the-mystery-of-the-blue-builders-beyond-reality/cover-art/gameCoverId,7328/", xargon.getCoreGameCoverUrl());
			
			SearchEngineImageInformation[] images = MobyGamesSearchEngine.getInstance().getEntryImages(xargon, Integer.MAX_VALUE, Integer.MAX_VALUE, true);
			assertEquals(7, images.length);
			
			images = MobyGamesSearchEngine.getInstance().getEntryImages(xargon, Integer.MAX_VALUE, Integer.MAX_VALUE, false);
			assertEquals(2, images.length);
			
			images = MobyGamesSearchEngine.getInstance().getEntryImages(xargon, 1, 1, true);
			assertEquals(1, images.length);
			
		} catch (IOException e) {
			e.printStackTrace();
		}
    }
}
