/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package exodos;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.dbgl.gui.interfaces.ProgressNotifyable;
import org.dbgl.model.FileLocation;
import org.dbgl.model.GamePack;
import org.dbgl.model.SearchResult;
import org.dbgl.model.SearchResult.ResultType;
import org.dbgl.model.aggregate.DosboxVersion;
import org.dbgl.model.aggregate.Profile;
import org.dbgl.model.conf.Autoexec;
import org.dbgl.model.conf.Settings;
import org.dbgl.model.conf.mount.DirMount;
import org.dbgl.model.conf.mount.ImageMount;
import org.dbgl.model.conf.mount.Mount;
import org.dbgl.model.entity.GamePackEntry;
import org.dbgl.model.factory.ProfileFactory;
import org.dbgl.model.repository.DosboxVersionRepository;
import org.dbgl.service.FileLocationService;
import org.dbgl.service.ITextService;
import org.dbgl.service.ImportExportProfilesService;
import org.dbgl.service.TextService;
import org.dbgl.util.FilesUtils;
import org.dbgl.util.ShortFilenameUtils;
import org.dbgl.util.StringRelatedUtils;
import org.dbgl.util.archive.ZipUtils;


public class Convert {

	private static final List<String> GAME_ARCS = Arrays.asList("GamesSTR.zip", "GamesSIM.zip", "GamesRPG.zip", "GamesADV.zip", "GamesACT.zip", "GamesWIN.zip", "Games.zip");

	private static final String GAMES_DIR = "games";
	private static final String UTIL_DIR = "util";

	private static final String MEAGRE_DIR = "Meagre";
	private static final String INIFILE_DIR = "IniFile";
	private static final String ABOUT_DIR = "About";

	private static final String EXTRAS_DIR = "Extras";
	private static final String MANUAL_DIR = "Manual";

	private static final String FRONT_DIR = "Front";
	private static final String BACK_DIR = "Back";
	private static final String MEDIA_DIR = "Media";
	private static final String ADVERT_DIR = "Advert";
	private static final String TITLE_DIR = "Title";
	private static final String SCREEN_DIR = "Screen";
	private static final String[] CAP_DIRS = {TITLE_DIR, SCREEN_DIR, FRONT_DIR, BACK_DIR, MEDIA_DIR, ADVERT_DIR};

	private static final String GPA_TITLE = "eXoDOS conversion";
	private static final String GPA_NOTES = StringUtils.EMPTY;
	private static final String GPA_AUTHOR = StringUtils.EMPTY;

	private static final String CONVERTER_TITLE = "eXoDOS converter";
	private static final String CONVERTER_VERSION = "0.91";

	private static final long BYTES_IN_MB = 1024L * 1024L;
	private static final long MAX_PART_SIZE_DEFAULT_IN_MB = 1024L * 16L;
	private static final String[] CDIMAGES = {".iso", ".cue", ".bin", ".img", ".gog"};

	private static final ITextService TEXT = TextService.getInstance();

	private static boolean verboseOutput_ = false;
	private static long maxPartSizeInMB_ = MAX_PART_SIZE_DEFAULT_IN_MB;

	private static Set<File> zipfilesUsingAltEncoding;

	public static final class FileIgnoreCaseComparator implements Comparator<File> {
		public int compare(File file1, File file2) {
			return file1.getPath().compareToIgnoreCase(file2.getPath());
		}
	}

	public static void main(String[] args) {
		System.out.println("Converts eXoDOS game packages into DBGL GamePackArchives (v" + CONVERTER_VERSION + ")\n");

		if (args.length < 2 || args.length > 6)
			displaySyntax();

		File inputDir = new File(args[0]);
		File tmpDir = new File(args[1]);

		boolean analyzeOnly = false;
		boolean keepExtractedMetaData = false;

		if (args.length > 2) {
			for (int i = 2; i < args.length; i++) {
				if (args[i].equalsIgnoreCase("-a"))
					analyzeOnly = true;
				else if (args[i].equalsIgnoreCase("-k"))
					keepExtractedMetaData = true;
				else if (args[i].equalsIgnoreCase("-v"))
					verboseOutput_ = true;
				else if (args[i].toLowerCase().startsWith("-s:")) {
					try {
						maxPartSizeInMB_ = Long.parseLong(args[i].substring(3));
					} catch (NumberFormatException e) {
						// ignore, use the default value
					}
				} else
					displaySyntax();
			}
		}

		zipfilesUsingAltEncoding = new HashSet<>();

		List<File> inputGamesZips = validateParameters(inputDir, tmpDir);
		File gamesDir = new File(inputDir, GAMES_DIR);

		if (analyzeOnly)
			System.out.println("* Analyze only");
		if (keepExtractedMetaData)
			System.out.println("* Keeping extracted data after processing");
		if (verboseOutput_)
			System.out.println("* Verbose output");
		if (maxPartSizeInMB_ != MAX_PART_SIZE_DEFAULT_IN_MB)
			System.out.println("* Target size of the GamePackArchives: " + maxPartSizeInMB_ + "MB");

		if (analyzeOnly || keepExtractedMetaData || verboseOutput_ || (maxPartSizeInMB_ != MAX_PART_SIZE_DEFAULT_IN_MB))
			System.out.println();

		List<DosboxVersion> dbversionsList = null;
		try {

			DosboxVersionRepository dosboxRepo = new DosboxVersionRepository();
			dbversionsList = dosboxRepo.listAll();

			if (DosboxVersionRepository.findDefault(dbversionsList) == null) {
				SearchResult result = FileLocationService.getInstance().findDosbox();
				if (result.result_ == ResultType.COMPLETE) {
					new DosboxVersionRepository().add(result.dosbox_);
					dbversionsList = dosboxRepo.listAll();
				}
				if (DosboxVersionRepository.findDefault(dbversionsList) == null) {
					System.out.println("DOSBox installation could not be located, exiting.");
					System.exit(1);
				}
			}

			if (verboseOutput_)
				System.out.println("Using DOSBox installation located in: [" + DosboxVersionRepository.findDefault(dbversionsList).getConfigurationCanonicalFile().getPath() + "]");
		} catch (SQLException e) {
			e.printStackTrace();
			System.exit(1);
		}

		DosboxVersion defaultDosboxVersion = DosboxVersionRepository.findDefault(dbversionsList);

		for (File inputGamesZip: inputGamesZips) {
			File tmpDirForGameZip = new File(tmpDir, FilenameUtils.getBaseName(inputGamesZip.getName()));

			File gamesGamesDir = extractMeagreMetaData(inputGamesZip, tmpDirForGameZip);

			GamePack gamePack = analyzeMeagreMetaData(gamesDir, gamesGamesDir, defaultDosboxVersion);

			if (!analyzeOnly)
				generateGamePackArchives(gamePack, tmpDirForGameZip, FilenameUtils.removeExtension(inputGamesZip.getName()), dbversionsList);

			if (!keepExtractedMetaData)
				cleanup(gamesGamesDir);
		}
	}

	private static void displaySyntax() {
		System.out.println("Use: Convert <inputexodosdir> <dstdir> [-a] [-k] [-v] [-s:size]");
		System.out.println("-a\t\tAnalyze only, don't generate GamePackArchives");
		System.out.println("-k\t\tKeep extracted meta data files after processing");
		System.out.println("-v\t\tVerbose output");
		System.out.println("-s:size\t\tTarget size of the GamePackArchives in MB, " + MAX_PART_SIZE_DEFAULT_IN_MB + " is the default (= " + MAX_PART_SIZE_DEFAULT_IN_MB / 1024L + " GB packages)");
		System.exit(1);
	}

	private static List<File> validateParameters(File inputDir, File tmpDir) {
		if (!inputDir.exists()) {
			System.out.println("The directory [" + inputDir + "] does not exist, exiting.");
			System.exit(1);
		}
		File gamesDir = new File(inputDir, GAMES_DIR);
		if (!gamesDir.exists()) {
			System.out.println("The directory [" + inputDir + "] does not contain the [" + GAMES_DIR + "] directory, exiting.");
			System.exit(1);
		}
		File utilDir = new File(inputDir, UTIL_DIR);
		if (!utilDir.exists()) {
			System.out.println("The directory [" + inputDir + "] does not contain the [" + UTIL_DIR + "] directory, exiting.");
			System.exit(1);
		}

		List<File> inputGamesZip = new ArrayList<>();
		for (String filename: GAME_ARCS) {
			File file = new File(gamesDir, filename);
			if (FilesUtils.isExistingFile(file))
				inputGamesZip.add(file);
		}

		if (inputGamesZip.isEmpty()) {
			System.out.println("None of the files [" + StringUtils.join(GAME_ARCS, ", ") + "] are found, exiting.");
			System.exit(1);
		}
		if (!tmpDir.exists()) {
			System.out.println("The directory [" + tmpDir + "] does not exist, exiting.");
			System.exit(1);
		}
		return inputGamesZip;
	}

	private static File extractMeagreMetaData(File inputGamesZip, File tmpDir) {
		System.out.println();
		System.out.println("===========================================");
		System.out.println(" Phase 1 of 3: Extracting Meagre meta-data");
		System.out.println("===========================================");
		System.out.println("Reading from: [" + inputGamesZip + "]");
		System.out.println("Writing to:   [" + tmpDir + "]");

		try {
			String mainGamesDir = findMainFolder(inputGamesZip);
			if (mainGamesDir == null) {
				System.out.println("The file [" + inputGamesZip + "] does not seem to have an inner games directory, exiting.");
				System.exit(1);
			}
			File gamesGamesDir = new File(tmpDir, mainGamesDir);
			if (!gamesGamesDir.exists()) {
				unzip(inputGamesZip, tmpDir);
			} else {
				System.out.println("Skipping extraction of [" + inputGamesZip + "] since [" + gamesGamesDir + "] already exists");
			}
			return gamesGamesDir;
		} catch (IOException e) {
			System.out.println("The file [" + inputGamesZip + "] did not fully extract into the [" + tmpDir + "] directory, exiting.");
			e.printStackTrace();
			System.exit(1);
		}
		return null;
	}

	private static GamePack analyzeMeagreMetaData(File gamesDir, File gamesGamesDir, DosboxVersion defaultDosboxVersion) {
		System.out.println();
		System.out.println("==========================================");
		System.out.println(" Phase 2 of 3: Analyzing Meagre meta-data");
		System.out.println("==========================================");
		System.out.println("Reading from: [" + gamesGamesDir + "]");

		GamePack gamePack = new GamePack();
		gamePack.setCreationApp(CONVERTER_TITLE);
		gamePack.setCreationAppVersion(CONVERTER_VERSION);
		gamePack.setCreationDate(new Date());
		gamePack.setTitle(GPA_TITLE);
		gamePack.setAuthor(GPA_AUTHOR);
		gamePack.setNotes(GPA_NOTES);
		gamePack.setCapturesAvailable(true);
		gamePack.setGamedataAvailable(true);
		gamePack.setMapperfilesAvailable(false);
		gamePack.setNativecommandsAvailable(false);
		gamePack.setVersion(ImportExportProfilesService.PROFILES_XML_FORMAT_VERSION);
		gamePack.setDosboxVersions(Collections.singleton(defaultDosboxVersion));

		File[] gameDirs = gamesGamesDir.listFiles();
		Arrays.sort(gameDirs, new FileIgnoreCaseComparator());

		for (int i = 0; i < gameDirs.length; i++) {

			File gameDir = gameDirs[i];
			File meagreDir = new File(gameDir, MEAGRE_DIR);
			File iniFileDir = new File(meagreDir, INIFILE_DIR);

			String gameDirName = gameDir.getName();

			File[] iniFiles = iniFileDir.listFiles(new FileFilter() {
				public boolean accept(File file) {
					return file.isFile() && file.getName().toLowerCase().endsWith(".ini");
				}
			});
			if (iniFiles.length != 1) {
				System.out.println("WARNING: " + gameDirName + ": Not exactly 1 ini file found (" + iniFiles.length + ")");
			}
			File iniFile = iniFiles.length > 0 ? iniFiles[0]: null;

			try {

				Profile profile = null;
				File zipFile = null;

				boolean favorite = false;
				String confPathAndFile = new File(gameDir, FileLocationService.DOSBOX_CONF_STRING).getPath();

				List<File> capFiles = new ArrayList<>();
				List<File> extraFiles = new ArrayList<>();

				if (iniFile != null) {

					Settings iniConf = new Settings();
					iniConf.setFileLocation(new FileLocation(iniFile.getPath()));
					System.out.print(iniConf.load(TEXT));

					String title = iniConf.getValue("Main", "name", StringUtils.EMPTY);
					String developer = join(new String[] {iniConf.getValue("Main", "developer", StringUtils.EMPTY), iniConf.getValue("Main", "designer", StringUtils.EMPTY),
							iniConf.getValue("Main", "designer2", StringUtils.EMPTY)},
						", ");
					String publisher = iniConf.getValue("Main", "publisher", StringUtils.EMPTY);
					String genre = join(new String[] {iniConf.getValue("Main", "genre", StringUtils.EMPTY), iniConf.getValue("Main", "subgenre", StringUtils.EMPTY),
							iniConf.getValue("Main", "subgenre2", StringUtils.EMPTY)},
						", ");
					String year = iniConf.getValue("Main", "year", StringUtils.EMPTY);
					String status = StringUtils.EMPTY;
					String aboutFilename = iniConf.getValue("Main", "about", StringUtils.EMPTY);
					String notes = StringUtils.isBlank(aboutFilename) ? StringUtils.EMPTY
							: org.apache.commons.io.FileUtils.readFileToString(new File(meagreDir, ABOUT_DIR + File.separatorChar + aboutFilename), Charset.defaultCharset());

					String[] links = {iniConf.getValue("Main", "extralink1", StringUtils.EMPTY), iniConf.getValue("Main", "manual", StringUtils.EMPTY),
							iniConf.getValue("Main", "extralink2", StringUtils.EMPTY), iniConf.getValue("Main", "extralink3", StringUtils.EMPTY),
							iniConf.getValue("Main", "extralink4", StringUtils.EMPTY), iniConf.getValue("Main", "extralink5", StringUtils.EMPTY),
							iniConf.getValue("Main", "extralink6", StringUtils.EMPTY), iniConf.getValue("Main", "extralink7", StringUtils.EMPTY)};
					String[] linkTitles = {iniConf.getValue("Main", "extra1", StringUtils.EMPTY), "Manual", iniConf.getValue("Main", "extra2", StringUtils.EMPTY),
							iniConf.getValue("Main", "extra3", StringUtils.EMPTY), iniConf.getValue("Main", "extra4", StringUtils.EMPTY), iniConf.getValue("Main", "extra5", StringUtils.EMPTY),
							iniConf.getValue("Main", "extra6", StringUtils.EMPTY), iniConf.getValue("Main", "extra7", StringUtils.EMPTY)};
					String[] capFilenames = {iniConf.getValue("Main", "title01", StringUtils.EMPTY), iniConf.getValue("Main", "screen01", StringUtils.EMPTY),
							iniConf.getValue("Main", "front01", StringUtils.EMPTY), iniConf.getValue("Main", "back01", StringUtils.EMPTY), iniConf.getValue("Main", "media01", StringUtils.EMPTY),
							iniConf.getValue("Main", "adv01", StringUtils.EMPTY)};

					for (int c = 0; c < capFilenames.length; c++) {
						if (StringUtils.isNotEmpty(capFilenames[c])) {
							File capFile = new File(meagreDir, CAP_DIRS[c] + File.separatorChar + capFilenames[c]);
							if (FilesUtils.isExistingFile(capFile))
								capFiles.add(capFile);
							else if (verboseOutput_)
								System.out.println(gameDirName + ": capture [" + capFile + "] not found, skipped");
						}
					}

					for (int c = 0; c < links.length; c++) {
						String link = links[c];
						if (StringUtils.isNotEmpty(link)) {
							if (!link.toLowerCase().startsWith("http")) {
								File extraFile = new File(meagreDir, ((c == 1) ? MANUAL_DIR: EXTRAS_DIR) + File.separatorChar + link);
								if (FilesUtils.isExistingFile(extraFile)) {
									extraFiles.add(extraFile);
									links[c] = FileLocationService.DOSROOT_DIR_STRING + gameDirName + File.separatorChar + EXTRAS_DIR + File.separatorChar + extraFile.getName();
								} else if (verboseOutput_)
									System.out.println(gameDirName + ": linked file [" + extraFile + "] not found, skipped");
							}
						}
					}

					profile = ProfileFactory.create(i, title, developer, publisher, genre, year, status, notes, favorite, links, linkTitles, defaultDosboxVersion, confPathAndFile);

					zipFile = new File(gamesDir, FilenameUtils.getBaseName(iniConf.getValue("Main", "executable", StringUtils.EMPTY)) + ".zip");

				} else {

					String[] links = new String[Profile.NR_OF_LINK_TITLES];
					Arrays.fill(links, StringUtils.EMPTY);
					profile = ProfileFactory.create(i, gameDirName, StringUtils.EMPTY, StringUtils.EMPTY, StringUtils.EMPTY, StringUtils.EMPTY, StringUtils.EMPTY, StringUtils.EMPTY, favorite, links,
						links, defaultDosboxVersion, confPathAndFile);

				}

				System.out.print(profile.resetAndLoadConfiguration());

				// minor bit of clean up of the actual dosbox configuration
				if ("64".equals(profile.getConfiguration().getValue("dosbox", "memsize")))
					profile.getConfiguration().setValue("dosbox", "memsize", "63");
				if ("overlay".equals(profile.getConfiguration().getValue("sdl", "output")))
					profile.getConfiguration().removeValue("sdl", "output");

				Autoexec autoexec = profile.getConfiguration().getAutoexec();
				autoexec.migrate(new FileLocation(GAMES_DIR, FileLocationService.getInstance().dosrootRelative()), FileLocationService.getInstance().getDosrootLocation());

				// Some extra sanity-checking on install.bat
				File gameFolderInstallbat = null, zipFileInstallbat = null;

				File installFile = new File(gameDir, "Install.bat");
				if (FilesUtils.isExistingFile(installFile)) {
					List<String> lines = org.apache.commons.io.FileUtils.readLines(installFile, Charset.defaultCharset());
					for (String s: lines) {
						if (s.startsWith("IF EXIST \"")) {
							int secondQuotes = s.lastIndexOf('\"');
							if (secondQuotes != -1) {
								gameFolderInstallbat = new File(s.substring(10, secondQuotes));
								if (!gameFolderInstallbat.getName().equalsIgnoreCase(gameDirName))
									System.out.println("WARNING: " + gameDirName
											+ ": This game's folder as found in games???.zip does not match the folder being checked for existence in install.bat (" + gameFolderInstallbat + ")");
							}
						} else if (s.startsWith("unzip \"")) {
							int secondQuotes = s.lastIndexOf('\"');
							if (secondQuotes != -1) {
								zipFileInstallbat = new File(gamesDir, s.substring(7, secondQuotes));
								if (!zipFileInstallbat.equals(zipFile) && verboseOutput_)
									System.out.println(gameDirName + ": This game's 'Main Executable' referenced in iniFile (" + zipFile + ") does not match the file being unzipped in install.bat ("
											+ zipFileInstallbat + ")");
							}
						}
					}
				} else {
					System.out.println("WARNING: " + gameDirName + ": This game's install.bat not found");
				}

				if (!FilesUtils.isExistingFile(zipFile)) {
					if ((zipFileInstallbat != null) && FilesUtils.isExistingFile(zipFileInstallbat)) {
						if (verboseOutput_)
							System.out.println("Zip file " + zipFile + " is missing but " + zipFileInstallbat + " does exist, using that file instead");
						zipFile = zipFileInstallbat;
					} else {
						if (gameDirName.equals("IBMMJ")) {
							zipFile = new File(gamesDir, "Mahjong (1986).zip");
							gameFolderInstallbat = new File("Mahjng86");
							autoexec.setGameMainPath(gameFolderInstallbat);
						} else if (gameDirName.equals("ANightma")) {
							zipFile = new File(gamesDir, "Nightmare on Elm Street, A (1989).zip");
						} else if (gameDirName.equals("BluBro")) {
							zipFile = new File(gamesDir, "Blues Brothers, The (1991).zip");
						} else if (gameDirName.equals("JimPower")) {
							zipFile = new File(gamesDir, "Jim Power - The Lost Dimension in 3D (1993).zip.zip");
						} else if (gameDirName.equals("RadixBey")) {
							zipFile = new File(gamesDir, "Radix - Beyond the Void (1995).zip");
						} else if (gameDirName.equals("SidLin96")) {
							zipFile = new File(gamesDir, "SideLine (1996).zip");
						} else if (gameDirName.equals("Sinaria")) {
							zipFile = new File(gamesDir, "Sinaria - Lost in Space (1994).zip");
						} else if (gameDirName.equals("Targ1982")) {
							zipFile = new File(gamesDir, "Target (IBM)(1982).zip");
						} else if (gameDirName.equals("bio101")) {
							zipFile = new File(gamesDir, "Biology 101 (1987).zip");
						} else if (gameDirName.equals("TheWorl")) {
							zipFile = new File(gamesDir, "World Name Game, The (1989).zip");
						} else if (gameDirName.equals("TNFS")) {
							zipFile = new File(gamesDir, "Need for Speed, The (1995).zip");
						} else if (gameDirName.equals("AFAB1985")) {
							zipFile = new File(gamesDir, "Fable, A (1985).zip");
						} else if (gameDirName.equals("shogun86")) {
							zipFile = new File(gamesDir, "James Clavell's Shogun (1986).zip");
						}
					}
				}

				if (zipUsingAltEncoding(zipFile))
					zipfilesUsingAltEncoding.add(zipFile);

				List<File> foldersInZip = readFoldersInZip(zipFile);
				if (gameFolderInstallbat != null && !foldersInZip.contains(new File(gameFolderInstallbat.getName()))) {
					if (zipFileInstallbat != null && FilesUtils.isExistingFile(zipFileInstallbat)) {
						List<File> foldersInZipFileInstallBat = readFoldersInZip(zipFileInstallbat);
						if (foldersInZipFileInstallBat.contains(new File(gameFolderInstallbat.getName()))) {
							zipFile = zipFileInstallbat;
							if (verboseOutput_)
								System.out.println("Zip file " + zipFile + " contains the game folder [\" + gameFolderInstallbat.getName() + \"], using that file instead");
						} else {
							System.out.println("WARNING: " + gameDirName + ": Game folder [" + gameFolderInstallbat.getName() + "] not found inside [" + zipFile + "] or [" + zipFileInstallbat + "]");
						}
					} else {
						System.out.println("WARNING: " + gameDirName + ": Game folder [" + gameFolderInstallbat.getName() + "] not found inside [" + zipFile + "]");
					}
				}

				if (!FilesUtils.isExistingFile(zipFile)) {
					if (verboseOutput_)
						System.out.println("Zip file " + zipFile + " not found, reverting to Install.bat to determine game zip");
					if (zipFileInstallbat != null && FilesUtils.isExistingFile(zipFileInstallbat)) {
						zipFile = zipFileInstallbat;
						if (verboseOutput_)
							System.out.println("Zip file " + zipFile + " referenced in Install.bat, using that file instead");
					}
				}

				List<File> filesInZip = readFilesInZip(zipFile);
				if (!fixupFileLocations(filesInZip, profile, zipFile)) {
					String main = autoexec.getGameMain();
					if (StringUtils.isNotEmpty(main))
						System.out.println("WARNING: " + gameDirName + ": Main file [" + main + "] not found inside [" + zipFile + "]");
				}

				boolean multipleRootEntries = isMultipleRootEntries(filesInZip);
				if (multipleRootEntries) {
					autoexec.setBaseDir(gameDir);
					if (verboseOutput_)
						System.out.println("INFO: " + gameDirName + " is moved one directory level deeper");
				}

				autoexec.migrate(FileLocationService.getInstance().getDosrootLocation(), new FileLocation(gameDirName, FileLocationService.getInstance().dosrootRelative()));

				gamePack.getEntries().add(new GamePackEntry(i, profile, gameDirName, capFiles, extraFiles, zipFile));

			} catch (IOException e) {
				System.out.println("SKIPPED " + gameDirName + " " + e.toString());
			}
		}
		Collections.sort(gamePack.getEntries());

		System.out.println("Analysis done");
		return gamePack;
	}

	private static void generateGamePackArchives(GamePack gamePack, File tmpDir, String baseFilename, List<DosboxVersion> dbversionsList) {
		try {
			System.out.println();
			System.out.println("===========================================");
			System.out.println(" Phase 3 of 3: Generating GamePackArchives");
			System.out.println("===========================================");

			List<GamePackEntry> remainingGamePackEntries = new ArrayList<>(gamePack.getEntries());

			while (!remainingGamePackEntries.isEmpty()) {

				long totalSize = 0L;
				List<GamePackEntry> currentGamePackEntries = new ArrayList<>();
				boolean reachedMaxSize = false;

				while (!remainingGamePackEntries.isEmpty() && !reachedMaxSize) {
					GamePackEntry gamePackEntry = remainingGamePackEntries.get(0);
					try {
						long gameSize = determineSize(gamePackEntry);
						if (currentGamePackEntries.isEmpty() || ((totalSize + gameSize) < (maxPartSizeInMB_ * BYTES_IN_MB))) {
							currentGamePackEntries.add(gamePackEntry);
							remainingGamePackEntries.remove(0);
							totalSize += gameSize;
						} else {
							reachedMaxSize = true;
						}
					} catch (IOException e) {
						System.out.println("skipping " + gamePackEntry.getProfile().getTitle() + ", " + e.toString());
						remainingGamePackEntries.remove(0);
					}
				}

				File currentOutputGpa = new File(tmpDir,
						baseFilename + "__" + FilesUtils.toSafeFilename(currentGamePackEntries.get(0).getProfile().getTitle())
								+ (currentGamePackEntries.size() > 1 ? " - " + FilesUtils.toSafeFilename(currentGamePackEntries.get(currentGamePackEntries.size() - 1).getProfile().getTitle())
										: StringUtils.EMPTY)
								+ FilesUtils.GAMEPACKARCHIVE_EXT);

				ZipOutputStream zipOutputStream = new ZipOutputStream(new FileOutputStream(currentOutputGpa));

				for (GamePackEntry game: currentGamePackEntries) {
					try {
						Profile prof = game.getProfile();
						System.out.print("Exporting " + prof.getTitle() + " ");

						try {
							List<File> capsList = game.getCapturesList();
							for (int c = 0; c < capsList.size(); c++) {
								File srcCapFile = capsList.get(c);
								ZipUtils.zipEntry(zipOutputStream, srcCapFile, new File(game.getArchiveCapturesDir(), String.valueOf(c) + "_" + srcCapFile.getName()));
							}
						} catch (IOException e) {
							throw new IOException(TEXT.get("dialog.export.error.exportcaptures", new Object[] {prof.getTitle(), StringRelatedUtils.toString(e)}), e);
						}

						File relativeGameDirInZip = game.getArchiveGameDir();
						File relativeExtrasGameDirInZip = new File(relativeGameDirInZip, EXTRAS_DIR);
						try {
							for (File srcExtraFile: game.getExtrasList()) {
								ZipUtils.zipEntry(zipOutputStream, srcExtraFile, new File(relativeExtrasGameDirInZip, srcExtraFile.getName()));
							}
							System.out.print('.');
							copyZipData(game.getZipFile(), relativeGameDirInZip, zipOutputStream);
						} catch (IOException e) {
							throw new IOException(TEXT.get("dialog.export.error.exportgamedata", new Object[] {prof.getTitle(), StringRelatedUtils.toString(e)}), e);
						}

					} catch (IOException e2) {
						System.out.println("WARNING: The file [" + game.getZipFile() + "] could not be copied (completely) properly into the [" + currentOutputGpa + "], this game may be corrupt");
						e2.printStackTrace();
					}
				}

				ImportExportProfilesService.export(gamePack, currentGamePackEntries, zipOutputStream);

				zipOutputStream.close();
				System.out.println("DBGL GamePackArchive " + currentOutputGpa + " successfully generated");
			}

			System.out.println("Finished.");

		} catch (ParserConfigurationException | TransformerException | IOException e) {
			e.printStackTrace();
		}
	}

	private static void cleanup(File gamesDir) {
		try {
			org.apache.commons.io.FileUtils.deleteDirectory(gamesDir);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private static boolean isMultipleRootEntries(List<File> list) {
		int found = 0;
		for (File file: list) {
			if (file.getParentFile() == null)
				found++;
		}
		return found >= 1;
	}

	private static boolean fixupFileLocations(List<File> list, Profile profile, File zipFile) {
		for (Mount m: profile.getNettoMountingPoints()) {
			if (m instanceof ImageMount) {
				File[] files = ((ImageMount)m).getImgPaths();
				String[] newFiles = ((ImageMount)m).getImgPathStrings();
				for (int i = 0; i < files.length; i++) {
					File f = files[i];
					if (!list.contains(f)) {
						File dst = findDir(list, f);
						if (dst != null) {
							newFiles[i] = dst.getPath();
							if (verboseOutput_)
								System.out.println("Image-mounted file [" + f + "] redirected to [" + dst + "]");
						} else {
							System.out.println("WARNING: Image-mounted file [" + f + "] not found inside [" + zipFile + "]");
						}
					}
				}
				((ImageMount)m).setImgPaths(newFiles);
			}
		}

		Autoexec autoexec = profile.getConfiguration().getAutoexec();
		String main = autoexec.getGameMain();
		File mainFile = new File(main);

		int isoIdx = containsIso(mainFile.getPath());
		if (isoIdx != -1)
			mainFile = new File(mainFile.getPath().substring(0, isoIdx));

		int fatIdx = FilesUtils.fatImageIndex(mainFile.getPath());
		if (fatIdx != -1)
			mainFile = new File(mainFile.getPath().substring(0, fatIdx));

		if (findMainFile(list, autoexec, profile.getNettoMountingPoints(), mainFile))
			return true;

		if (mainFile.getName().contains("~")) {
			List<File> shortFilesList = ShortFilenameUtils.convertToShortFileSet(list);
			return findMainFile(shortFilesList, autoexec, profile.getNettoMountingPoints(), mainFile);
		}

		return false;
	}

	private static boolean findMainFile(List<File> list, Autoexec autoexec, List<Mount> nettoMountingPoints, File mainFile) {
		if (list.contains(mainFile))
			return true;

		File newMainFile = findSuitableExtension(mainFile, list);
		if (newMainFile != null) {
			autoexec.setGameMain(newMainFile.getPath());
			if (verboseOutput_)
				System.out.println("Main file [" + mainFile + "] was using wrong file extension, changed to [" + newMainFile.getPath() + "]");
			return true;
		}

		File parent1 = mainFile.getParentFile();
		if (parent1 != null) {
			File parent2 = parent1.getParentFile();
			if (parent2 != null) {
				newMainFile = new File(parent2, mainFile.getName());
				if (list.contains(newMainFile)) {
					autoexec.setGameMainPath(parent2);
					if (verboseOutput_)
						System.out.println("Main file [" + mainFile + "] redirected to parent directory [" + parent2.getPath() + "]");
					return true;
				}

				newMainFile = findSuitableExtension(newMainFile, list);
				if (newMainFile != null) {
					autoexec.setGameMain(newMainFile.getPath());
					if (verboseOutput_)
						System.out.println("Main file [" + mainFile + "] was using wrong file extension and dir, changed to [" + newMainFile.getPath() + "]");
					return true;
				}
			}
		}

		String[] setPaths = autoexec.getSetPathsFromCustomSections();
		if (setPaths != null && mainFile.getName().toLowerCase().startsWith("win")) {
			File mainBaseFolder = mainFile.getParentFile();
			for (String setPath: setPaths) {
				char pd = setPath.toUpperCase().charAt(0);

				for (Mount m: nettoMountingPoints) {
					if (m instanceof DirMount && m.getDrive() == pd) {
						File cp = new File(((DirMount)m).getPathString(), setPath.substring(3));
						File f1 = new File(cp, mainFile.getName());
						newMainFile = findSuitableExtension(f1, list);
						if (newMainFile != null) {
							autoexec.setGameMain(newMainFile.getPath());
							if (verboseOutput_)
								System.out.println("Main file [" + mainFile + "] located using set path, changed to [" + newMainFile.getPath() + "]");

							// Check and fix path to Windows parameter executable(s)
							String params = autoexec.getParameters();
							if (StringUtils.isNotEmpty(params)) {
								String[] paramArray = StringUtils.split(params);
								String[] fixedParamArray = StringUtils.split(params);
								for (int i = 0; i < paramArray.length; i++) {
									if (paramArray[i].startsWith("/") || paramArray[i].startsWith("-"))
										continue; // unlikely to be file parameter, accept in any case

									String p = fixParameterPath(list, mainBaseFolder, ((DirMount)m).getPathString(), paramArray[i]);
									if (p == null) {
										if (verboseOutput_)
											System.out.println("INFO: Parameter [" + paramArray[i] + "] not found, might not be a file or folder");
									} else {
										fixedParamArray[i] = p;
									}
								}
								autoexec.setParameters(StringUtils.join(fixedParamArray, ' '));
								if (verboseOutput_)
									System.out.println("Main file parameter(s) [" + params + "] changed to [" + autoexec.getParameters() + "]");
							}
							return true;
						}
					}
				}
			}
		}

		return false;
	}

	private static String fixParameterPath(List<File> list, File mainBaseFolder, String mountPath, String param) {
		File newParamFile = findSuitableExtension(new File(FilenameUtils.normalize(new File(mainBaseFolder, param).getPath())), list);
		if (newParamFile != null)
			return newParamFile.getPath().substring(mountPath.length());
		newParamFile = findSuitableExtension(new File(FilenameUtils.normalize(new File(mainBaseFolder.getParentFile(), param).getPath())), list);
		if (newParamFile != null)
			return newParamFile.getPath().substring(mountPath.length());
		newParamFile = findSuitableExtension(new File(FilenameUtils.normalize(new File(mountPath, param).getPath())), list);
		if (newParamFile != null)
			return newParamFile.getPath().substring(mountPath.length());
		return null;
	}

	private static File findDir(List<File> list, File file) {
		for (File f: list)
			if (f.getName().equals(file.getName()))
				return new File(f.getParentFile(), f.getName());
		return null;
	}

	private static int containsIso(String mountPath) {
		for (String ext: CDIMAGES) {
			int idx = mountPath.toLowerCase().indexOf(ext + File.separatorChar);
			if (idx != -1) {
				return idx + ext.length();
			}
		}
		return -1;
	}

	private static boolean zipUsingAltEncoding(File zipFile) throws IOException {
		ZipFile zfile = null;
		try {
			zfile = new ZipFile(zipFile);
			for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
				try {
					entries.nextElement();
				} catch (IllegalArgumentException e) {
					return true;
				}
			}
		} finally {
			if (zfile != null)
				zfile.close();
		}
		return false;
	}

	private static String getZipEntryName(ZipEntry entry) {
		return entry.getName().replace((char)15, '\u263C');
	}

	private static List<File> readFilesInZip(File zipFile) throws IOException {
		List<File> result = new ArrayList<>();
		ZipFile zfile = null;
		try {
			zfile = new ZipFile(zipFile, zipfilesUsingAltEncoding.contains(zipFile) ? Charset.forName("CP437"): StandardCharsets.UTF_8);
			for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
				try {
					ZipEntry entry = entries.nextElement();
					String name = getZipEntryName(entry);
					if (!entry.isDirectory()) {
						result.add(new File(FilesUtils.toNativePath(name)));
					}
				} catch (IllegalArgumentException e) {
					System.out.println("WARNING: Zip file [" + zipFile + "] contains an entry with problematic characters in its filename");
				}
			}
		} finally {
			if (zfile != null)
				zfile.close();
		}
		return result;
	}

	private static List<File> readFoldersInZip(File zipFile) throws IOException {
		List<File> result = new ArrayList<>();
		ZipFile zfile = null;
		try {
			zfile = new ZipFile(zipFile, zipfilesUsingAltEncoding.contains(zipFile) ? Charset.forName("CP437"): StandardCharsets.UTF_8);
			for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
				try {
					ZipEntry entry = entries.nextElement();
					File filename = new File(FilesUtils.toNativePath(getZipEntryName(entry)));
					if (entry.isDirectory() && !result.contains(filename)) {
						result.add(filename);
					} else {
						File folder = filename.getParentFile();
						if (folder != null && !result.contains(folder))
							result.add(folder);
					}
				} catch (IllegalArgumentException e) {
					// Ignore, warning already given
				}
			}
		} finally {
			if (zfile != null)
				zfile.close();
		}
		return result;
	}

	private static long determineSize(GamePackEntry game) throws IOException {
		try {
			long result = org.apache.commons.io.FileUtils.sizeOf(game.getZipFile());
			for (File file: game.getExtrasList())
				result = result + org.apache.commons.io.FileUtils.sizeOf(file);
			for (File file: game.getCapturesList())
				result = result + org.apache.commons.io.FileUtils.sizeOf(file);
			result += 1024 * 4; // reserved 4 KB for profiles.xml data
			return result;
		} catch (Exception e) {
			throw new IOException("Could not determine game size " + e.toString());
		}
	}

	private static long sizeInBytes(ZipFile zf) throws IOException {
		long bytes = 0;
		for (Enumeration<? extends ZipEntry> entries = zf.entries(); entries.hasMoreElements();) {
			try {
				ZipEntry entry = entries.nextElement();
				bytes += entry.getSize();
			} catch (IllegalArgumentException e) {
				System.out.println("WARNING: Zip file [" + zf.getName() + "] contains an entry with problematic characters in its filename");
			}
		}
		return bytes;
	}

	private static String findMainFolder(File zipFile) throws IOException {
		ZipFile zfile = null;
		try {
			zfile = new ZipFile(zipFile, zipfilesUsingAltEncoding.contains(zipFile) ? Charset.forName("CP437"): StandardCharsets.UTF_8);
			for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
				try {
					ZipEntry entry = entries.nextElement();
					if (entry.isDirectory())
						return getZipEntryName(entry);
				} catch (IllegalArgumentException e) {
					System.out.println("WARNING: Zip file [" + zipFile + "] contains an entry with problematic characters in its filename");
				}
			}
		} finally {
			if (zfile != null)
				zfile.close();
		}
		return null;
	}

	private static void unzip(File zipFile, File dstDir) throws IOException {
		ProgressNotifyable prog = new ProgressNotifyable() {
			private long total_;
			private long progress_;
			private String lastInfo_;

			public void setTotal(long total) {
				total_ = total;
			}

			public void incrProgress(long progress) {
				progress_ += progress;
				String info = String.format("\rExtracting %s: %3.1f%%", zipFile, progress_ * 100.0 / total_);
				if (!info.equals(lastInfo_)) {
					System.out.print(info);
					lastInfo_ = info;
				}
			}

			public void setProgress(long progress) {
			}
		};

		ZipFile zfile = null;
		try {
			zfile = new ZipFile(zipFile);
			prog.setTotal(sizeInBytes(zfile));
			for (Enumeration<? extends ZipEntry> entries = zfile.entries(); entries.hasMoreElements();) {
				try {
					ZipEntry entry = entries.nextElement();
					File dstFile = new File(dstDir, getZipEntryName(entry));
					ZipUtils.extractEntry(zfile, entry, dstFile, prog);
				} catch (IllegalArgumentException e) {
					System.out.println("WARNING: Zip file [" + zipFile + "] contains an entry with problematic characters in its filename");
				}
			}
			System.out.printf("\rExtracting %s: Done  \n", zipFile);
		} finally {
			if (zfile != null)
				zfile.close();
		}
	}

	private static void copyZipData(File srcFile, File baseDirectory, ZipOutputStream zos) throws ZipException, IOException {
		ZipFile srcZipFile = null;
		try {
			srcZipFile = new ZipFile(srcFile, zipfilesUsingAltEncoding.contains(srcFile) ? Charset.forName("CP437"): StandardCharsets.UTF_8);
			long sizeInBytes = sizeInBytes(srcZipFile);
			for (Enumeration<? extends ZipEntry> entries = srcZipFile.entries(); entries.hasMoreElements();) {
				try {
					ZipEntry srcEntry = entries.nextElement();
					File dstFilename = new File(baseDirectory, getZipEntryName(srcEntry));
					ZipEntry dstEntry = new ZipEntry(FilesUtils.toArchivePath(dstFilename, srcEntry.isDirectory()));
					dstEntry.setComment(srcEntry.getComment());
					dstEntry.setTime(srcEntry.getTime());
					zos.putNextEntry(dstEntry);
					if (!srcEntry.isDirectory()) {
						IOUtils.copyLarge(srcZipFile.getInputStream(srcEntry), zos);
					}
					zos.closeEntry();

					long progress = srcEntry.getSize();
					if (((float)progress / (float)sizeInBytes) > 0.03)
						System.out.print('.');
				} catch (IllegalArgumentException e) {
					System.out.println("WARNING: Zip file [" + srcFile + "] contains an entry with problematic characters in its filename");
				}
			}
			System.out.println(". Done");
		} finally {
			if (srcZipFile != null)
				srcZipFile.close();
		}
	}

	private static File findSuitableExtension(File mainFile, List<File> list) {
		for (String extension: FilesUtils.EXECUTABLES) {
			File newMainFile = new File(FilenameUtils.removeExtension(mainFile.getPath()) + extension);
			if (list.contains(newMainFile))
				return newMainFile;
		}
		return null; // not found
	}

	public static String join(String[] array, String separator) {
		StringBuffer buf = new StringBuffer();
		boolean first = true;
		for (String s: array) {
			if (StringUtils.isNotEmpty(s)) {
				if (!first)
					buf.append(separator);
				buf.append(s);
				first = false;
			}
		}
		return buf.toString();
	}
}
