/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.dialog.wizard;

import java.io.File;
import java.io.IOException;
import org.apache.commons.lang3.StringUtils;
import org.dbgl.gui.GeneralPurposeDialogs;
import org.dbgl.gui.abstractdialog.JobWizardDialog;
import org.dbgl.gui.controls.BrowseButton;
import org.dbgl.gui.controls.BrowseButton.BrowseType;
import org.dbgl.gui.controls.BrowseButton.CanonicalType;
import org.dbgl.gui.thread.DFendImportThread;
import org.dbgl.gui.thread.DFendReloadedImportThread;
import org.dbgl.model.aggregate.DosboxVersion;
import org.dbgl.service.FileLocationService;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ExpandAdapter;
import org.eclipse.swt.events.ExpandEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ExpandBar;
import org.eclipse.swt.widgets.ExpandItem;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;


public class DFendImportDialog extends JobWizardDialog<Object> {

	private DosboxVersion defaultDbversion_;
	private ExpandItem orginalExpandItem_;
	private Text location_, dfrLocation_, dfrConfsLocation_;
	private Button cleanup_, dfrCleanup_;

	public DFendImportDialog(Shell parent, DosboxVersion dbversion) {
		super(parent, "dfendimportdialog");
		defaultDbversion_ = dbversion;
	}

	@Override
	protected String getDialogTitle() {
		return text_.get("dialog.dfendimport.title");
	}

	@Override
	protected void onShellCreated() {
		Group optionsGroup = createGroup(shell_, text_.get("dialog.dfendimport.options"), new FillLayout());

		ExpandBar bar = new ExpandBar(optionsGroup, SWT.V_SCROLL);

		Composite originalComposite = createComposite(bar, 3);
		location_ = createLabelAndText(originalComposite, text_.get("dialog.dfendimport.dfendpath"), FileLocationService.DFEND_STRING);
		new BrowseButton(originalComposite).connect(shell_, location_, null, BrowseType.FILE, CanonicalType.DFEND, false, null);
		cleanup_ = createLabelAndCheckButton(originalComposite, text_.get("dialog.dfendimport.cleanup"), text_.get("dialog.dfendimport.removesections"), true);
		createLabel(originalComposite);

		orginalExpandItem_ = createExpandItem(bar, text_.get("dialog.dfendimport.original.title"), false, originalComposite);

		Composite reloadedComposite = createComposite(bar, 3);
		dfrLocation_ = createLabelAndText(reloadedComposite, text_.get("dialog.dfendimport.reloaded.path"), FileLocationService.DFEND_RELOADED_PATH_STRING);
		new BrowseButton(reloadedComposite).connect(shell_, dfrLocation_, null, BrowseType.DIR, CanonicalType.NONE, false, null);
		dfrConfsLocation_ = createLabelAndText(reloadedComposite, text_.get("dialog.dfendimport.reloaded.exportedconfspath"));
		new BrowseButton(reloadedComposite).connect(shell_, dfrConfsLocation_, null, BrowseType.DIR, CanonicalType.NONE, false, null);
		dfrCleanup_ = createLabelAndCheckButton(reloadedComposite, text_.get("dialog.dfendimport.cleanup"), text_.get("dialog.dfendimport.reloaded.removesections"), false);
		createLabel(reloadedComposite);

		createExpandItem(bar, text_.get("dialog.dfendimport.reloaded.title"), true, reloadedComposite);

		bar.addExpandListener(new ExpandAdapter() {
			public void itemCollapsed(ExpandEvent e) {
				bar.getItem((((ExpandItem)e.item).getText().equals(text_.get("dialog.dfendimport.reloaded.title"))) ? 0: 1).setExpanded(true);
				display_.asyncExec(new Runnable() {
					public void run() {
						shell_.layout();
					}
				});
			}

			public void itemExpanded(ExpandEvent e) {
				bar.getItem((((ExpandItem)e.item).getText().equals(text_.get("dialog.dfendimport.reloaded.title"))) ? 0: 1).setExpanded(false);
				display_.asyncExec(new Runnable() {
					public void run() {
						shell_.layout();
					}
				});
			}
		});

		addStep(optionsGroup);

		addFinalStep(text_.get("dialog.dfendimport.progress"), text_.get("dialog.dfendimport.startimport"));
	}

	@Override
	protected boolean onNext(int step) {
		if (step == 0) {
			if (!isValid())
				return false;
		} else if (step == 1) {
			try {
				if (orginalExpandItem_.getExpanded())
					job_ = new DFendImportThread(log_, progressBar_, status_, new File(location_.getText()), cleanup_.getSelection(), defaultDbversion_);
				else
					// maybe also import data files in the future
					job_ = new DFendReloadedImportThread(log_, progressBar_, status_, new File(dfrLocation_.getText()), new File(dfrConfsLocation_.getText()), dfrCleanup_.getSelection(),
							defaultDbversion_);
			} catch (IOException e) {
				GeneralPurposeDialogs.warningMessage(shell_, e);
				return false;
			}
		} else if (step == 2) {
			if (job_.isEverythingOk())
				GeneralPurposeDialogs.infoMessage(shell_, text_.get("dialog.dfendimport.notice.importok"));
			else
				GeneralPurposeDialogs.warningMessage(shell_, text_.get("dialog.dfendimport.error.problem"));

			status_.setText(text_.get("dialog.dfendimport.reviewlog"));
			status_.pack();

			result_ = job_;
		}
		return true;
	}

	private boolean isValid() {
		GeneralPurposeDialogs.initErrorDialog();
		if (orginalExpandItem_.getExpanded()) {
			if (StringUtils.isBlank(location_.getText())) {
				GeneralPurposeDialogs.addError(text_.get("dialog.dfendimport.required.location"), location_);
			}
		} else {
			if (StringUtils.isBlank(dfrLocation_.getText())) {
				GeneralPurposeDialogs.addError(text_.get("dialog.dfendimport.reloaded.required.location"), dfrLocation_);
			}
			if (StringUtils.isBlank(dfrConfsLocation_.getText())) {
				GeneralPurposeDialogs.addError(text_.get("dialog.dfendimport.reloaded.required.confslocation"), dfrConfsLocation_);
			}
		}
		return !GeneralPurposeDialogs.displayErrorDialog(shell_);
	}
}